-- Structures.hs
----------
module Structures
    (   CityGraph(..),
        Route(..),
        Edge(..),
        Node(..),
        City,
        Distance,
        getCityGraphFromEdges,
        getRouteFromString,
        getEdgesFromLines
    ) where


-- import Data.String
import Data.Map(Map, fromListWith, empty)
import qualified Data.Set as Set
-- import Data.List
import Data.List.Split
import Data.Function (on)
import Control.DeepSeq

type City = Int
type Distance = Int

-- Edge consist of its index (Int), and a distance (Int)
data Edge = Edge Int Distance deriving (Eq, Show, Read)

-- Data structure representing graph of cities. It's an Map of Lists
-- Note, we expect small number of cities & edges, so O(logk) access should be negligible
-- Using Lists at the 2nd level for iterative traversal of neighbours.
data CityGraph = CityGraph (Map City [Edge]) deriving (Eq, Show, Read)

-- Given a list of edges (node, node, distance), return a CityGraph.
getCityGraphFromEdges :: [(City, City, Distance)] -> CityGraph
getCityGraphFromEdges [] = CityGraph empty
getCityGraphFromEdges edgeList = CityGraph edgeMap
    where edgeMap = fromListWith (++) (fmap (\(n1,n2,d) -> (n1, [(Edge n2 d)])) edgeList)

-- Route is just a list of Nodes (Indicies)
data Route = Route [City] deriving (Eq, Ord, Show, Read)

getRouteFromString :: String -> Route
getRouteFromString [] = Route []
getRouteFromString s = Route (fmap (\x -> read x :: Int) (splitOn " " s))

getEdgesFromLines :: [String] -> [(Int, Int, Int)]
getEdgesFromLines [] = []
getEdgesFromLines l = let splitList = fmap (\x -> splitOn "," x) l
                      in fmap (\x -> (read (x !! 0) :: Int, read (x !! 1) :: Int, read (x !! 2) :: Int)) splitList

-- States in A* search
data Node = Node {
    city :: City,
    path :: [City],
    gCost :: Distance,   -- Cost so far
    fCost :: Distance    -- Estimated total cost (gCost + heuristic)
} deriving (Show)

-- this part needed to be compatible with deepseq.
instance NFData Node where
    rnf (Node c p g f) = rnf c `seq` rnf p `seq` rnf g `seq` rnf f

-- Equality and ordering for priority queue
instance Eq Node where
    n1 == n2 = fCost n1 == fCost n2
instance Ord Node where
    compare = compare `on` fCost