module Serial
    ( tspSearch
    ) where

import Structures

import qualified Data.HashSet as HashSet
import qualified Data.Set as Set
import qualified Data.Map as Map
import qualified Data.PQueue.Min as PQ
import Data.Maybe (fromJust)
import AStarLib

-- This is the serial implementation of A*.
tspSearch :: CityGraph -> Route
tspSearch cityGraph = Route (reverse (path bestNode))
  where
    cities = Serial.getCities cityGraph
    startCity = head cities  -- arbitrary
    goalState = Set.fromList cities
    initialNode = Node startCity [startCity] 0 (heuristic startCity [startCity] goalState cityGraph)
    frontier = PQ.singleton initialNode
    visitedStates = HashSet.empty
    bestNode = astarSearch frontier visitedStates cityGraph goalState startCity


-- Get all unique cities in the graph, which is just the keys of the graph.
getCities :: CityGraph -> [City]
getCities (CityGraph graph) = Map.keys graph

astarSearch :: PQ.MinQueue Node -> HashSet.HashSet (City, Set.Set City) -> CityGraph -> Set.Set City -> City -> Node
astarSearch frontier visitedStates cityGraph goalState startCity
  | PQ.null frontier = error "No solution found"
  | isGoal currentNode goalState startCity = currentNode
  | (currentCity, currentVisited) `HashSet.member` visitedStates = astarSearch restQueue visitedStates cityGraph goalState startCity
  -- | (currentCity, currentVisited) `Set.member` visitedStates' = astarSearch restQueue visitedStates' cityGraph goalState startCity
  | otherwise = astarSearch newFrontier newVisitedStates cityGraph goalState startCity
  where
    (currentNode, restQueue) = fromJust (PQ.minView frontier)
    currentCity = city currentNode
    currentVisited = Set.fromList (path currentNode)
    newVisitedStates = HashSet.insert (currentCity, currentVisited) visitedStates
    successors = expandNode currentNode cityGraph goalState startCity
    newFrontier = foldr PQ.insert restQueue successors


