{-# LANGUAGE BangPatterns #-}
module Fractal (iterationToColor, pixelToCoord, mandelbrotIterations, juliaIterations) where

import Data.Word (Word8)

maxIter :: Int
maxIter = 1000

iterationToColor :: Int -> (Word8, Word8, Word8)
iterationToColor i
  | i == maxIter = (0, 0, 0)
  | otherwise    =
      let t = sqrt (fromIntegral i / fromIntegral maxIter)
          r = floor (9 * (1 - t) * t * t * t * 255) -- polynomials for colors found online
          g = floor (15 * (1 - t)*(1 - t)*t*t * 255)
          b = floor (8.5 * (1 - t)*(1 - t)*(1 - t)*t * 255)
      in (fromIntegral r, fromIntegral g, fromIntegral b)


pixelToCoord :: Int -> Int -> Int -> Int -> (Double, Double) -> (Double, Double)
pixelToCoord x y width height (minRe, maxIm) =
    let re = minRe + (fromIntegral x / fromIntegral width) * (5)
        im = maxIm - (fromIntegral y / fromIntegral height) * (4)
    in (re, im)


mandelbrotIterations :: Double -> Double -> Int
mandelbrotIterations cr ci = go 0 0 0
  where
    go !i !zr !zi
      | i == maxIter || zr*zr + zi*zi > 4.0 = i
      | otherwise =
          let zr' = zr*zr - zi*zi + cr
              zi' = 2*zr*zi + ci
          in go (i+1) zr' zi'


juliaIterations :: (Double, Double) -> Double -> Double -> Int
juliaIterations (cr, ci) zr zi = go 0 zr zi
  where
    go !i !zr !zi
      | i == maxIter || zr*zr + zi*zi > 4.0 = i
      | otherwise =
          let zr' = zr*zr - zi*zi + cr
              zi' = 2*zr*zi + ci
          in go (i+1) zr' zi'
