module top_module (
    // ======= Driver I/O =======
    input logic clk,  // 50 MHz
    input logic clk_24,  // 24 MHz
    input logic reset,
    input chipselect,

    output logic [63:0] readdata,  // {12'd0, column_num, row_num, endOfField, 15'd0, dout}
    input logic read,
    input logic [63:0] writedata,  // {32'd0, x, y}
    input logic write,


    // ======= VGA =======
    output logic [7:0] VGA_R,
    VGA_G,
    VGA_B,
    output logic VGA_CLK,
    VGA_HS,
    VGA_VS,
    VGA_BLANK_N,
    output logic VGA_SYNC_N,
    // ======= Debugging =======
    output [9:0] LED,
    // ======= Camera =======
    output wire SCLK,  // GPIO_0[0]
    inout logic SDA,  // GPIO_0[1]
    input logic VS,  // GPIO_0[2] 
    input logic HREF,  // GPIO_0[3]
    input logic PCLK,  // GPIO_0[4]
    output logic MCLK,  // GPIO_0[5]  		25MHz
    input logic [7:0] DATA,  // GPIO_O[13:6]
    output logic CAM_RESET,  // GPIO_0[14] 
    output wire PWDN

);

    //////////////////////////////////////////////////////////////////
    // Clock divider    
    logic clk_25;

    clk_div clk_div_0 (
        .clk_ref(clk),
        .clk_out(clk_25)
    );


    //////////////////////////////////////////////////////////////////
    // Camera and Asyn FIFO
    wire [ 9:0] data_count_r;
    wire [15:0] dout;  // RGB565 format
    wire [ 9:0] cam_debug;

    // Camera module (camera_interface.sv), 24MHz
    camera_interface m0 (
        .clk_50(clk_24),
        .clk_24(clk_24),
        .rst_n(~reset),  // rst_n is active-low, reset is active-high
        //asyn_fifo IO
        .rd_en(read_en),
        .data_count_r(data_count_r),
        .dout(dout),
        //camera pinouts
        .cmos_pclk(PCLK),
        .cmos_href(HREF),
        .cmos_vsync(VS),
        .cmos_db(DATA),
        .cmos_sda(SDA),
        .cmos_scl(SCLK),
        .cmos_rst_n(CAM_RESET),  // CAM_RESET is active-low
        .cmos_pwdn(PWDN),
        .cmos_xclk(MCLK),
        //Debugging
        .led(cam_debug)
    );

    //////////////////////////////////////////////////////////////////
    // FIFO
    wire read_en;

    always_ff @(posedge clk) begin
        if (data_count_r >= 128) begin // start reading when there are 128 pixels in the FIFO
            read_en <= 1;
        end else begin
            read_en <= 0;
        end
    end


    //////////////////////////////////////////////////////////////////
    // VGA module
    logic [10:0]  hcount;
    logic [9:0]  vcount;
    logic    endOfField;


    vga_counter counter_0 (
        .clk50(clk),
        .reset(reset),
        .enable(1),
        .hcount(hcount),
        .vcount(vcount),
        .VGA_CLK(VGA_CLK),  // 25 MHz
        .VGA_HS(VGA_HS),
        .VGA_VS(VGA_VS),
        .VGA_BLANK_N(VGA_BLANK_N),
        .VGA_SYNC_N(VGA_SYNC_N),
        .endOfField(endOfField)
    );

    //////////////////////////////////////////////////////////////////////
    // RGB565 to RGB888
    logic [7:0] dout_r_8, dout_g_8, dout_b_8;

    RGB565_to_RGB888 rgb565_to_rgb888 (
        .r_5(dout[15:11]),
        .g_6(dout[10:5]),
        .b_5(dout[4:0]),
        .r_8(dout_r_8),
        .g_8(dout_g_8),
        .b_8(dout_b_8)
    );

    //////////////////////////////////////////////////////////////////
    // VGA display
    logic [15:0] ball_x = 320;
    logic [15:0] ball_y = 240;
    assign inside_ball = (ball_x - hcount[10:1]) * (ball_x - hcount[10:1]) + (ball_y - vcount[9:0]) * (ball_y - vcount[9:0]) < 1024;

    always_ff @(posedge clk) begin
        if (reset) begin
            VGA_R  <= 8'd0;
            VGA_G  <= 8'd0;
            VGA_B  <= 8'd0;
            ball_x <= 320;
            ball_y <= 240;
        end else begin
            if (VGA_BLANK_N) begin
                if (inside_ball) begin
                    // ball set to grey 
                    VGA_R <= 8'h88;
                    VGA_G <= 8'h88;
                    VGA_B <= 8'h88;
                end else begin
                    // set to camera data
                    VGA_R <= dout_r_8;
                    VGA_G <= dout_g_8;
                    VGA_B <= dout_b_8;
                end
            end
            // HPS write to FPGA
            if (chipselect && write) begin
                ball_x <= writedata[15:0];
                ball_y <= writedata[31:16];
            end
            // HPS read from FPGA
            if (chipselect && read) begin
                readdata <= {
                    12'd0, hcount[10:1], vcount[9:0], endOfField, 15'd0, dout
                };
            end else begin
                readdata <= 64'd0;
            end
        end
    end

    //////////////////////////////////////////////////////////////////
    // debug

    // assign LED[0] = read_en;
    // assign LED[1] = 1'b0;
    // assign LED[2] = 1'b1;
    assign LED[7:0] = cam_debug[7:0];


endmodule



module clk_div (
    input  logic clk_ref,
    output logic clk_out
);

    reg [27:0] counter = 28'd0;
    parameter DIV = 28'd2;

    always @(posedge clk_ref) begin
        counter <= counter + 28'd1;
        if (counter >= (DIV - 1)) counter <= 28'd0;
        clk_out <= (counter < DIV / 2) ? 1'b1 : 1'b0;
    end
endmodule


module RGB565_to_RGB888 (
    input  logic [4:0] r_5,
    b_5,
    input  logic [5:0] g_6,
    output logic [7:0] r_8,
    g_8,
    b_8
);

    always_comb begin
        r_8 = {r_5, r_5[2:0]};
        g_8 = {g_6, g_6[1:0]};
        b_8 = {b_5, b_5[2:0]};
    end

endmodule
