/* * Device driver for the VGA video generator
 *
 * A Platform device implemented using the misc subsystem
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod vga_ball.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree vga_ball.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include <linux/unistd.h>
#include "user_interface.h"

#define DRIVER_NAME "user_interface"

/* Device registers */
//#define BG_RED(x) (x)
//#define BG_GREEN(x) ((x)+1)
//#define BG_BLUE(x) ((x)+2)

#define UI_BPM_PLAYBACK(x) ((x)+20)
#define UI_STEP_TRACK(x) ((x)+22)


/*
 * Information about our device
 */
struct user_interface_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
	user_interface_props_t props;
} dev;

/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
*/ 

/*
static void write_background(vga_ball_color_t *background)
{
	iowrite8(background->red, BG_RED(dev.virtbase) );
	iowrite8(background->green, BG_GREEN(dev.virtbase) );
	iowrite8(background->blue, BG_BLUE(dev.virtbase) );
	dev.background = *background;
}
*/

// Writes x and y coordinates 
static void read_props(user_interface_props_t *props)
{ 
	unsigned int bpm_playback = ioread16(UI_BPM_PLAYBACK(dev.virtbase));
	unsigned int step_track = ioread16(UI_STEP_TRACK(dev.virtbase));
	props->step = (unsigned char)step_track;
        props->track = (unsigned char) (step_track >> 8);
        props->bpm =  (unsigned short)(bpm_playback & 0x00007FFF);
	printk(KERN_INFO "Here: %hu", props->bpm);
	props->playback = (unsigned char) ((bpm_playback & 0x00008000) >> 15);
	dev.props = *props;
} 


/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long user_interface_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	//vga_ball_arg_t vla;
	user_interface_props_t props;

	switch (cmd) {
	/*case VGA_BALL_WRITE_BACKGROUND:
		if (copy_from_user(&vla, (vga_ball_arg_t *) arg,
				   sizeof(vga_ball_arg_t)))
			return -EACCES;
		write_background(&vla.background);
		break;

	case VGA_BALL_READ_BACKGROUND:
	  	vla.background = dev.background;
		if (copy_to_user((vga_ball_arg_t *) arg, &vla,
				 sizeof(vga_ball_arg_t)))
			return -EACCES;
		break;
	// NEW CASES HERE: */
	case USER_INTERFACE_WRITE_PROPS:
		if (copy_from_user(&props,(user_interface_props_t * ) arg, sizeof(user_interface_props_t )))
		       return -EACCES;
		break;	

	case USER_INTERFACE_READ_PROPS:
		read_props(&props);
		if (copy_to_user((user_interface_props_t *) arg, &props, sizeof(user_interface_props_t)))
			return -EACCES;
		break;


	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations user_interface_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = user_interface_ioctl,



};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice user_interface_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &user_interface_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init user_interface_probe(struct platform_device *pdev)
{
	int ret;

	/* Register ourselves as a misc device: creates /dev/vga_ball */
	ret = misc_register(&user_interface_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
        
	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&user_interface_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int user_interface_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&user_interface_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id user_interface_of_match[] = {
	{ .compatible = "csee4840,user_interface-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, user_interface_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver user_interface_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(user_interface_of_match),
	},
	.remove	= __exit_p(user_interface_remove),
};

/* Called when the module is loaded: set things up */
static int __init user_interface_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&user_interface_driver, user_interface_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit user_interface_exit(void)
{
	platform_driver_unregister(&user_interface_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(user_interface_init);
module_exit(user_interface_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Brandon Cruz, Columbia University");
MODULE_DESCRIPTION("Sequencer UI driver");
