/*
 * Avalon memory-mapped peripheral that acts as RAM for an audio sample
 * Team EmbeddedSequencer - Columbia University
 *
 ================ TODO ================
 - should be able to return to the store state so that a new sample can be loaded
 - Add some signals for debuggin

 */

module driver_interface #(parameter MEM_DEPTH = 48000)
(
    // system interface
    input logic clk,
	  input logic reset,
    
    // avalon interface
		input logic [31:0]  writedata,
		input logic write,
		input chipselect,
		input logic address,

    // codec interface
    input logic advance,
    output logic [23:0] leftSample,
    output logic [23:0] rightSample
);

// internal signals
logic [31:0] control;               // register that contains the control registers
logic [15:0] mem_out;        
logic [15:0] w_r_address;           // Need 16 bits to access every memory address
logic mem_we;
logic [15:-16] fract_index;
logic [15:-16] fract_index_sum;
logic [15:-16] pitch_shift;
logic [3:0] semitone;

enum {RST, WAIT_STORE, STORE, WAIT_READ, READ, DONE_READ} state;

// Internal Modules
ram #( .MEM_DEPTH(MEM_DEPTH) ) audio_ram
(
    .clk(clk),
    .we(mem_we),
    .addr(w_r_address),
    .d_in(writedata[15:0]),
    .d_out(mem_out)
);

pitch_shifter p1(.semitone(semitone), .pitch_shift(pitch_shift));



// state logic
always_ff @(posedge clk)
  if (reset) state <= RST;
  else case(state)
    RST: begin // reset internal signals
      w_r_address <= 16'b0;
      fract_index <= 32'b0;
      control <= 32'b0;
      mem_we <= 1'b0;
      state <= WAIT_STORE;
      end
    WAIT_STORE: if (write && chipselect) begin
        mem_we <= 1'b1;
        state <= STORE;
      end else begin
        mem_we <= 1'b0;
        state <= WAIT_STORE;
      end
    STORE: begin
        if (w_r_address == MEM_DEPTH-1) begin
            w_r_address <= 16'b0;   // reset for read
            fract_index <= 32'b0;
            mem_we <= 1'b0;
            state <= WAIT_READ;     // sample stored, continue to read mode
        end else if(!write) begin         // wait for falling edge of write
            w_r_address <= w_r_address + 16'b1;
            state <= WAIT_STORE;
            mem_we <= 1'b0; 
        end else begin
            state <= STORE;
            mem_we <= 1'b1;
        end
      end
    WAIT_READ: 
        if (advance) state <= READ;
        else if (write && chipselect) begin // arrival of a new wav file
            w_r_address <= 16'b0;
            fract_index <= 32'b0;
            mem_we <= 1'b1;
            state <= STORE;
        end else begin
            state <= WAIT_READ;
        end
    READ: 
        if (!advance) begin      // wait for falling edge           
            //if (w_r_address == MEM_DEPTH-1) w_r_address <= 16'b0; // wraparound
            //else w_r_address <= w_r_address + 16'b1; 
            if (w_r_address == MEM_DEPTH - 1) begin fract_index <= 32'b0; w_r_address <= 16'b0; state <= DONE_READ; end
            else if (fract_index_sum[15:0] > MEM_DEPTH - 1) begin fract_index <= 32'b0; w_r_address <= 16'b0; state <= DONE_READ; end 
            else begin fract_index <= fract_index_sum; w_r_address <= fract_index_sum[15:0]; state <= WAIT_READ; end 
        end else if (write && chipselect) begin // arrival of a new wav file
            w_r_address <= 16'b0;
            fract_index <= 32'b0;
            mem_we <= 1'b1;
            state <= STORE;
        end else begin
            state <= READ;
        end

    DONE_READ:
        if (write && chipselect) begin // arrival of a new wav file
            w_r_address <= 16'b0;
            fract_index <= 32'b0;
            mem_we <= 1'b1;
            state <= STORE;
        end else begin
            state <= DONE_READ;
        end
  endcase

// output logic
always_comb begin
  case(state)
    READ: begin
        if (!advance) begin  
            fract_index_sum = fract_index + pitch_shift;
            leftSample = {mem_out, 8'b0};
            rightSample = {mem_out, 8'b0};
        end else begin
            leftSample = {mem_out, 8'b0};
            rightSample = {mem_out, 8'b0};
            fract_index_sum = 32'b0;
        end
        end
    WAIT_READ:begin
        leftSample = {mem_out, 8'b0};
        rightSample = {mem_out, 8'b0};
        fract_index_sum = 32'b0;   
    end
    default: begin // make sure data is ready before advance signal arrives
      leftSample = 24'b0;
      rightSample = 24'b0;
      fract_index_sum = 32'b0;
      end
  endcase
end
endmodule

module pitch_shifter(input logic [3:0] semitone,
                   output logic [15:-16] pitch_shift);
always_comb begin 
    case(semitone)
        4'd0:       pitch_shift = 32'h0001_0000;
        4'd1:       pitch_shift = 32'h0001_0f3b;
        4'd2:       pitch_shift = 32'h0001_1f5c; 
        4'd3:       pitch_shift = 32'h0001_306f;
        4'd4:       pitch_shift = 32'h0001_4289; 
        4'd5:       pitch_shift = 32'h0001_55b5; 
        4'd6:       pitch_shift = 32'h0001_6a09; 
        4'd7:       pitch_shift = 32'h0001_7f91; 
        4'd8:       pitch_shift = 32'h0001_9660; 
        4'd9:       pitch_shift = 32'h0001_ae8a; 
        4'd10:      pitch_shift = 32'h0001_c824; 
        4'd11:      pitch_shift = 32'h0001_e3c3;
        default:    pitch_shift = 32'h0001_0000;
    endcase
end
endmodule

