/*
 * Avalon memory-mapped peripheral that generates VGA
 *
 * Stephen A. Edwards
 * Columbia University
 */

module vga_ball(input logic        clk,
	        input logic 	   reset,
		input logic [31:0]  writedata,
		input logic 	   write,
		input 		   chipselect,
		input logic [2:0]  address,

		input left_chan_ready,
		input right_chan_ready,

		output logic [15:0] sample_data_l,
		output logic sample_valid_l,
		output logic [15:0] sample_data_r,
		output logic sample_valid_r,

		output logic [7:0] VGA_R, VGA_G, VGA_B,
		output logic 	   VGA_CLK, VGA_HS, VGA_VS,
		                   VGA_BLANK_n,
		output logic 	   VGA_SYNC_n);
   parameter slot0_x = 120;
   parameter slot1_x = 270;
   parameter slot2_x = 420;
   parameter slot_y  = 160;
   parameter front   = 64;
   parameter back    = 64;
   parameter up      = 4;
   parameter ELF_X      = 64;
   parameter ELF_Y      = 288;
   parameter ENEMY_X      = 448;
   parameter ENEMY_Y      = 288;
   parameter PROTECT_R      = 96;
   parameter PROTECT_BIAS   = 64;
   parameter BLOOD_ELF_X  = 32;
   parameter BLOOD_ELF_Y  = 180;
   parameter BLOOD_ENEMY_X  = 416;
   parameter BLOOD_ENEMY_Y  = 180;
   parameter BLOOD_AZHDAHA_X  = 64;
   parameter BLOOD_AZHDAHA_Y  = 64;
   parameter AZHDAHA_X  = 64;
   parameter AZHDAHA_Y  = 180;
   logic [10:0]	   hcount;
   logic [9:0]     vcount;
   logic [31:0] ppu_info,sound;
   logic [31:0] wall, target, azhdaha, slot, slot0, slot1, slot2, char, elf, weapon;	
   vga_counters counters(.clk50(clk), .*);

   always_ff @(posedge clk)
     if (reset) begin
	wall <= 32'd_0;
	target <= 32'd_0;
	azhdaha <= 32'd_0;
	slot <= 32'd_0;
	slot0 <= 32'd_0;
	slot1 <= 32'd_0;
	slot2 <= 32'd_0;
	char <= 32'd_0;
	elf <= 32'd_0;
	weapon <= 32'd_0;
	sound <= 32'd_0;
     end else if (chipselect && write)
       case (address)
	 3'h0 : azhdaha<= writedata;
	 3'h1 : sound <= writedata;
	 3'h2 : wall <= writedata;
	 3'h3 : slot <= writedata;
	 3'h4 : weapon <= writedata;
	 3'h5 : elf <= writedata;
	 3'h6 : char <= writedata;
	 3'h7 : target <= writedata;// sound
       endcase

/*
       logic [9:0]  slot0_x;
       logic [9:0]  slot0_y;
       logic [5:0]  slot0_p;
       logic [4:0]  slot0_c;
       logic slot0_v;
       logic [15:0] slot0_address;
       logic [7:0]  slot0_output;

       logic [9:0]  slot1_x;
       logic [9:0]  slot1_y;
       logic [5:0]  slot1_p;
       logic [4:0]  slot1_c;
       logic slot1_v;
       logic [15:0] slot1_address;
       logic [7:0]  slot1_output;
*/       

       logic ball_enable_0;
       logic ball_enable_1;
       logic ball_enable_enemy_0;
       logic ball_enable_enemy_1;
       logic protect_enable_enemy;
       logic protect_enable_elf;
       logic slot_v;
       logic [2:0]  slot0_p;
       logic [15:0] slot0_address;
       logic [7:0]  slot0_output;

       logic [2:0]  slot1_p;
       logic [15:0] slot1_address;
       logic [7:0]  slot1_output;

       logic [2:0]  slot2_p;
       logic [15:0] slot2_address;
       logic [7:0]  slot2_output;

       soc_system_slot_pic slot_pic0(.address(slot0_address), .clk(clk), .clken(1), .reset_req(0), .readdata(slot0_output));
       soc_system_slot_pic slot_pic1(.address(slot1_address), .clk(clk), .clken(1), .reset_req(0), .readdata(slot1_output));
       soc_system_slot_pic slot_pic2(.address(slot2_address), .clk(clk), .clken(1), .reset_req(0), .readdata(slot2_output));
       logic [1:0] slot0_en; 
       logic [1:0] slot1_en;   
       logic [1:0] slot2_en;  

       
       logic fire_f;
       logic [9:0]  wall_x;
       logic [9:0]  wall_y;
       logic [5:0]  wall_p;
       logic [4:0]  wall_c;
       logic [12:0] wall_address;
       logic [12:0] wall_address_1;
       logic [12:0] fire_address;
       logic [12:0] fire_0_address;
       logic [7:0]  wall_output; 
       logic [7:0]  wall_output_1; 
       logic [7:0]  fire_output; 
       logic [7:0]  fire_0_output; 
       soc_system_wall_pic wall_pic(.address(wall_address), .clk(clk), .clken(1), .reset_req(0), .readdata(wall_output));  
       soc_system_wall_pic wall_pic_1(.address(wall_address_1), .clk(clk), .clken(1), .reset_req(0), .readdata(wall_output_1)); 
       soc_system_wall_pic fire_pic(.address(fire_address), .clk(clk), .clken(1), .reset_req(0), .readdata(fire_output)); 
       soc_system_wall_pic fire_0_pic(.address(fire_0_address), .clk(clk), .clken(1), .reset_req(0), .readdata(fire_0_output)); 
       logic [1:0] wall_en;    
       logic [1:0] wall_1_en;    
       logic [1:0] fire_en;     
       logic [17:0] fire_rand_en;   
       logic [3:0] fire_count; 
       logic [1:0] blood_elf_en; 
       logic [1:0] blood_elf_shadow_en; 
       logic [1:0] blood_enemy_en; 
       logic [1:0] blood_enemy_shadow_en; 
       logic [1:0] blood_azhdaha_en;
       logic [1:0] blood_azhdaha_shadow_en; 
       logic [5:0] blood_percent_azhdaha;
       logic [4:0] blood_percent_elf;
       logic [4:0] blood_percent_enemy;
       logic blood_azhdaha_v;   
       logic blood_elf_v;   
       logic blood_enemy_v;  
       logic wall_v;     

       logic [9:0]  weapon_enemy_x;
       logic [9:0]  weapon_enemy_y;
       logic weapon_enemy_p;
       logic weapon_attack_enemy;
       logic [9:0]  weapon_x;
       logic [9:0]  weapon_y;
       logic [5:0]  weapon_p;
       logic [2:0]  weapon_t;
       logic weapon_protect_elf;  
       logic weapon_protect_enemy;    
       logic weapon_attack;     
       logic [9:0] weapon_address;
       logic [7:0]  weapon_output; 
       soc_system_weapon_pic weapon_pic(.address(weapon_address), .clk(clk), .clken(1), .reset_req(0), .readdata(weapon_output));  
       logic [1:0] weapon_en;

       logic [9:0]  elf_x;
       logic [9:0]  elf_y;
       logic [1:0]  elf_p;
       logic [1:0]  elf_c;
       logic elf_f;
       logic elf_v;      
       logic [12:0] elf_address;
       logic [7:0]  elf_output; 
       soc_system_elf_pic elf_pic(.address(elf_address), .clk(clk), .clken(1), .reset_req(0), .readdata(elf_output));  
       logic [1:0] elf_en;

       logic [9:0]  enemy_x;
       logic [9:0]  enemy_y;
       logic [1:0]  enemy_p;
       logic [1:0]  enemy_c;
       logic enemy_f;
       logic enemy_v;     
       logic [12:0] enemy_address;
       logic [7:0]  enemy_output; 
       soc_system_elf_pic enemy_pic(.address(enemy_address), .clk(clk), .clken(1), .reset_req(0), .readdata(enemy_output));  
       logic [1:0] enemy_en;

       logic [17:0] atk_azhdaha;
       logic [9:0]  azhdaha_x;
       logic [9:0]  azhdaha_y;
       logic [5:0]  azhdaha_p;
       logic [4:0]  azhdaha_c;
       logic azhdaha_v;      
       logic [15:0] azhdaha_address;
       logic [7:0]  azhdaha_output; 
       soc_system_azhdaha_pic azhdaha_pic(.address(azhdaha_address), .clk(clk), .clken(1), .reset_req(0), .readdata(azhdaha_output));  
       logic [1:0] azhdaha_en; 

	
       logic [11:0] char_address;
       logic [7:0]  char_output; 
       soc_system_char_pic char_pic(.address(char_address), .clk(clk), .clken(1), .reset_req(0), .readdata(char_output));  
       logic [1:0] char_en;

       logic [7:0]  font_address;
       logic [7:0]  font_output; 
       soc_system_words_pic font_pic(.address(font_address), .clk(clk), .clken(1), .reset_req(0), .readdata(font_output));  
       logic [1:0] font_en;

       logic [9:0]  target_x;
       logic [9:0]  target_y;
       logic [5:0]  target_p;
       logic [4:0]  target_c;
       logic target_v;
       soc_system_char_pic target_pic(.address(target_address), .clk(clk), .clken(1), .reset_req(0), .readdata(target_output));  
       logic [11:0] target_address;
       logic [7:0]  target_output;
       logic [1:0]  target_en;
 
   always_comb begin
	
	tune <= sound[5:0];
       
       weapon_enemy_x <= slot[31:22];
       weapon_enemy_y <= slot[21:12];
       weapon_enemy_p <= slot[11];
       weapon_attack_enemy <= slot[10];
       slot0_p <= slot[3:1];
       slot1_p <= slot[6:4];
       slot2_p <= slot[9:7];
       slot_v <= slot[0]; 

       msg_visible <= wall[31:18];
       fire_count <= wall[17:14];
       blood_percent_elf <= wall[13:9];
       blood_percent_enemy <= wall[8:4];
       fire_f <= wall[3];
       blood_elf_v <= wall[2];
       blood_enemy_v <= wall[1];
       wall_v <= wall[0];


       
       atk_azhdaha <= azhdaha[28:11];
       blood_percent_azhdaha <= azhdaha[10:5];
       azhdaha_p <= azhdaha[4:2];
       blood_azhdaha_v <= azhdaha[1];
       azhdaha_v <= azhdaha[0];

       target_x <= target[31:22];
       target_y <= target[21:12];
       target_p <= target[11:6];
       target_c <= target[5:1];
       target_v <= target[0];

       elf_x <= elf[31:22];
       elf_y <= elf[21:12];
       enemy_p <= elf[11:10];
       elf_p <= elf[9:8];
       enemy_c <= elf[7:6];
       elf_c <= elf[5:4];
       enemy_f <= elf[3];
       elf_f <= elf[2];
       enemy_v <= elf[1];
       elf_v <= elf[0];

       weapon_x <= weapon[31:22];
       weapon_y <= weapon[21:12];
       weapon_p <= weapon[11:6];
       weapon_t <= weapon[5:3];
       weapon_protect_elf <= weapon[2];
       weapon_protect_enemy <= weapon[1];
       weapon_attack  <= weapon[0];

   end
//**********************************************************************************************************************************************//
//
//
// Slot0 position state
//
//**********************************************************************************************************************************************//
	/*******************************************************/
	/**** Message display block, generated by sv_gen.py ****/
	/*******************************************************/

	// Is message visible
	logic LV_en;
	logic ATK_en;
	logic DEF_en;
	logic COIN_en;
	logic SLOT_en;
	logic CATCH_en;
	logic BATTLE_en;
	logic BOSS_en;
	logic ONLINE_en;

	logic [13:0] msg_visible;
	// Whether show message
	logic [13:0] msg_display;
	// Selected message index
	logic [4:0] msg_selected;
	// Whether the current message is selected
	logic cur_msg_selected;
	// Font pixel index of a line, [0,8)
	logic [2:0] font_pix_idx;

	always_ff @(posedge clk) begin

		if((hcount[10:1] >= 10'd512) && (hcount[10:1] < 10'd544) && (vcount >= 10'd36) && (vcount < 10'd46) && 1) begin
			COIN_en <= 1;
			cur_msg_selected <= (msg_selected==1);
			case((hcount[10:1]-512)>>3)
				2'd0: font_address <= 8'd10+((vcount[9:0]-36)>>1);  //c
				2'd1: font_address <= 8'd70+((vcount[9:0]-36)>>1);  //o
				2'd2: font_address <= 8'd40+((vcount[9:0]-36)>>1);  //i
				2'd3: font_address <= 8'd65+((vcount[9:0]-36)>>1);  //n
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-512)>>0;
		end
		else
			COIN_en <= 0;

		if((hcount[10:1] >= 10'd512) && (hcount[10:1] < 10'd536) && (vcount >= 10'd114) && (vcount < 10'd124) && 1) begin
			ATK_en <= 1;
			cur_msg_selected <= (msg_selected==1);
			case((hcount[10:1]-512)>>3)
				2'd0: font_address <= 8'd0+((vcount[9:0]-114)>>1);  //a
				2'd1: font_address <= 8'd95+((vcount[9:0]-114)>>1); //t
				2'd2: font_address <= 8'd50+((vcount[9:0]-114)>>1); //k
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-512)>>0;
		end
		else
			ATK_en <= 0;

		if((hcount[10:1] >= 10'd512) && (hcount[10:1] < 10'd528) && (vcount >= 10'd146) && (vcount < 10'd156) && 1) begin
			DEF_en <= 1;
			cur_msg_selected <= (msg_selected==1);
			case((hcount[10:1]-512)>>3)
				2'd0: font_address <= 8'd35+((vcount[9:0]-146)>>1);  //h
				2'd1: font_address <= 8'd75+((vcount[9:0]-146)>>1);  //p
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-512)>>0;
		end
		else
			DEF_en <= 0;

		if((hcount[10:1] >= 10'd512) && (hcount[10:1] < 10'd528) && (vcount >= 10'd82) && (vcount < 10'd92) && 1) begin
			LV_en <= 1;
			cur_msg_selected <= (msg_selected==1);
			case((hcount[10:1]-512)>>3)
				2'd0: font_address <= 8'd55+((vcount[9:0]-82)>>1);  //l
				2'd1: font_address <= 8'd105+((vcount[9:0]-82)>>1);  //v
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-512)>>0;
		end
		else
			LV_en <= 0;

		/**
		* message: SLOT
		* h_start: 32
		* v_start: 32
		* font_width: 8
		* font_height: 10
		*/
		if((hcount[10:1] >= 10'd32) && (hcount[10:1] < 10'd64) && (vcount >= 10'd32) && (vcount < 10'd42) && 1) begin
			SLOT_en <= 1;
			cur_msg_selected <= (msg_selected==1);
			case((hcount[10:1]-32)>>3)
				2'd0: font_address <= 8'd90+((vcount[9:0]-32)>>1);
				2'd1: font_address <= 8'd55+((vcount[9:0]-32)>>1);
				2'd2: font_address <= 8'd70+((vcount[9:0]-32)>>1);
				2'd3: font_address <= 8'd95+((vcount[9:0]-32)>>1);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-32)>>0;
		end
		else
			SLOT_en <= 0;

		/**
		* message: CATCH
		* h_start: 72
		* v_start: 32
		* font_width: 8
		* font_height: 10
		*/
		if((hcount[10:1] >= 10'd72) && (hcount[10:1] < 10'd112) && (vcount >= 10'd32) && (vcount < 10'd42) && 1) begin
			CATCH_en <= 1;
			cur_msg_selected <= (msg_selected==2);
			case((hcount[10:1]-72)>>3)
				3'd0: font_address <= 8'd10+((vcount[9:0]-32)>>1);
				3'd1: font_address <= 8'd0+((vcount[9:0]-32)>>1);
				3'd2: font_address <= 8'd95+((vcount[9:0]-32)>>1);
				3'd3: font_address <= 8'd10+((vcount[9:0]-32)>>1);
				3'd4: font_address <= 8'd35+((vcount[9:0]-32)>>1);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-72)>>0;
		end
		else
			CATCH_en <= 0;

		/**
		* message: BATTLE
		* h_start: 120
		* v_start: 32
		* font_width: 8
		* font_height: 10
		*/
		if((hcount[10:1] >= 10'd120) && (hcount[10:1] < 10'd168) && (vcount >= 10'd32) && (vcount < 10'd42) && 1) begin
			BATTLE_en <= 1;
			cur_msg_selected <= (msg_selected==3);
			case((hcount[10:1]-120)>>3)
				3'd0: font_address <= 8'd5+((vcount[9:0]-32)>>1);
				3'd1: font_address <= 8'd0+((vcount[9:0]-32)>>1);
				3'd2: font_address <= 8'd95+((vcount[9:0]-32)>>1);
				3'd3: font_address <= 8'd95+((vcount[9:0]-32)>>1);
				3'd4: font_address <= 8'd55+((vcount[9:0]-32)>>1);
				3'd5: font_address <= 8'd20+((vcount[9:0]-32)>>1);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-120)>>0;
		end
		else
			BATTLE_en <= 0;

		/**
		* message: BOSS
		* h_start: 176
		* v_start: 32
		* font_width: 8
		* font_height: 10
		*/
		if((hcount[10:1] >= 10'd176) && (hcount[10:1] < 10'd208) && (vcount >= 10'd32) && (vcount < 10'd42) && 1) begin
			BOSS_en <= 1;
			cur_msg_selected <= (msg_selected==4);
			case((hcount[10:1]-176)>>3)
				2'd0: font_address <= 8'd5+((vcount[9:0]-32)>>1);
				2'd1: font_address <= 8'd70+((vcount[9:0]-32)>>1);
				2'd2: font_address <= 8'd90+((vcount[9:0]-32)>>1);
				2'd3: font_address <= 8'd90+((vcount[9:0]-32)>>1);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-176)>>0;
		end
		else
			BOSS_en <= 0;

		/**
		* message: ONLINE
		* h_start: 216
		* v_start: 32
		* font_width: 8
		* font_height: 10
		*/
		if((hcount[10:1] >= 10'd216) && (hcount[10:1] < 10'd264) && (vcount >= 10'd32) && (vcount < 10'd42) && 1) begin
			ONLINE_en <= 1;
			cur_msg_selected <= (msg_selected==5);
			case((hcount[10:1]-216)>>3)
				3'd0: font_address <= 8'd70+((vcount[9:0]-32)>>1);
				3'd1: font_address <= 8'd65+((vcount[9:0]-32)>>1);
				3'd2: font_address <= 8'd55+((vcount[9:0]-32)>>1);
				3'd3: font_address <= 8'd40+((vcount[9:0]-32)>>1);
				3'd4: font_address <= 8'd65+((vcount[9:0]-32)>>1);
				3'd5: font_address <= 8'd20+((vcount[9:0]-32)>>1);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-216)>>0;
		end
		else
			ONLINE_en <= 0;
              
		/**
		* message: PRESS A TO CATCH IT!
		* h_start: 64
		* v_start: 220
		* font_width: 32
		* font_height: 40
		*/
                if((hcount[10:1] >= 10'd48) && (hcount[10:1] < 10'd592) && (vcount >= 10'd220) && (vcount < 10'd260) && msg_visible[1]) begin
			msg_display[1] <= 1;
			cur_msg_selected <= (msg_selected==1);
			case((hcount[10:1]-48)>>5)
				5'd0: font_address <= 8'd75+((vcount[9:0]-220)>>3);
				5'd1: font_address <= 8'd85+((vcount[9:0]-220)>>3);
				5'd2: font_address <= 8'd20+((vcount[9:0]-220)>>3);
				5'd3: font_address <= 8'd90+((vcount[9:0]-220)>>3);
				5'd4: font_address <= 8'd90+((vcount[9:0]-220)>>3);
				5'd5: font_address <= 8'd180+((vcount[9:0]-220)>>3);
				5'd6: font_address <= 8'd0+((vcount[9:0]-220)>>3);
				5'd7: font_address <= 8'd180+((vcount[9:0]-220)>>3);
				5'd8: font_address <= 8'd10+((vcount[9:0]-220)>>3);
				5'd9: font_address <= 8'd0+((vcount[9:0]-220)>>3);
				5'd10: font_address <= 8'd95+((vcount[9:0]-220)>>3);
				5'd11: font_address <= 8'd10+((vcount[9:0]-220)>>3);
				5'd12: font_address <= 8'd35+((vcount[9:0]-220)>>3);
				5'd13: font_address <= 8'd180+((vcount[9:0]-220)>>3);
				5'd14: font_address <= 8'd40+((vcount[9:0]-220)>>3);
				5'd15: font_address <= 8'd95+((vcount[9:0]-220)>>3);
				5'd16: font_address <= 8'd185+((vcount[9:0]-220)>>3);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-48)>>2;
		end
		else
			msg_display[1] <= 0;

		/**
		* message: CATCH ELF FIRST!
		* h_start: 64
		* v_start: 220
		* font_width: 32
		* font_height: 40
		*/
		if((hcount[10:1] >= 10'd64) && (hcount[10:1] < 10'd576) && (vcount >= 10'd220) && (vcount < 10'd260) && msg_visible[2]) begin
			msg_display[2] <= 1;
			cur_msg_selected <= (msg_selected==2);
			case((hcount[10:1]-64)>>5)
				4'd0: font_address <= 8'd10+((vcount[9:0]-220)>>3);
				4'd1: font_address <= 8'd0+((vcount[9:0]-220)>>3);
				4'd2: font_address <= 8'd95+((vcount[9:0]-220)>>3);
				4'd3: font_address <= 8'd10+((vcount[9:0]-220)>>3);
				4'd4: font_address <= 8'd35+((vcount[9:0]-220)>>3);
				4'd5: font_address <= 8'd180+((vcount[9:0]-220)>>3);
				4'd6: font_address <= 8'd20+((vcount[9:0]-220)>>3);
				4'd7: font_address <= 8'd55+((vcount[9:0]-220)>>3);
				4'd8: font_address <= 8'd25+((vcount[9:0]-220)>>3);
				4'd9: font_address <= 8'd180+((vcount[9:0]-220)>>3);
				4'd10: font_address <= 8'd25+((vcount[9:0]-220)>>3);
				4'd11: font_address <= 8'd40+((vcount[9:0]-220)>>3);
				4'd12: font_address <= 8'd85+((vcount[9:0]-220)>>3);
				4'd13: font_address <= 8'd90+((vcount[9:0]-220)>>3);
				4'd14: font_address <= 8'd95+((vcount[9:0]-220)>>3);
				4'd15: font_address <= 8'd185+((vcount[9:0]-220)>>3);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-64)>>2;
		end
		else
			msg_display[2] <= 0;

		/**
		* message: ATK PRESS A
		* h_start: 32
		* v_start: 128
		* font_width: 16
		* font_height: 20
		*/
		if((hcount[10:1] >= 10'd32) && (hcount[10:1] < 10'd208) && (vcount >= 10'd128) && (vcount < 10'd148) && msg_visible[3]) begin
			msg_display[3] <= 1;
			cur_msg_selected <= (msg_selected==3);
			case((hcount[10:1]-32)>>4)
				4'd0: font_address <= 8'd0+((vcount[9:0]-128)>>2);
				4'd1: font_address <= 8'd95+((vcount[9:0]-128)>>2);
				4'd2: font_address <= 8'd50+((vcount[9:0]-128)>>2);
				4'd3: font_address <= 8'd180+((vcount[9:0]-128)>>2);
				4'd4: font_address <= 8'd75+((vcount[9:0]-128)>>2);
				4'd5: font_address <= 8'd85+((vcount[9:0]-128)>>2);
				4'd6: font_address <= 8'd20+((vcount[9:0]-128)>>2);
				4'd7: font_address <= 8'd90+((vcount[9:0]-128)>>2);
				4'd8: font_address <= 8'd90+((vcount[9:0]-128)>>2);
				4'd9: font_address <= 8'd180+((vcount[9:0]-128)>>2);
				4'd10: font_address <= 8'd0+((vcount[9:0]-128)>>2);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-32)>>1;
		end
		else
			msg_display[3] <= 0;

		/**
		* message: DEF PRESS B
		* h_start: 32
		* v_start: 160
		* font_width: 16
		* font_height: 20
		*/
		if((hcount[10:1] >= 10'd32) && (hcount[10:1] < 10'd208) && (vcount >= 10'd160) && (vcount < 10'd180) && msg_visible[4]) begin
			msg_display[4] <= 1;
			cur_msg_selected <= (msg_selected==4);
			case((hcount[10:1]-32)>>4)
				4'd0: font_address <= 8'd15+((vcount[9:0]-160)>>2);
				4'd1: font_address <= 8'd20+((vcount[9:0]-160)>>2);
				4'd2: font_address <= 8'd25+((vcount[9:0]-160)>>2);
				4'd3: font_address <= 8'd180+((vcount[9:0]-160)>>2);
				4'd4: font_address <= 8'd75+((vcount[9:0]-160)>>2);
				4'd5: font_address <= 8'd85+((vcount[9:0]-160)>>2);
				4'd6: font_address <= 8'd20+((vcount[9:0]-160)>>2);
				4'd7: font_address <= 8'd90+((vcount[9:0]-160)>>2);
				4'd8: font_address <= 8'd90+((vcount[9:0]-160)>>2);
				4'd9: font_address <= 8'd180+((vcount[9:0]-160)>>2);
				4'd10: font_address <= 8'd5+((vcount[9:0]-160)>>2);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-32)>>1;
		end
		else
			msg_display[4] <= 0;

		/**
		* message: STOP PRESS B
		* h_start: 32
		* v_start: 64
		* font_width: 16
		* font_height: 20
		*/
		if((hcount[10:1] >= 10'd32) && (hcount[10:1] < 10'd224) && (vcount >= 10'd64) && (vcount < 10'd84) && msg_visible[5]) begin
			msg_display[5] <= 1;
			cur_msg_selected <= (msg_selected==5);
			case((hcount[10:1]-32)>>4)
				4'd0: font_address <= 8'd90+((vcount[9:0]-64)>>2);
				4'd1: font_address <= 8'd95+((vcount[9:0]-64)>>2);
				4'd2: font_address <= 8'd70+((vcount[9:0]-64)>>2);
				4'd3: font_address <= 8'd75+((vcount[9:0]-64)>>2);
				4'd4: font_address <= 8'd180+((vcount[9:0]-64)>>2);
				4'd5: font_address <= 8'd75+((vcount[9:0]-64)>>2);
				4'd6: font_address <= 8'd85+((vcount[9:0]-64)>>2);
				4'd7: font_address <= 8'd20+((vcount[9:0]-64)>>2);
				4'd8: font_address <= 8'd90+((vcount[9:0]-64)>>2);
				4'd9: font_address <= 8'd90+((vcount[9:0]-64)>>2);
				4'd10: font_address <= 8'd180+((vcount[9:0]-64)>>2);
				4'd11: font_address <= 8'd5+((vcount[9:0]-64)>>2);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-32)>>1;
		end
		else
			msg_display[5] <= 0;

		/**
		* message: RESTART PRESS Y
		* h_start: 32
		* v_start: 96
		* font_width: 16
		* font_height: 20
		*/
		if((hcount[10:1] >= 10'd32) && (hcount[10:1] < 10'd272) && (vcount >= 10'd96) && (vcount < 10'd116) && msg_visible[6]) begin
			msg_display[6] <= 1;
			cur_msg_selected <= (msg_selected==6);
			case((hcount[10:1]-32)>>4)
				4'd0: font_address <= 8'd85+((vcount[9:0]-96)>>2);
				4'd1: font_address <= 8'd20+((vcount[9:0]-96)>>2);
				4'd2: font_address <= 8'd90+((vcount[9:0]-96)>>2);
				4'd3: font_address <= 8'd95+((vcount[9:0]-96)>>2);
				4'd4: font_address <= 8'd0+((vcount[9:0]-96)>>2);
				4'd5: font_address <= 8'd85+((vcount[9:0]-96)>>2);
				4'd6: font_address <= 8'd95+((vcount[9:0]-96)>>2);
				4'd7: font_address <= 8'd180+((vcount[9:0]-96)>>2);
				4'd8: font_address <= 8'd75+((vcount[9:0]-96)>>2);
				4'd9: font_address <= 8'd85+((vcount[9:0]-96)>>2);
				4'd10: font_address <= 8'd20+((vcount[9:0]-96)>>2);
				4'd11: font_address <= 8'd90+((vcount[9:0]-96)>>2);
				4'd12: font_address <= 8'd90+((vcount[9:0]-96)>>2);
				4'd13: font_address <= 8'd180+((vcount[9:0]-96)>>2);
				4'd14: font_address <= 8'd120+((vcount[9:0]-96)>>2);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-32)>>1;
		end
		else
			msg_display[6] <= 0;

		/**
		* message: congratulations!
		* h_start: 64
		* v_start: 220
		* font_width: 32
		* font_height: 40
		*/
		if((hcount[10:1] >= 10'd64) && (hcount[10:1] < 10'd576) && (vcount >= 10'd220) && (vcount < 10'd260) && msg_visible[7]) begin
			msg_display[7] <= 1;
			cur_msg_selected <= (msg_selected==7);
			case((hcount[10:1]-64)>>5)
				4'd0: font_address <= 8'd10+((vcount[9:0]-220)>>3);
				4'd1: font_address <= 8'd70+((vcount[9:0]-220)>>3);
				4'd2: font_address <= 8'd65+((vcount[9:0]-220)>>3);
				4'd3: font_address <= 8'd30+((vcount[9:0]-220)>>3);
				4'd4: font_address <= 8'd85+((vcount[9:0]-220)>>3);
				4'd5: font_address <= 8'd0+((vcount[9:0]-220)>>3);
				4'd6: font_address <= 8'd95+((vcount[9:0]-220)>>3);
				4'd7: font_address <= 8'd100+((vcount[9:0]-220)>>3);
				4'd8: font_address <= 8'd55+((vcount[9:0]-220)>>3);
				4'd9: font_address <= 8'd0+((vcount[9:0]-220)>>3);
				4'd10: font_address <= 8'd95+((vcount[9:0]-220)>>3);
				4'd11: font_address <= 8'd40+((vcount[9:0]-220)>>3);
				4'd12: font_address <= 8'd70+((vcount[9:0]-220)>>3);
				4'd13: font_address <= 8'd65+((vcount[9:0]-220)>>3);
				4'd14: font_address <= 8'd90+((vcount[9:0]-220)>>3);
				4'd15: font_address <= 8'd185+((vcount[9:0]-220)>>3);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-64)>>2;
		end
		else
			msg_display[7] <= 0;

		/**
		* message: +20
		* h_start: 308
		* v_start: 128
		* font_width: 8
		* font_height: 10
		*//*
		if((hcount[10:1] >= 10'd560) && (hcount[10:1] < 10'd584) && (vcount >= 10'd54) && (vcount < 10'd64) && msg_visible[6]) begin
			msg_display[6] <= 1;
			cur_msg_selected <= (msg_selected==6);
			case((hcount[10:1]-560)>>3)
				2'd0: font_address <= 8'd205+((vcount[9:0]-54)>>1);
				2'd1: font_address <= 8'd140+((vcount[9:0]-54)>>1);
				2'd2: font_address <= 8'd130+((vcount[9:0]-54)>>1);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-560)>>0;
		end
		else
			msg_display[6] <= 0;*/

		/**
		* message: -20
		* h_start: 308
		* v_start: 128
		* font_width: 8
		* font_height: 10
		*/
		if((hcount[10:1] >= 10'd176) && (hcount[10:1] < 10'd464) && (vcount >= 10'd220) && (vcount < 10'd260) && msg_visible[8]) begin
			msg_display[8] <= 1;
			cur_msg_selected <= (msg_selected==1);
			case((hcount[10:1]-176)>>5)
				4'd0: font_address <= 8'd30+((vcount[9:0]-220)>>3);
				4'd1: font_address <= 8'd0+((vcount[9:0]-220)>>3);
				4'd2: font_address <= 8'd60+((vcount[9:0]-220)>>3);
				4'd3: font_address <= 8'd20+((vcount[9:0]-220)>>3);
				4'd4: font_address <= 8'd180+((vcount[9:0]-220)>>3);
				4'd5: font_address <= 8'd70+((vcount[9:0]-220)>>3);
				4'd6: font_address <= 8'd105+((vcount[9:0]-220)>>3);
				4'd7: font_address <= 8'd20+((vcount[9:0]-220)>>3);
				4'd8: font_address <= 8'd85+((vcount[9:0]-220)>>3);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-176)>>2;
		end
		else
			msg_display[8] <= 0;

		/**
		* message: +200
		* h_start: 304
		* v_start: 128
		* font_width: 8
		* font_height: 10
		*/
		if((hcount[10:1] >= 10'd552) && (hcount[10:1] < 10'd584) && (vcount >= 10'd54) && (vcount < 10'd64) && msg_visible[9]) begin
			msg_display[9] <= 1;
			cur_msg_selected <= (msg_selected==8);
			case((hcount[10:1]-552)>>3)
				2'd0: font_address <= 8'd205+((vcount[9:0]-54)>>1);
				2'd1: font_address <= 8'd140+((vcount[9:0]-54)>>1);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-552)>>0;
		end
		else
			msg_display[9] <= 0;

		/**
		* message: *2
		* h_start: 312
		* v_start: 128
		* font_width: 8
		* font_height: 10
		*/
		if((hcount[10:1] >= 10'd568) && (hcount[10:1] < 10'd584) && (vcount >= 10'd54) && (vcount < 10'd64) && msg_visible[10]) begin
			msg_display[10] <= 1;
			cur_msg_selected <= (msg_selected==9);
			case((hcount[10:1]-568)>>3)
				1'd0: font_address <= 8'd215+((vcount[9:0]-54)>>1);
				1'd1: font_address <= 8'd140+((vcount[9:0]-54)>>1);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-568)>>0;
		end
		else
			msg_display[10] <= 0;

		/**
		* message: HERE IS AZHDAHA!
		* h_start: 64
		* v_start: 220
		* font_width: 32
		* font_height: 40
		*/
		if((hcount[10:1] >= 10'd64) && (hcount[10:1] < 10'd576) && (vcount >= 10'd220) && (vcount < 10'd260) && msg_visible[11]) begin
			msg_display[11] <= 1;
			cur_msg_selected <= (msg_selected==1);
			case((hcount[10:1]-64)>>5)
				4'd0: font_address <= 8'd35+((vcount[9:0]-220)>>3);
				4'd1: font_address <= 8'd20+((vcount[9:0]-220)>>3);
				4'd2: font_address <= 8'd85+((vcount[9:0]-220)>>3);
				4'd3: font_address <= 8'd20+((vcount[9:0]-220)>>3);
				4'd4: font_address <= 8'd180+((vcount[9:0]-220)>>3);
				4'd5: font_address <= 8'd40+((vcount[9:0]-220)>>3);
				4'd6: font_address <= 8'd90+((vcount[9:0]-220)>>3);
				4'd7: font_address <= 8'd180+((vcount[9:0]-220)>>3);
				4'd8: font_address <= 8'd0+((vcount[9:0]-220)>>3);
				4'd9: font_address <= 8'd125+((vcount[9:0]-220)>>3);
				4'd10: font_address <= 8'd35+((vcount[9:0]-220)>>3);
				4'd11: font_address <= 8'd15+((vcount[9:0]-220)>>3);
				4'd12: font_address <= 8'd0+((vcount[9:0]-220)>>3);
				4'd13: font_address <= 8'd35+((vcount[9:0]-220)>>3);
				4'd14: font_address <= 8'd0+((vcount[9:0]-220)>>3);
				4'd15: font_address <= 8'd185+((vcount[9:0]-220)>>3);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-64)>>2;
		end
		else
			msg_display[11] <= 0;

		/**
		* message: -40
		* h_start: 308
		* v_start: 128
		* font_width: 8
		* font_height: 10
		*/
/*		if((hcount[10:1] >= 10'd560) && (hcount[10:1] < 10'd584) && (vcount >= 10'd54) && (vcount < 10'd64) && msg_visible[11]) begin
			msg_display[11] <= 1;
			cur_msg_selected <= (msg_selected==11);
			case((hcount[10:1]-560)>>3)
				2'd0: font_address <= 8'd205+((vcount[9:0]-54)>>1);
				2'd1: font_address <= 8'd145+((vcount[9:0]-54)>>1);
				2'd2: font_address <= 8'd125+((vcount[9:0]-54)>>1);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-560)>>0;
		end
		else
			msg_display[11] <= 0;*/

		/**
		* message: super effective!
		* h_start: 256
		* v_start: 128
		* font_width: 8
		* font_height: 10
		*/
		if((hcount[10:1] >= 10'd256) && (hcount[10:1] < 10'd384) && (vcount >= 10'd360) && (vcount < 10'd370) && msg_visible[12]) begin
			msg_display[12] <= 1;
			cur_msg_selected <= (msg_selected==12);
			case((hcount[10:1]-256)>>3)
				4'd0: font_address <= 8'd90+((vcount[9:0]-360)>>1);
				4'd1: font_address <= 8'd100+((vcount[9:0]-360)>>1);
				4'd2: font_address <= 8'd75+((vcount[9:0]-360)>>1);
				4'd3: font_address <= 8'd20+((vcount[9:0]-360)>>1);
				4'd4: font_address <= 8'd85+((vcount[9:0]-360)>>1);
				4'd5: font_address <= 8'd180+((vcount[9:0]-360)>>1);
				4'd6: font_address <= 8'd20+((vcount[9:0]-360)>>1);
				4'd7: font_address <= 8'd25+((vcount[9:0]-360)>>1);
				4'd8: font_address <= 8'd25+((vcount[9:0]-360)>>1);
				4'd9: font_address <= 8'd20+((vcount[9:0]-360)>>1);
				4'd10: font_address <= 8'd10+((vcount[9:0]-360)>>1);
				4'd11: font_address <= 8'd95+((vcount[9:0]-360)>>1);
				4'd12: font_address <= 8'd40+((vcount[9:0]-360)>>1);
				4'd13: font_address <= 8'd105+((vcount[9:0]-360)>>1);
				4'd14: font_address <= 8'd20+((vcount[9:0]-360)>>1);
				4'd15: font_address <= 8'd185+((vcount[9:0]-360)>>1);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-256)>>0;
		end
		else
			msg_display[12] <= 0;

		/**
		* message: no hurt
		* h_start: 292
		* v_start: 128
		* font_width: 8
		* font_height: 10
		*/
		if((hcount[10:1] >= 10'd292) && (hcount[10:1] < 10'd348) && (vcount >= 10'd370) && (vcount < 10'd380) && msg_visible[13]) begin
			msg_display[13] <= 1;
			cur_msg_selected <= (msg_selected==13);
			case((hcount[10:1]-292)>>3)
				3'd0: font_address <= 8'd65+((vcount[9:0]-370)>>1);
				3'd1: font_address <= 8'd70+((vcount[9:0]-370)>>1);
				3'd2: font_address <= 8'd180+((vcount[9:0]-370)>>1);
				3'd3: font_address <= 8'd35+((vcount[9:0]-370)>>1);
				3'd4: font_address <= 8'd100+((vcount[9:0]-370)>>1);
				3'd5: font_address <= 8'd85+((vcount[9:0]-370)>>1);
				3'd6: font_address <= 8'd95+((vcount[9:0]-370)>>1);
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-292)>>0;
		end
		else
			msg_display[13] <= 0;
	end

	/*******************************************************/
	/************** Message display block end **************/
	/*******************************************************/


  always_ff @(posedge clk) begin

      if (hcount[10:1] >= slot0_x && hcount[10:1] <= (slot0_x + 10'd100) && vcount[9:0] >= slot_y && vcount[9:0] <= (slot_y + 10'd128) && slot_v) begin
        
        slot0_en <= 2'b1;
	case(slot0_p)
	    6'b000000 : slot0_address <= hcount[10:1] - slot0_x + (vcount[9:0] - slot_y) * 100;
	    6'b000001 : slot0_address <= hcount[10:1] - slot0_x + (vcount[9:0] - slot_y) * 100 + 12800;
	    6'b000010 : slot0_address <= hcount[10:1] - slot0_x + (vcount[9:0] - slot_y) * 100 + 25600;
	    6'b000011 : slot0_address <= hcount[10:1] - slot0_x + (vcount[9:0] - slot_y) * 100 + 38400;
	    6'b000100 : slot0_address <= hcount[10:1] - slot0_x + (vcount[9:0] - slot_y) * 100 + 51200;
	endcase
      
      end

      else begin

        slot0_en <= 2'b0;

      end

    end
//**********************************************************************************************************************************************//
//
//
// Slot1 position state
//
//**********************************************************************************************************************************************//


  always_ff @(posedge clk) begin

      if (hcount[10:1] >= slot1_x && hcount[10:1] <= (slot1_x + 10'd100) && vcount[9:0] >= slot_y && vcount[9:0] <= (slot_y + 10'd128) && slot_v) begin
        
        slot1_en <= 2'b1;
	case(slot1_p)
	    6'b000000 : slot1_address <= hcount[10:1] - slot1_x + (vcount[9:0] - slot_y) * 100;
	    6'b000001 : slot1_address <= hcount[10:1] - slot1_x + (vcount[9:0] - slot_y) * 100 + 12800;
	    6'b000010 : slot1_address <= hcount[10:1] - slot1_x + (vcount[9:0] - slot_y) * 100 + 25600;
	    6'b000011 : slot1_address <= hcount[10:1] - slot1_x + (vcount[9:0] - slot_y) * 100 + 38400;
	    6'b000100 : slot1_address <= hcount[10:1] - slot1_x + (vcount[9:0] - slot_y) * 100 + 51200;
	endcase
      
      end

      else begin

        slot1_en <= 2'b0;

      end

    end

//**********************************************************************************************************************************************//
//
//
// slot2 position state
//
//**********************************************************************************************************************************************//

  always_ff @(posedge clk) begin

      if (hcount[10:1] >= slot2_x && hcount[10:1] <= (slot2_x + 10'd100) && vcount[9:0] >= slot_y && vcount[9:0] <= (slot_y + 10'd128) && slot_v) begin
        
        slot2_en <= 2'b1;
	case(slot2_p)
	    6'b000000 : slot2_address <= hcount[10:1] - slot2_x + (vcount[9:0] - slot_y) * 100;
	    6'b000001 : slot2_address <= hcount[10:1] - slot2_x + (vcount[9:0] - slot_y) * 100 + 12800;
	    6'b000010 : slot2_address <= hcount[10:1] - slot2_x + (vcount[9:0] - slot_y) * 100 + 25600;
	    6'b000011 : slot2_address <= hcount[10:1] - slot2_x + (vcount[9:0] - slot_y) * 100 + 38400;
	    6'b000100 : slot2_address <= hcount[10:1] - slot2_x + (vcount[9:0] - slot_y) * 100 + 51200;
	endcase
      
      end

      else begin

        slot2_en <= 2'b0;

      end

    end
//**********************************************************************************************************************************************//
//
//
// Azhdaha position state
//
//**********************************************************************************************************************************************//

  always_ff @(posedge clk) begin

      if (hcount[10:1] >= AZHDAHA_X && hcount[10:1] <= (AZHDAHA_X + 10'd512) && vcount[9:0] >= AZHDAHA_Y && vcount[9:0] <= (AZHDAHA_Y + 10'd256) && azhdaha_v) begin
        
        azhdaha_en <= 2'b1;
	case(azhdaha_p)
	    6'b000000 : azhdaha_address <= hcount[10:3] - AZHDAHA_X/4 + (vcount[9:2] - AZHDAHA_Y/4) * 128;
	    6'b000001 : azhdaha_address <= hcount[10:3] - AZHDAHA_X/4 + (vcount[9:2] - AZHDAHA_Y/4) * 128 + 8192;
	    6'b000010 : azhdaha_address <= hcount[10:3] - AZHDAHA_X/4 + (vcount[9:2] - AZHDAHA_Y/4) * 128 + 16384;
	    6'b000011 : azhdaha_address <= hcount[10:3] - AZHDAHA_X/4 + (vcount[9:2] - AZHDAHA_Y/4) * 128 + 24576;
	endcase
      
      end

      else begin

        azhdaha_en <= 2'b0;

      end

    end
//**********************************************************************************************************************************************//
//
//
// Target position state
//
//**********************************************************************************************************************************************//


  always_ff @(posedge clk) begin

      if ((hcount[10:1] - target_x) > 0 && (hcount[10:1] - target_x) < 10'd16 && (vcount[9:0] - target_y) > 0 && (vcount[9:0] - target_y) <= 10'd16  && target_v ==1 ) begin
        
        target_en <= 2'b1;
	target_address <= hcount[10:1] - target_x + (vcount[9:0] - target_y) * 16;
      end

      else begin

        target_en <= 2'b0;

      end

    end

//**********************************************************************************************************************************************//
//
//
// Blood_elf position state
//
//**********************************************************************************************************************************************//

  always_ff @(posedge clk) begin

      if ((hcount[10:1] - BLOOD_ELF_X) > 0 && (hcount[10:1] - BLOOD_ELF_X) < 10'd4*blood_percent_elf && (vcount[9:0] - BLOOD_ELF_Y) > 0 && (vcount[9:0] - BLOOD_ELF_Y) <= 10'd4 && blood_elf_v) begin
        
        blood_elf_en <= 2'b1;
      end

      else begin

        blood_elf_en <= 2'b0;

      end

    end

  always_ff @(posedge clk) begin

      if ((hcount[10:1] - BLOOD_ELF_X) > 0 && (hcount[10:1] - BLOOD_ELF_X) < 10'd4*blood_percent_elf && (vcount[9:0] - BLOOD_ELF_Y) > 4 && (vcount[9:0] - BLOOD_ELF_Y) <= 10'd6 && blood_elf_v) begin
        
        blood_elf_shadow_en <= 2'b1;
      end

      else begin

        blood_elf_shadow_en <= 2'b0;

      end

    end

//**********************************************************************************************************************************************//
//
//
// Blood_enemy position state
//
//**********************************************************************************************************************************************//

  always_ff @(posedge clk) begin

      if ((hcount[10:1] - BLOOD_ENEMY_X) > 0 && (hcount[10:1] - BLOOD_ENEMY_X) < 10'd4*blood_percent_enemy && (vcount[9:0] - BLOOD_ENEMY_Y) > 0 && (vcount[9:0] - BLOOD_ENEMY_Y) <= 10'd4 && blood_enemy_v) begin
        
        blood_enemy_en <= 2'b1;
      end

      else begin

        blood_enemy_en <= 2'b0;

      end

    end

  always_ff @(posedge clk) begin

      if ((hcount[10:1] - BLOOD_ENEMY_X) > 0 && (hcount[10:1] - BLOOD_ENEMY_X) < 10'd4*blood_percent_enemy && (vcount[9:0] - BLOOD_ENEMY_Y) > 4 && (vcount[9:0] - BLOOD_ENEMY_Y) <= 10'd6 && blood_enemy_v) begin
        
        blood_enemy_shadow_en <= 2'b1;
      end

      else begin

        blood_enemy_shadow_en <= 2'b0;

      end

    end

//**********************************************************************************************************************************************//
//
//
// Blood_azhdaha position state
//
//**********************************************************************************************************************************************//

  always_ff @(posedge clk) begin

      if ((hcount[10:1] - BLOOD_AZHDAHA_X) > 0 && (hcount[10:1] - BLOOD_AZHDAHA_X) < 10'd8*blood_percent_azhdaha && (vcount[9:0] - BLOOD_AZHDAHA_Y) > 0 && (vcount[9:0] - BLOOD_AZHDAHA_Y) <= 10'd4 && blood_azhdaha_v) begin
        
        blood_azhdaha_en <= 2'b1;
      end

      else begin

        blood_azhdaha_en <= 2'b0;

      end

    end

  always_ff @(posedge clk) begin

      if ((hcount[10:1] - BLOOD_AZHDAHA_X) > 0 && (hcount[10:1] - BLOOD_AZHDAHA_X) < 10'd8*blood_percent_azhdaha && (vcount[9:0] - BLOOD_AZHDAHA_Y) > 4 && (vcount[9:0] - BLOOD_AZHDAHA_Y) <= 10'd6 && blood_azhdaha_v) begin
        
        blood_azhdaha_shadow_en <= 2'b1;
      end

      else begin

        blood_azhdaha_shadow_en <= 2'b0;

      end

    end
//**********************************************************************************************************************************************//
//
//
// ELF position state
//
//**********************************************************************************************************************************************//

  always_ff @(posedge clk) begin
      if(elf_c == 2'b00)begin
        if (hcount[10:1] >= elf_x && hcount[10:1] <= (elf_x + 10'd128) && vcount[9:0] >= elf_y && vcount[9:0] <= (elf_y + 10'd128) && elf_v) begin
        
          elf_en <= 2'b1;
          if(elf_f)begin
	    case(elf_p)
	         2'b00 : elf_address <= hcount[10:2] - elf_x/2 + (vcount[9:1] - elf_y/2) * 64;
	         2'b01 : elf_address <= hcount[10:2] - elf_x/2 + (vcount[9:1] - elf_y/2) * 64 + 4096;
	    endcase
	  end
          else begin
	    case(elf_p)
	         2'b00 : elf_address <= (64 - (hcount[10:2] - elf_x/2)) + (vcount[9:1] - elf_y/2) * 64;
	         2'b01 : elf_address <= (64 - (hcount[10:2] - elf_x/2)) + (vcount[9:1] - elf_y/2) * 64 + 4096;
	    endcase
          end
        end 
        else begin
          elf_en <= 2'b0;
        end
      end
      else if(elf_c == 2'b01)begin
        if (hcount[10:1] >= elf_x && hcount[10:1] <= (elf_x + 10'd128) && vcount[9:0] >= (elf_y + up) && vcount[9:0] <= ((elf_y + up) + 10'd128) && elf_v) begin
        
          elf_en <= 2'b1;
          if(elf_f)begin
	    case(elf_p)
	         2'b00 : elf_address <= hcount[10:2] - elf_x/2 + (vcount[9:1] - (elf_y + up)/2) * 64;
	         2'b01 : elf_address <= hcount[10:2] - elf_x/2 + (vcount[9:1] - (elf_y + up)/2) * 64 + 4096;
	    endcase
	  end
          else begin
	    case(elf_p)
	         2'b00 : elf_address <= (64 - (hcount[10:2] - elf_x/2)) + (vcount[9:1] - (elf_y + up)/2) * 64;
	         2'b01 : elf_address <= (64 - (hcount[10:2] - elf_x/2)) + (vcount[9:1] - (elf_y + up)/2) * 64 + 4096;
	    endcase
          end
        end  
        else begin
          elf_en <= 2'b0;
        end
      end
      else if(elf_c == 2'b10)begin
        if (hcount[10:1] >= (elf_x + front) && hcount[10:1] <= ((elf_x + front) + 10'd128) && vcount[9:0] >= elf_y && vcount[9:0] <= (elf_y + 10'd128) && elf_v) begin
        
          elf_en <= 2'b1;
          if(elf_f)begin
	    case(elf_p)
	         2'b00 : elf_address <= hcount[10:2] - (elf_x + front)/2 + (vcount[9:1] - elf_y/2) * 64;
	         2'b01 : elf_address <= hcount[10:2] - (elf_x + front)/2 + (vcount[9:1] - elf_y/2) * 64 + 4096;
	    endcase
	  end
          else begin
	    case(elf_p)
	         2'b00 : elf_address <= (64 - (hcount[10:2] - (elf_x + front)/2)) + (vcount[9:1] - elf_y/2) * 64;
	         2'b01 : elf_address <= (64 - (hcount[10:2] - (elf_x + front)/2)) + (vcount[9:1] - elf_y/2) * 64 + 4096;
	    endcase
          end
        end  
        else begin
          elf_en <= 2'b0;
        end
      end
      else begin
        if (hcount[10:1] >= (elf_x - back) && hcount[10:1] <= ((elf_x - back) + 10'd128) && vcount[9:0] >= elf_y && vcount[9:0] <= (elf_y + 10'd128) && elf_v) begin
        
          elf_en <= 2'b1;
          if(elf_f)begin
	    case(elf_p)
	         2'b00 : elf_address <= hcount[10:2] - (elf_x - back)/2 + (vcount[9:1] - elf_y/2) * 64;
	         2'b01 : elf_address <= hcount[10:2] - (elf_x - back)/2 + (vcount[9:1] - elf_y/2) * 64 + 4096;
	    endcase
	  end
          else begin
	    case(elf_p)
	         2'b00 : elf_address <= (64 - (hcount[10:2] - (elf_x - back)/2)) + (vcount[9:1] - elf_y/2) * 64;
	         2'b01 : elf_address <= (64 - (hcount[10:2] - (elf_x - back)/2)) + (vcount[9:1] - elf_y/2) * 64 + 4096;
	    endcase
          end
        end  
        else begin
          elf_en <= 2'b0;
        end
      end
    end
//**********************************************************************************************************************************************//
//
//
// Enemy position state
//
//**********************************************************************************************************************************************//

  always_ff @(posedge clk) begin

      if(enemy_c == 2'b00) begin
        if (hcount[10:1] >= 480 && hcount[10:1] <= (480 + 10'd128) && vcount[9:0] >= 280 && vcount[9:0] <= (280 + 10'd128) && enemy_v) begin
        
        enemy_en <= 2'b1;
	  case(enemy_p)
	     2'b00 : enemy_address <= hcount[10:2] - 480/2 + (vcount[9:1] - 280/2) * 64;
	     2'b01 : enemy_address <= hcount[10:2] - 480/2 + (vcount[9:1] - 280/2) * 64 + 4096;
	  endcase
        end
        else begin
          enemy_en <= 2'b0;
        end
      end

      else if(enemy_c == 2'b01) begin
        if (hcount[10:1] >= 480 && hcount[10:1] <= (480 + 10'd128) && vcount[9:0] >= (280 + up) && vcount[9:0] <= (280 + up + 10'd128) && enemy_v) begin
        
        enemy_en <= 2'b1;
	  case(enemy_p)
	     2'b00 : enemy_address <= hcount[10:2] - 480/2 + (vcount[9:1] - (280 + up)/2) * 64;
	     2'b01 : enemy_address <= hcount[10:2] - 480/2 + (vcount[9:1] - (280 + up)/2) * 64 + 4096;
	  endcase
        end
        else begin
          enemy_en <= 2'b0;
        end
      end

      else if(enemy_c == 2'b10) begin
        if (hcount[10:1] >= (480 - front) && hcount[10:1] <= ((480 - front) + 10'd128) && vcount[9:0] >= 280 && vcount[9:0] <= (280 + 10'd128) && enemy_v) begin
        
        enemy_en <= 2'b1;
	  case(enemy_p)
	     2'b00 : enemy_address <= hcount[10:2] - (480 - front)/2 + (vcount[9:1] - 280/2) * 64;
	     2'b01 : enemy_address <= hcount[10:2] - (480 - front)/2 + (vcount[9:1] - 280/2) * 64 + 4096;
	  endcase
        end
        else begin
          enemy_en <= 2'b0;
        end
      end

      else begin
        if (hcount[10:1] >= (480 + back) && hcount[10:1] <= ((480 + back) + 10'd128) && vcount[9:0] >= 280 && vcount[9:0] <= (280 + 10'd128) && enemy_v) begin
        
        enemy_en <= 2'b1;
	  case(enemy_p)
	     2'b00 : enemy_address <= hcount[10:2] - (480 + back)/2 + (vcount[9:1] - 280/2) * 64;
	     2'b01 : enemy_address <= hcount[10:2] - (480 + back)/2 + (vcount[9:1] - 280/2) * 64 + 4096;
	  endcase
        end
        else begin
          enemy_en <= 2'b0;
        end
      end
    end
//**********************************************************************************************************************************************//
//
//
// Weapon ball 
//
//**********************************************************************************************************************************************//

  always_ff @(posedge clk) begin

      if (((hcount[10:1] - weapon_x)*(hcount[10:1] - weapon_x) + (vcount[9:0] - weapon_y) * (vcount[9:0] - weapon_y)) <= 100 && weapon_attack) begin
	ball_enable_0 <= 1;
      end   
      else begin
	   ball_enable_0 <= 0;
      end     
      if (((hcount[10:1] - weapon_x)*(hcount[10:1] - weapon_x) + (vcount[9:0] - weapon_y) * (vcount[9:0] - weapon_y)) <= 256 && ((hcount[10:1] - weapon_x)*(hcount[10:1] - weapon_x) + (vcount[9:0] - weapon_y) * (vcount[9:0] - weapon_y)) > 100 && weapon_attack) begin
	ball_enable_1 <= 1;
      end
      else begin
	   ball_enable_1 <= 0;
      end
    end
//**********************************************************************************************************************************************//
//
//
// Weapon ball Enemy
//
//**********************************************************************************************************************************************//


  always_ff @(posedge clk) begin

      if (((hcount[10:1] - weapon_enemy_x)*(hcount[10:1] - weapon_enemy_x) + (vcount[9:0] - weapon_enemy_y) * (vcount[9:0] - weapon_enemy_y)) <= 100 && weapon_attack_enemy) begin
	ball_enable_enemy_0 <= 1;
      end 
      else begin
	   ball_enable_enemy_0 <= 0;
      end     
      
      if (((hcount[10:1] - weapon_enemy_x)*(hcount[10:1] - weapon_enemy_x) + (vcount[9:0] - weapon_enemy_y) * (vcount[9:0] - weapon_enemy_y)) <= 256 && ((hcount[10:1] - weapon_enemy_x)*(hcount[10:1] - weapon_enemy_x) + (vcount[9:0] - weapon_enemy_y) * (vcount[9:0] - weapon_enemy_y)) > 100 && weapon_attack_enemy) begin
	ball_enable_enemy_1 <= 1;
      end
      else begin
	   ball_enable_enemy_1 <= 0;
      end
    end
//**********************************************************************************************************************************************//
//
//
// protect enemy
//
//**********************************************************************************************************************************************//
  always_ff @(posedge clk) begin

      if (((hcount[10:1] - (ENEMY_X + PROTECT_BIAS))*(hcount[10:1] - (ENEMY_X + PROTECT_BIAS)) + (vcount[9:0] - (ENEMY_Y + PROTECT_BIAS))*(vcount[9:0] - (ENEMY_Y + PROTECT_BIAS))) < 9216 && ((hcount[10:1] - (ENEMY_X + PROTECT_BIAS))*(hcount[10:1] - (ENEMY_X + PROTECT_BIAS)) + (vcount[9:0] - (ENEMY_Y + PROTECT_BIAS))*(vcount[9:0] - (ENEMY_Y + PROTECT_BIAS))) > 8464 && weapon_protect_enemy) begin
	protect_enable_enemy <= 1;
      end  
      else begin
	   protect_enable_enemy <= 0;
      end
    end

//**********************************************************************************************************************************************//
//
//
// protect elf
//
//**********************************************************************************************************************************************//


  always_ff @(posedge clk) begin

      if (((hcount[10:1] - (ELF_X + PROTECT_BIAS))*(hcount[10:1] - (ELF_X + PROTECT_BIAS)) + (vcount[9:0] - (ELF_Y + PROTECT_BIAS))*(vcount[9:0] - (ELF_Y + PROTECT_BIAS))) < 9216 && ((hcount[10:1] - (ELF_X + PROTECT_BIAS))*(hcount[10:1] - (ELF_X + PROTECT_BIAS)) + (vcount[9:0] - (ELF_Y + PROTECT_BIAS))*(vcount[9:0] - (ELF_Y + PROTECT_BIAS))) > 8464 && weapon_protect_elf) begin
	protect_enable_elf <= 1;
      end  
      else begin
	   protect_enable_elf <= 0;
      end
    end

//**********************************************************************************************************************************************//
//
//
// wall 
//
//**********************************************************************************************************************************************//


  always_ff @(posedge clk) begin

      if (vcount[9:0] < 32 | vcount[9:0] > 415) begin
        
        wall_en <= 2'b1;
	if(wall_v == 0)
	  wall_address <= hcount[10:1] % 32 + vcount[4:0] * 32 + 4096;
	else
	  wall_address <= hcount[10:1] % 32 + vcount[4:0] * 32 + 5120;
      end
      else begin

        wall_en <= 2'b0;

      end

    end

  always_ff @(posedge clk) begin

      if ((vcount[9:0] >= 32 && vcount[9:0] <= 447) && (hcount[10:1] < 32 | hcount[10:1] > 609)) begin
        
        wall_1_en <= 2'b1;
	if(wall_v == 0)
	  wall_address_1 <= hcount[10:1] % 32 + vcount[4:0] * 32 + 3072; ////this sentence is really wierd and sb, only 3072 cannot be used.
	else
	  wall_address_1 <= hcount[10:1] % 32 + vcount[4:0] * 32 + 2048;
      end
      else begin

        wall_1_en <= 2'b0;

      end

    end

  always_ff @(posedge clk) begin
      
      if(fire_count == 0)begin
        if ((hcount[10:1] > 31 && hcount[10:1] < 608) && 0) begin
        
          fire_en <= 2'b1;
	  if(fire_f == 0)
	    fire_0_address <= hcount[10:1] % 32 + vcount[4:0] * 32;
	  else
	    fire_0_address <= hcount[10:1] % 32 + vcount[4:0] * 32 + 1024;
        end
        else begin

          fire_en <= 2'b0;

        end
      end
      else begin
        if ((hcount[10:1] > 31 && hcount[10:1] < 608) && (vcount[9:0] > (415 - 64) && vcount[9:0] < 415)) begin
        
          fire_en <= 2'b1;
	  if(fire_f == 0)
	    fire_0_address <= hcount[10:1] % 32 + (vcount[9:0] - 351)/2 * 32 + 3072;
	  else
	    fire_0_address <= hcount[10:1] % 32 + (vcount[9:0] - 351)/2 * 32 + 1024;
        end
        else begin

          fire_en <= 2'b0;

        end
      end
      
    end
  always_ff @(posedge clk) begin
        if ((hcount[10:1] > 31 && hcount[10:1] < 64 && atk_azhdaha[0]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[0] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[0] <= 0;

        end
        if ((hcount[10:1] > 63 && hcount[10:1] < 96 && atk_azhdaha[1]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[1] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[1] <= 0;

        end
        if ((hcount[10:1] > 95 && hcount[10:1] < 128 && atk_azhdaha[2]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[2] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[2] <= 0;

        end
        if ((hcount[10:1] > 127 && hcount[10:1] < 160 && atk_azhdaha[3]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[3] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[3] <= 0;

        end
        if ((hcount[10:1] > 159 && hcount[10:1] < 192 && atk_azhdaha[4]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin

          fire_rand_en[4] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[4] <= 0;

        end
        if ((hcount[10:1] > 191 && hcount[10:1] < 224  && atk_azhdaha[5]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[5] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[5] <= 0;

        end
        if ((hcount[10:1] > 223 && hcount[10:1] < 256 && atk_azhdaha[6]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[6] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[6] <= 0;

        end
        if ((hcount[10:1] > 255 && hcount[10:1] < 288 && atk_azhdaha[7]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[7] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[7] <= 0;

        end
        if ((hcount[10:1] > 287 && hcount[10:1] < 320 && atk_azhdaha[8]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[8] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[8] <= 0;

        end
        if ((hcount[10:1] > 319 && hcount[10:1] < 352 && atk_azhdaha[9]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[9] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[9] <= 0;

        end
        if ((hcount[10:1] > 351 && hcount[10:1] < 384 && atk_azhdaha[10]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[10] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[10] <= 0;

        end
        if ((hcount[10:1] > 383 && hcount[10:1] < 416 && atk_azhdaha[11]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[11] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[11] <= 0;

        end
        if ((hcount[10:1] > 415 && hcount[10:1] < 448 && atk_azhdaha[12]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[12] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[12] <= 0;

        end
        if ((hcount[10:1] > 447 && hcount[10:1] < 480 && atk_azhdaha[13]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[13] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[13] <= 0;

        end
        if ((hcount[10:1] > 479 && hcount[10:1] < 512 && atk_azhdaha[14]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[14] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[14] <= 0;

        end
        if ((hcount[10:1] > 511 && hcount[10:1] < 544 && atk_azhdaha[15]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[15] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[15] <= 0;

        end
        if ((hcount[10:1] > 543 && hcount[10:1] < 576 && atk_azhdaha[16]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[16] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[16] <= 0;

        end
        if ((hcount[10:1] > 575 && hcount[10:1] < 608 && atk_azhdaha[17]) && (vcount[9:0] > (415 -12*32) && vcount[9:0] < 415)) begin
        
          fire_rand_en[17] <= 1;
	  if(fire_f == 0)
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32;
	  else
	    fire_address <= hcount[10:1] % 32 + (vcount[9:0] - (415 -12*32))/12 * 32 + 1024;
        end
        else begin

          fire_rand_en[17] <= 0;

        end
    end
//**********************************************************************************************************************************************//
//
//
// chars
//
//**********************************************************************************************************************************************//

  always_ff @(posedge clk) begin

      if ((hcount[10:1] >= 560 && hcount[10:1] < 576) && (vcount[9:0] < 160 && vcount[9:0] > 143)) begin
        
        char_en <= 2'b1;
	char_address <= {(char[7:0]/100),vcount[3:0],hcount[4:1]};
      end
      else if ((hcount[10:1] >= 576 && hcount[10:1] < 592) && (vcount[9:0] < 160 && vcount[9:0] > 143)) begin
        
        char_en <= 2'b1;
	char_address <= {(char[7:0]/10)%10,vcount[3:0],hcount[4:1]};
      end
      else if ((hcount[10:1] >= 592 && hcount[10:1] < 608) && (vcount[9:0] < 160 && vcount[9:0] > 143)) begin
        
        char_en <= 2'b1;
	char_address <= {(char[7:0]%10),vcount[3:0],hcount[4:1]};
      end
      else if ((hcount[10:1] >= 560 && hcount[10:1] < 576) && (vcount[9:0] < 128 && vcount[9:0] > 111)) begin
        
        char_en <= 2'b1;
	char_address <= {(char[15:8]/100),vcount[3:0],hcount[4:1]};
      end
      else if ((hcount[10:1] >= 576 && hcount[10:1] < 592) && (vcount[9:0] < 128 && vcount[9:0] > 111)) begin
        
        char_en <= 2'b1;
	char_address <= {((char[15:8]/10)%10),vcount[3:0],hcount[4:1]};
      end
      else if ((hcount[10:1] >= 592 && hcount[10:1] < 608) && (vcount[9:0] < 128 && vcount[9:0] > 111)) begin
        
        char_en <= 2'b1;
	char_address <= {(char[15:8]%10),vcount[3:0],hcount[4:1]};
      end
      else if ((hcount[10:1] >= 560 && hcount[10:1] < 576) && (vcount[9:0] < 96 && vcount[9:0] > 79)) begin
        
        char_en <= 2'b1;
	char_address <= {(char[23:16]/100),vcount[3:0],hcount[4:1]};
      end
      else if ((hcount[10:1] >= 576 && hcount[10:1] < 592) && (vcount[9:0] < 96 && vcount[9:0] > 79)) begin
        
        char_en <= 2'b1;
	char_address <= {((char[23:16]/10)%10),vcount[3:0],hcount[4:1]};
      end
      else if ((hcount[10:1] >= 592 && hcount[10:1] < 608) && (vcount[9:0] < 96 && vcount[9:0] > 79)) begin
        
        char_en <= 2'b1;
	char_address <= {(char[23:16]%10),vcount[3:0],hcount[4:1]};
      end
      else if ((hcount[10:1] >= 560 && hcount[10:1] < 576) && (vcount[9:0] < 48 && vcount[9:0] > 31)) begin
        
        char_en <= 2'b1;
	char_address <= {(char[31:24]/100),vcount[3:0],hcount[4:1]};
      end
      else if ((hcount[10:1] >= 576 && hcount[10:1] < 592) && (vcount[9:0] < 48 && vcount[9:0] > 31)) begin
        
        char_en <= 2'b1;
	char_address <= {((char[31:24]/10)%10),vcount[3:0],hcount[4:1]};
      end
      else if ((hcount[10:1] >= 592 && hcount[10:1] < 608) && (vcount[9:0] < 48 && vcount[9:0] > 31)) begin
        
        char_en <= 2'b1;
	char_address <= {((char[31:24]%10)),vcount[3:0],hcount[4:1]};
      end
      else begin

        char_en <= 2'b0;

      end

    end
//----------------Sound-----------------
parameter int DIV_VALUES[9] = '{
    	  20'h0,
    	  20'h2E978, // DOC4
    	  20'h29919, // RED4
    	  20'h25085, // MIE4
    	  20'h22F44, // FAF4
   	  20'h1F23F, // SOG4
   	  20'h1BBE4, // LAA4
    	  20'h18B77, // TIB4
    	  20'h17545  // DOC5
	};

	logic [19:0] freq_counter;
	logic freq_clock;
        logic [5:0] tune;
	
	always_ff @(posedge clk) begin
    	  if (reset) begin
            freq_counter <= 0;
            freq_clock <= 0;
          end 
	  else begin
            if ((freq_counter == (DIV_VALUES[tune] >> 1) - 1) && tune) begin
              freq_counter <= 0;
              freq_clock <= ~freq_clock;
            end 
	    else begin
              freq_counter <= freq_counter + 1;
            end
    	  end
	end

	always_ff @(posedge clk) begin
	  if(reset) begin
	    sample_valid_l <= 0; sample_valid_r <= 0;
	  end
	  else if(left_chan_ready == 1 && right_chan_ready == 1 && tune != 0) begin

	    sample_valid_l <= 1; sample_valid_r <= 1;
	    sample_data_l <= 16'h8fff * freq_counter;
     	    sample_data_r <= 16'h8fff * freq_counter;
	  end
	  else begin
	    sample_valid_l <= 0; sample_valid_r <= 0;
	  end
	end


	
 always_comb begin
      {VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0};
   if (VGA_BLANK_n ) begin
    if(SLOT_en && font_output[font_pix_idx])begin
	{VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    end
    else if(CATCH_en && font_output[font_pix_idx])begin
	{VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    end
    else if(BATTLE_en && font_output[font_pix_idx])begin
	{VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    end
    else if(BOSS_en && font_output[font_pix_idx])begin
	{VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    end
    else if(ONLINE_en && font_output[font_pix_idx])begin
	{VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    end
    else if(LV_en && font_output[font_pix_idx])begin
	{VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
    end    
    else if(COIN_en && font_output[font_pix_idx])begin
	{VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
    end
    else if(DEF_en && font_output[font_pix_idx])begin
	{VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
    end
    else if(ATK_en && font_output[font_pix_idx])begin
	{VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
    end
    else if(msg_display[1])begin
	if(msg_visible[1] * font_output[font_pix_idx])
	  {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
	else 
	  {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    end
    else if(msg_display[2])begin
	if(msg_visible[2] * font_output[font_pix_idx])
	  {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
	else 
	  {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    end
    else if(msg_display[3])begin
	if(msg_visible[3] * font_output[font_pix_idx])
	  {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
	else 
	  {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    end
    else if(msg_display[4])begin
	if(msg_visible[4] * font_output[font_pix_idx])
	  {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
	else 
	  {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    end
    else if(msg_display[5])begin
	if(msg_visible[5] * font_output[font_pix_idx])
	  {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
	else 
	  {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    end
    else if(msg_display[6])begin
	if(msg_visible[6] * font_output[font_pix_idx])
	  {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
	else 
	  {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    end
    else if(msg_display[7])begin
	if(msg_visible[7] * font_output[font_pix_idx])
	  {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
	else 
	  {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    end
    else if(msg_display[8])begin
	if(msg_visible[8] * font_output[font_pix_idx])
	  {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
	else 
	  {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    end
    else if(msg_display[9])begin
	if(msg_visible[9] * font_output[font_pix_idx])
	  {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
	else 
	  {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    end
    else if(msg_display[10])begin
	if(msg_visible[10] * font_output[font_pix_idx])
	  {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
	else 
	  {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    end
    else if(msg_display[11])begin
	if(msg_visible[11] * font_output[font_pix_idx])
	  {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
	else 
	  {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    end
    else if(msg_display[12])begin
	if(msg_visible[12] * font_output[font_pix_idx])
	  {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
	else 
	  {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    end
    else if(msg_display[13])begin
	if(msg_visible[13] * font_output[font_pix_idx])
	  {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
	else 
	  {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    end
    else if(slot0_en)begin
	    case(slot0_output * slot_v)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h01: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h40};
    8'h02: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h80};
    8'h03: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hC0};
    8'h04: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hFF};
    8'h05: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h00};
    8'h06: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h40};
    8'h07: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h80};
    8'h08: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hC0};
    8'h09: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hFF};
    8'h0A: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h00};
    8'h0B: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h40};
    8'h0C: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h80};
    8'h0D: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hC0};
    8'h0E: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hFF};
    8'h0F: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h00};
    8'h10: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h40};
    8'h11: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h80};
    8'h12: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hC0};
    8'h13: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hFF};
    8'h14: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h00};
    8'h15: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h40};
    8'h16: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h80};
    8'h17: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hC0};
    8'h18: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hFF};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h20: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h80};
    8'h21: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hC0};
    8'h22: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hFF};
    8'h23: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h00};
    8'h24: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h40};
    8'h25: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h80};
    8'h26: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hC0};
    8'h27: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hFF};
    8'h28: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h00};
    8'h29: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h40};
    8'h2A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h80};
    8'h2B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hC0};
    8'h2C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hFF};
    8'h2D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h00};
    8'h2E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h40};
    8'h2F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h80};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h40: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hFF};
    8'h41: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h00};
    8'h42: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h40};
    8'h43: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h80};
    8'h44: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hC0};
    8'h45: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hFF};
    8'h46: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h00};
    8'h47: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h40};
    8'h48: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h80};
    8'h49: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hC0};
    8'h4A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hFF};
    8'h4B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h00};
    8'h4C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h40};
    8'h4D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h80};
    8'h4E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hC0};
    8'h4F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hFF};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h60: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h40};
    8'h61: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h80};
    8'h62: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hC0};
    8'h63: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hFF};
    8'h64: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h00};
    8'h65: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h40};
    8'h66: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h80};
    8'h67: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hC0};
    8'h68: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hFF};
    8'h69: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h00};
    8'h6A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h40};
    8'h6B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h80};
    8'h6C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hC0};
    8'h6D: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hFF};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(slot1_en)begin
	    case(slot1_output * slot_v)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h01: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h40};
    8'h02: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h80};
    8'h03: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hC0};
    8'h04: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hFF};
    8'h05: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h00};
    8'h06: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h40};
    8'h07: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h80};
    8'h08: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hC0};
    8'h09: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hFF};
    8'h0A: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h00};
    8'h0B: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h40};
    8'h0C: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h80};
    8'h0D: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hC0};
    8'h0E: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hFF};
    8'h0F: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h00};
    8'h10: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h40};
    8'h11: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h80};
    8'h12: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hC0};
    8'h13: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hFF};
    8'h14: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h00};
    8'h15: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h40};
    8'h16: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h80};
    8'h17: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hC0};
    8'h18: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hFF};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h20: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h80};
    8'h21: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hC0};
    8'h22: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hFF};
    8'h23: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h00};
    8'h24: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h40};
    8'h25: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h80};
    8'h26: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hC0};
    8'h27: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hFF};
    8'h28: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h00};
    8'h29: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h40};
    8'h2A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h80};
    8'h2B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hC0};
    8'h2C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hFF};
    8'h2D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h00};
    8'h2E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h40};
    8'h2F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h80};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h40: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hFF};
    8'h41: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h00};
    8'h42: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h40};
    8'h43: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h80};
    8'h44: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hC0};
    8'h45: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hFF};
    8'h46: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h00};
    8'h47: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h40};
    8'h48: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h80};
    8'h49: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hC0};
    8'h4A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hFF};
    8'h4B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h00};
    8'h4C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h40};
    8'h4D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h80};
    8'h4E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hC0};
    8'h4F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hFF};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h60: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h40};
    8'h61: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h80};
    8'h62: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hC0};
    8'h63: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hFF};
    8'h64: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h00};
    8'h65: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h40};
    8'h66: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h80};
    8'h67: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hC0};
    8'h68: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hFF};
    8'h69: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h00};
    8'h6A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h40};
    8'h6B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h80};
    8'h6C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hC0};
    8'h6D: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hFF};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(slot2_en)begin
	    case(slot2_output * slot_v)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h01: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h40};
    8'h02: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h80};
    8'h03: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hC0};
    8'h04: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hFF};
    8'h05: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h00};
    8'h06: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h40};
    8'h07: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h80};
    8'h08: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hC0};
    8'h09: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hFF};
    8'h0A: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h00};
    8'h0B: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h40};
    8'h0C: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h80};
    8'h0D: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hC0};
    8'h0E: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hFF};
    8'h0F: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h00};
    8'h10: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h40};
    8'h11: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h80};
    8'h12: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hC0};
    8'h13: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hFF};
    8'h14: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h00};
    8'h15: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h40};
    8'h16: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h80};
    8'h17: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hC0};
    8'h18: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hFF};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h20: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h80};
    8'h21: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hC0};
    8'h22: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hFF};
    8'h23: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h00};
    8'h24: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h40};
    8'h25: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h80};
    8'h26: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hC0};
    8'h27: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hFF};
    8'h28: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h00};
    8'h29: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h40};
    8'h2A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h80};
    8'h2B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hC0};
    8'h2C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hFF};
    8'h2D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h00};
    8'h2E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h40};
    8'h2F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h80};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h40: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hFF};
    8'h41: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h00};
    8'h42: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h40};
    8'h43: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h80};
    8'h44: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hC0};
    8'h45: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hFF};
    8'h46: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h00};
    8'h47: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h40};
    8'h48: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h80};
    8'h49: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hC0};
    8'h4A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hFF};
    8'h4B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h00};
    8'h4C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h40};
    8'h4D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h80};
    8'h4E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hC0};
    8'h4F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hFF};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h60: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h40};
    8'h61: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h80};
    8'h62: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hC0};
    8'h63: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hFF};
    8'h64: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h00};
    8'h65: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h40};
    8'h66: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h80};
    8'h67: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hC0};
    8'h68: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hFF};
    8'h69: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h00};
    8'h6A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h40};
    8'h6B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h80};
    8'h6C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hC0};
    8'h6D: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hFF};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(wall_en && (wall_output != 8'h00))begin
	    case(wall_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h01: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h40};
    8'h02: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h80};
    8'h03: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hC0};
    8'h04: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hFF};
    8'h05: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h00};
    8'h06: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h40};
    8'h07: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h80};
    8'h08: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hC0};
    8'h09: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hFF};
    8'h0A: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h00};
    8'h0B: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h40};
    8'h0C: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h80};
    8'h0D: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hC0};
    8'h0E: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hFF};
    8'h0F: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h00};
    8'h10: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h40};
    8'h11: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h80};
    8'h12: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hC0};
    8'h13: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hFF};
    8'h14: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h00};
    8'h15: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h40};
    8'h16: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h80};
    8'h17: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hC0};
    8'h18: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hFF};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h20: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h80};
    8'h21: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hC0};
    8'h22: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hFF};
    8'h23: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h00};
    8'h24: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h40};
    8'h25: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h80};
    8'h26: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hC0};
    8'h27: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hFF};
    8'h28: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h00};
    8'h29: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h40};
    8'h2A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h80};
    8'h2B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hC0};
    8'h2C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hFF};
    8'h2D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h00};
    8'h2E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h40};
    8'h2F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h80};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h40: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hFF};
    8'h41: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h00};
    8'h42: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h40};
    8'h43: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h80};
    8'h44: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hC0};
    8'h45: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hFF};
    8'h46: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h00};
    8'h47: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h40};
    8'h48: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h80};
    8'h49: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hC0};
    8'h4A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hFF};
    8'h4B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h00};
    8'h4C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h40};
    8'h4D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h80};
    8'h4E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hC0};
    8'h4F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hFF};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h60: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h40};
    8'h61: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h80};
    8'h62: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hC0};
    8'h63: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hFF};
    8'h64: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h00};
    8'h65: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h40};
    8'h66: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h80};
    8'h67: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hC0};
    8'h68: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hFF};
    8'h69: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h00};
    8'h6A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h40};
    8'h6B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h80};
    8'h6C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hC0};
    8'h6D: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hFF};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(wall_1_en && (wall_output_1 != 8'h00))begin
	    case(wall_output_1)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h01: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h40};
    8'h02: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h80};
    8'h03: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hC0};
    8'h04: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hFF};
    8'h05: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h00};
    8'h06: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h40};
    8'h07: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h80};
    8'h08: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hC0};
    8'h09: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hFF};
    8'h0A: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h00};
    8'h0B: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h40};
    8'h0C: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h80};
    8'h0D: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hC0};
    8'h0E: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hFF};
    8'h0F: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h00};
    8'h10: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h40};
    8'h11: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h80};
    8'h12: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hC0};
    8'h13: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hFF};
    8'h14: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h00};
    8'h15: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h40};
    8'h16: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h80};
    8'h17: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hC0};
    8'h18: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hFF};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h20: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h80};
    8'h21: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hC0};
    8'h22: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hFF};
    8'h23: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h00};
    8'h24: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h40};
    8'h25: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h80};
    8'h26: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hC0};
    8'h27: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hFF};
    8'h28: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h00};
    8'h29: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h40};
    8'h2A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h80};
    8'h2B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hC0};
    8'h2C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hFF};
    8'h2D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h00};
    8'h2E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h40};
    8'h2F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h80};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h40: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hFF};
    8'h41: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h00};
    8'h42: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h40};
    8'h43: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h80};
    8'h44: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hC0};
    8'h45: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hFF};
    8'h46: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h00};
    8'h47: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h40};
    8'h48: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h80};
    8'h49: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hC0};
    8'h4A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hFF};
    8'h4B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h00};
    8'h4C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h40};
    8'h4D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h80};
    8'h4E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hC0};
    8'h4F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hFF};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h60: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h40};
    8'h61: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h80};
    8'h62: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hC0};
    8'h63: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hFF};
    8'h64: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h00};
    8'h65: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h40};
    8'h66: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h80};
    8'h67: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hC0};
    8'h68: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hFF};
    8'h69: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h00};
    8'h6A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h40};
    8'h6B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h80};
    8'h6C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hC0};
    8'h6D: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hFF};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(weapon_en)begin
	    case(weapon_output * weapon_attack)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h01: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h40};
    8'h02: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h80};
    8'h03: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hC0};
    8'h04: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hFF};
    8'h05: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h00};
    8'h06: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h40};
    8'h07: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h80};
    8'h08: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hC0};
    8'h09: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hFF};
    8'h0A: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h00};
    8'h0B: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h40};
    8'h0C: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h80};
    8'h0D: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hC0};
    8'h0E: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hFF};
    8'h0F: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h00};
    8'h10: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h40};
    8'h11: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h80};
    8'h12: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hC0};
    8'h13: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hFF};
    8'h14: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h00};
    8'h15: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h40};
    8'h16: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h80};
    8'h17: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hC0};
    8'h18: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hFF};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h20: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h80};
    8'h21: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hC0};
    8'h22: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hFF};
    8'h23: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h00};
    8'h24: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h40};
    8'h25: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h80};
    8'h26: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hC0};
    8'h27: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hFF};
    8'h28: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h00};
    8'h29: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h40};
    8'h2A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h80};
    8'h2B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hC0};
    8'h2C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hFF};
    8'h2D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h00};
    8'h2E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h40};
    8'h2F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h80};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h40: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hFF};
    8'h41: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h00};
    8'h42: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h40};
    8'h43: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h80};
    8'h44: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hC0};
    8'h45: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hFF};
    8'h46: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h00};
    8'h47: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h40};
    8'h48: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h80};
    8'h49: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hC0};
    8'h4A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hFF};
    8'h4B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h00};
    8'h4C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h40};
    8'h4D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h80};
    8'h4E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hC0};
    8'h4F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hFF};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h60: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h40};
    8'h61: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h80};
    8'h62: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hC0};
    8'h63: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hFF};
    8'h64: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h00};
    8'h65: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h40};
    8'h66: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h80};
    8'h67: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hC0};
    8'h68: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hFF};
    8'h69: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h00};
    8'h6A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h40};
    8'h6B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h80};
    8'h6C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hC0};
    8'h6D: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hFF};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(elf_en && (elf_output != 8'h7C))begin
	    case(elf_output * elf_v)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h01: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h40};
    8'h02: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h80};
    8'h03: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hC0};
    8'h04: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hFF};
    8'h05: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h00};
    8'h06: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h40};
    8'h07: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h80};
    8'h08: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hC0};
    8'h09: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hFF};
    8'h0A: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h00};
    8'h0B: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h40};
    8'h0C: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h80};
    8'h0D: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hC0};
    8'h0E: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hFF};
    8'h0F: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h00};
    8'h10: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h40};
    8'h11: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h80};
    8'h12: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hC0};
    8'h13: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hFF};
    8'h14: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h00};
    8'h15: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h40};
    8'h16: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h80};
    8'h17: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hC0};
    8'h18: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hFF};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h20: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h80};
    8'h21: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hC0};
    8'h22: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hFF};
    8'h23: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h00};
    8'h24: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h40};
    8'h25: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h80};
    8'h26: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hC0};
    8'h27: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hFF};
    8'h28: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h00};
    8'h29: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h40};
    8'h2A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h80};
    8'h2B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hC0};
    8'h2C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hFF};
    8'h2D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h00};
    8'h2E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h40};
    8'h2F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h80};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h40: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hFF};
    8'h41: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h00};
    8'h42: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h40};
    8'h43: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h80};
    8'h44: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hC0};
    8'h45: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hFF};
    8'h46: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h00};
    8'h47: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h40};
    8'h48: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h80};
    8'h49: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hC0};
    8'h4A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hFF};
    8'h4B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h00};
    8'h4C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h40};
    8'h4D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h80};
    8'h4E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hC0};
    8'h4F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hFF};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h60: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h40};
    8'h61: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h80};
    8'h62: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hC0};
    8'h63: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hFF};
    8'h64: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h00};
    8'h65: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h40};
    8'h66: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h80};
    8'h67: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hC0};
    8'h68: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hFF};
    8'h69: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h00};
    8'h6A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h40};
    8'h6B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h80};
    8'h6C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hC0};
    8'h6D: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hFF};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(enemy_en && (enemy_output != 8'h7C))begin
	    case(enemy_output * enemy_v)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h01: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h40};
    8'h02: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h80};
    8'h03: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hC0};
    8'h04: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hFF};
    8'h05: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h00};
    8'h06: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h40};
    8'h07: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h80};
    8'h08: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hC0};
    8'h09: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hFF};
    8'h0A: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h00};
    8'h0B: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h40};
    8'h0C: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h80};
    8'h0D: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hC0};
    8'h0E: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hFF};
    8'h0F: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h00};
    8'h10: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h40};
    8'h11: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h80};
    8'h12: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hC0};
    8'h13: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hFF};
    8'h14: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h00};
    8'h15: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h40};
    8'h16: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h80};
    8'h17: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hC0};
    8'h18: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hFF};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h20: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h80};
    8'h21: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hC0};
    8'h22: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hFF};
    8'h23: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h00};
    8'h24: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h40};
    8'h25: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h80};
    8'h26: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hC0};
    8'h27: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hFF};
    8'h28: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h00};
    8'h29: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h40};
    8'h2A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h80};
    8'h2B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hC0};
    8'h2C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hFF};
    8'h2D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h00};
    8'h2E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h40};
    8'h2F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h80};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h40: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hFF};
    8'h41: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h00};
    8'h42: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h40};
    8'h43: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h80};
    8'h44: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hC0};
    8'h45: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hFF};
    8'h46: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h00};
    8'h47: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h40};
    8'h48: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h80};
    8'h49: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hC0};
    8'h4A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hFF};
    8'h4B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h00};
    8'h4C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h40};
    8'h4D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h80};
    8'h4E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hC0};
    8'h4F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hFF};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h60: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h40};
    8'h61: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h80};
    8'h62: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hC0};
    8'h63: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hFF};
    8'h64: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h00};
    8'h65: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h40};
    8'h66: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h80};
    8'h67: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hC0};
    8'h68: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hFF};
    8'h69: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h00};
    8'h6A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h40};
    8'h6B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h80};
    8'h6C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hC0};
    8'h6D: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hFF};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(azhdaha_en && (azhdaha_output != 8'h00))begin
      case(azhdaha_output * azhdaha_v)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h01: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h40};
    8'h02: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h80};
    8'h03: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hC0};
    8'h04: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hFF};
    8'h05: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h00};
    8'h06: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h40};
    8'h07: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h80};
    8'h08: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hC0};
    8'h09: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hFF};
    8'h0A: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h00};
    8'h0B: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h40};
    8'h0C: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h80};
    8'h0D: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hC0};
    8'h0E: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hFF};
    8'h0F: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h00};
    8'h10: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h40};
    8'h11: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h80};
    8'h12: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hC0};
    8'h13: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hFF};
    8'h14: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h00};
    8'h15: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h40};
    8'h16: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h80};
    8'h17: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hC0};
    8'h18: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hFF};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h20: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h80};
    8'h21: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hC0};
    8'h22: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hFF};
    8'h23: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h00};
    8'h24: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h40};
    8'h25: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h80};
    8'h26: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hC0};
    8'h27: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hFF};
    8'h28: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h00};
    8'h29: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h40};
    8'h2A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h80};
    8'h2B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hC0};
    8'h2C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hFF};
    8'h2D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h00};
    8'h2E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h40};
    8'h2F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h80};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h40: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hFF};
    8'h41: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h00};
    8'h42: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h40};
    8'h43: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h80};
    8'h44: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hC0};
    8'h45: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hFF};
    8'h46: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h00};
    8'h47: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h40};
    8'h48: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h80};
    8'h49: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hC0};
    8'h4A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hFF};
    8'h4B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h00};
    8'h4C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h40};
    8'h4D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h80};
    8'h4E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hC0};
    8'h4F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hFF};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h60: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h40};
    8'h61: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h80};
    8'h62: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hC0};
    8'h63: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hFF};
    8'h64: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h00};
    8'h65: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h40};
    8'h66: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h80};
    8'h67: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hC0};
    8'h68: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hFF};
    8'h69: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h00};
    8'h6A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h40};
    8'h6B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h80};
    8'h6C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hC0};
    8'h6D: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hFF};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
      endcase	
    end
    else if(char_en)begin
      case(char_output)
        8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
        8'h01: {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
      endcase	
    end
    else if(target_en)begin
      case(target_output * target_v)
        8'h00: {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
        8'h01: {VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'h00};
      endcase
    end	
    else if(ball_enable_0 && weapon_t == 3)begin
	{VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    end
    else if(ball_enable_1 && weapon_t == 3)begin
	{VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h00};
    end
    else if(ball_enable_enemy_0 && weapon_enemy_p == 0)begin
	{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hF0};
    end
    else if(ball_enable_enemy_1 && weapon_enemy_p == 0)begin
	{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hF0};
    end
    else if(protect_enable_enemy)begin
	{VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    end
    else if(protect_enable_elf)begin
	{VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    end 
    else if(blood_elf_en)begin
	{VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    end 
    else if(blood_elf_shadow_en)begin
	{VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    end
    else if(blood_enemy_en)begin
	{VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    end
    else if(blood_enemy_shadow_en)begin
	{VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    end
    else if(blood_azhdaha_en)begin
	{VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h00};
    end
    else if(blood_azhdaha_shadow_en)begin
	{VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hF0, 8'hC0};
    end
    else if(fire_en && (fire_0_output != 8'h00))begin
	    case(fire_0_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h01: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h40};
    8'h02: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h80};
    8'h03: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hC0};
    8'h04: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hFF};
    8'h05: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h00};
    8'h06: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h40};
    8'h07: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'h80};
    8'h08: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hC0};
    8'h09: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h40, 8'hFF};
    8'h0A: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h00};
    8'h0B: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h40};
    8'h0C: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'h80};
    8'h0D: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hC0};
    8'h0E: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h80, 8'hFF};
    8'h0F: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h00};
    8'h10: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h40};
    8'h11: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'h80};
    8'h12: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hC0};
    8'h13: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hC0, 8'hFF};
    8'h14: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h00};
    8'h15: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h40};
    8'h16: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'h80};
    8'h17: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hC0};
    8'h18: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hFF, 8'hFF};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h20: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h80};
    8'h21: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hC0};
    8'h22: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'hFF};
    8'h23: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h00};
    8'h24: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h40};
    8'h25: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'h80};
    8'h26: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hC0};
    8'h27: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h80, 8'hFF};
    8'h28: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h00};
    8'h29: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h40};
    8'h2A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'h80};
    8'h2B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hC0};
    8'h2C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hC0, 8'hFF};
    8'h2D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h00};
    8'h2E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h40};
    8'h2F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'h80};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h40: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hFF};
    8'h41: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h00};
    8'h42: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h40};
    8'h43: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'h80};
    8'h44: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hC0};
    8'h45: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hC0, 8'hFF};
    8'h46: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h00};
    8'h47: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h40};
    8'h48: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'h80};
    8'h49: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hC0};
    8'h4A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'hFF, 8'hFF};
    8'h4B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h00};
    8'h4C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h40};
    8'h4D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'h80};
    8'h4E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hC0};
    8'h4F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h00, 8'hFF};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h60: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h40};
    8'h61: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h80};
    8'h62: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hC0};
    8'h63: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'hFF};
    8'h64: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h00};
    8'h65: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h40};
    8'h66: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'h80};
    8'h67: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hC0};
    8'h68: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h00, 8'hFF};
    8'h69: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h00};
    8'h6A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h40};
    8'h6B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'h80};
    8'h6C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hC0};
    8'h6D: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h40, 8'hFF};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[0] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
else if(fire_rand_en[1] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[2] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[3] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[4] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[5] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[6] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[7] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[8] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[9] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[10] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[11] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[12] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[13] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[14] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[15] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[16] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
    else if(fire_rand_en[17] && (fire_output != 8'h00))begin
	    case(fire_output)
    8'h00: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
    8'h19: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h00};
    8'h1A: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h40};
    8'h1B: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'h80};
    8'h1C: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hC0};
    8'h1D: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h00, 8'hFF};
    8'h1E: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h00};
    8'h1F: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'h40, 8'h40};
    8'h30: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hC0};
    8'h31: {VGA_R, VGA_G, VGA_B} = {8'h40, 8'hFF, 8'hFF};
    8'h32: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h00};
    8'h33: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h40};
    8'h34: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'h80};
    8'h35: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hC0};
    8'h36: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h00, 8'hFF};
    8'h37: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h00};
    8'h38: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h40};
    8'h39: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'h80};
    8'h3A: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hC0};
    8'h3B: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h40, 8'hFF};
    8'h3C: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h00};
    8'h3D: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h40};
    8'h3E: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80};
    8'h3F: {VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'hC0};
    8'h50: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h00};
    8'h51: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h40};
    8'h52: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'h80};
    8'h53: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hC0};
    8'h54: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h40, 8'hFF};
    8'h55: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h00};
    8'h56: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h40};
    8'h57: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'h80};
    8'h58: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hC0};
    8'h59: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'h80, 8'hFF};
    8'h5A: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h00};
    8'h5B: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h40};
    8'h5C: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'h80};
    8'h5D: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hC0};
    8'h5E: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hC0, 8'hFF};
    8'h5F: {VGA_R, VGA_G, VGA_B} = {8'hC0, 8'hFF, 8'h00};
    8'h6E: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h00};
    8'h6F: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h40};
    8'h70: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'h80};
    8'h71: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hC0};
    8'h72: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'h80, 8'hFF};
    8'h73: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h00};
    8'h74: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h40};
    8'h75: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'h80};
    8'h76: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hC0};
    8'h77: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hC0, 8'hFF};
    8'h78: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h00};
    8'h79: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h40};
    8'h7A: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'h80};
    8'h7B: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hC0};
    8'h7C: {VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
    endcase	
	end
  end

/*
	if (hcount < 11'd_1120 && hcount >= 11'd_160 && vcount < 10'd_400)
	 		{VGA_R, VGA_G, VGA_B} = PPU_out;
	else
			{VGA_R, VGA_G, VGA_B} =
             			{8'h00, 8'h00, 8'h00};
*/	
end	       
endmodule

module vga_counters(
 input logic 	     clk50, reset,
 output logic [10:0] hcount,  // hcount[10:1] is pixel column
 output logic [9:0]  vcount,  // vcount[9:0] is pixel row
 output logic 	     VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC +
                            HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC +
                            VBACK_PORCH; // 525

   logic endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;
       
   logic endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) &
		      !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1'b0; // For putting sync on the green signal; unused
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: rising edge sensitive
   
endmodule
