/*
 * Avalon memory-mapped peripheral that generates VGA
 *
 * Stephen A. Edwards
 * Columbia University
 */

module cat_invaders(input logic        clk,
	        input logic 	    reset,

		input logic         left_channel_ready,
		input logic         right_channel_ready,

		input logic [15:0]  writedata,
		input logic 	    write,
		input 		    chipselect,
		input logic [7:0]   address,

    		output logic [15:0] readdata,

		output logic [7:0]  VGA_R, VGA_G, VGA_B,
		output logic 	    VGA_CLK, VGA_HS, VGA_VS,
		                    VGA_BLANK_n,
		output logic 	    VGA_SYNC_n,

		output logic        left_channel_valid,
		output logic        right_channel_valid,

		output logic [15:0] left_channel_data,
		output logic [15:0] right_channel_data,

                output logic        eof_irq

);
parameter CAT_SIZE = 32;
parameter CAT_MATRIX_SIZE_X = 352;
parameter CAT_MATRIX_SIZE_Y = 160;

/********************************************
   Register Group #0
   Dog position; and projectile position and visibility
*********************************************/
     logic [3:0] cat_bullet;              //cat bullet in flight?
     logic       dog_bullet;              //dog bullet in flight?
     logic [9:0] dog_pos_x;               //dog position x           
     logic [9:0] dog_project_x;           //dog bullet position x
     logic [9:0] dog_project_y;           //dog bullet position y
     logic [9:0] cat_project_x[3:0];      //cat 0~3 bullet position x   
     logic [9:0] cat_project_y[3:0];      //cat 0~3 bullet position y
     logic [9:0] cat_array_pos_x;         //cat upper west corner position x
     logic [9:0] cat_array_pos_y;         //cat upper west corner position y
     logic [9:0] mystery_pos_x;           //mystery position
     logic       mystery_vis;		  // mystery visible



/********************************************
   Register Group #1
   Miscellaneous 
*********************************************/
     logic [3:0]  level;           //game difficulty level 0~7
     logic [1:0]  game_status;     //game status (start, in process, game over)
     logic [1:0]  life;            //lives remaining 0~3
     logic [15:0] score;          //score
     logic [3:0]  dog_ani_state;
     logic        cat_ani_state;


ctrl_regs ctr_regs_0(.*);



   logic [10:0]	   hcount;
   logic [9:0]     vcount;
   logic           irq;
   logic [14:0]    frame_cnt;
   logic           frame_active;

   logic [7:0] 	   background_r, background_g, background_b;
   logic play, play_cat, play_dog;
   logic [3:0]  vol, vol_cat, vol_dog;
   logic [3:0] delay;
   
 
	
   vga_counters counters(.clk50(clk), .*);

   audio_player audio(.clk50(clk), .*);


/****************************************************************** 
display start
*******************************************************************/
logic [23:0] display_rgb;

logic [10:0] display_addr;

logic test;

logic [10:0] cat2_addr;
logic [10:0] cat3_addr;
logic [10:0] cat4_addr;
logic [10:0] explode_addr;
logic [15:0] poodle_addr;
logic [15:0] mystery_addr;
logic [15:0] barrier_addr;
logic [15:0] game_over_addr;
logic [15:0] num_addr;
logic [15:0] others_addr;
//logic [15:0] others_explosion_addr;
logic [15:0] cat_mat_addr;
logic [15:0] barrier_mat_addr;
logic [15:0] bone_addr;

logic [7:0] display_data;
logic [7:0] cat2_data;
logic [7:0] cat3_data;
logic [7:0] cat4_data;
logic [7:0] explode_data;
logic [7:0] poodle_data;
logic [7:0] mystery_data;
logic [7:0] barrier_data;
logic [7:0] game_over_data;
logic [7:0] num_data;
logic [7:0] others_data;
logic [7:0] cat_mat_data;
logic [7:0] barrier_mat_data;
logic [7:0] bone_data;

logic [7:0] output_data;

logic [15:0] cat_mat_s_x;
logic [15:0] cat_mat_s_y;

logic [15:0] cat_mat_e_x;
logic [15:0] cat_mat_e_y;

/////////////////////////mouse
logic [15:0] mouse1_s_x;
logic [15:0] mouse1_s_y;

logic [15:0] mouse1_e_x;
logic [15:0] mouse1_e_y;

logic [15:0] mouse2_s_x;
logic [15:0] mouse2_s_y;

logic [15:0] mouse2_e_x;
logic [15:0] mouse2_e_y;

logic [15:0] mouse3_s_x;
logic [15:0] mouse3_s_y;

logic [15:0] mouse3_e_x;
logic [15:0] mouse3_e_y;

/////////////////////////
//bone

logic [15:0] bone_s_x;
logic [15:0] bone_s_y;

logic [15:0] bone_e_x;
logic [15:0] bone_e_y;

//
logic [4:1] ccc;

logic b_write;
logic b_cs;
logic [17:0] b_write_addr;
logic [15:0] b_writedata;


logic c_write;
logic c_cs;
logic [17:0] c_write_addr;
logic [15:0] c_writedata;


soc_system_ROM_cat_1 cat_display(.address(display_addr), .clk(clk), .clken(1), .reset(reset), .reset_req(0), .readdata(display_data)); 
soc_system_ROM_cat_2 cat2_display(.address(cat2_addr), .clk(clk), .clken(1), .reset(reset), .reset_req(0), .readdata(cat2_data)); 
soc_system_ROM_cat_3 cat3_display(.address(cat3_addr), .clk(clk), .clken(1), .reset(reset), .reset_req(0), .readdata(cat3_data)); 
soc_system_ROM_cat_4 cat4_display(.address(cat4_addr), .clk(clk), .clken(1), .reset(reset), .reset_req(0), .readdata(cat4_data)); 
soc_system_ROM_explode explode_display(.address(explode_addr), .clk(clk), .clken(1), .reset(reset), .reset_req(0), .readdata(explode_data)); 
soc_system_cat_matrix cat_mat_display(.address(cat_mat_addr), .address2(c_write_addr), .chipselect2(c_cs), .clk(clk), .clken(1), .clken2(1), .reset(reset), .reset_req(0), .write2(c_write), .writedata2(c_writedata), .readdata(cat_mat_data)); 

soc_system_ROM_Poodles poodle_display(.address(poodle_addr), .clk(clk), .clken(1), .reset(reset), .reset_req(0), .readdata(poodle_data)); 
soc_system_ROM_mystery mystery_display(.address(mystery_addr), .clk(clk), .clken(1), .reset(reset), .reset_req(0), .readdata(mystery_data)); 

soc_system_ROM_barrier barrier_display(.address(barrier_addr), .clk(clk), .clken(1), .reset(reset), .reset_req(0), .readdata(barrier_data)); 
soc_system_barrier_matrix barrier_mat(.address(barrier_mat_addr), .address2(b_write_addr), .chipselect2(b_cs), .clk(clk), .clken(1), .clken2(1), .reset(reset), .reset_req(0), .write2(b_write), .writedata2(b_writedata), .readdata(barrier_mat_data)); 

soc_system_ROM_game_over game_over_display(.address(game_over_addr), .clk(clk), .clken(1), .reset(reset), .reset_req(0), .readdata(game_over_data)); 

soc_system_ROM_num num_display(.address(num_addr), .clk(clk), .clken(1), .reset(reset), .reset_req(0), .readdata(num_data)); 

soc_system_ROM_others others_display(.address(others_addr), .clk(clk), .clken(1), .reset(reset), .reset_req(0), .readdata(others_data)); 

soc_system_ROM_bone bone_display(.address(bone_addr), .clk(clk), .clken(1), .reset(reset), .reset_req(0), .readdata(bone_data)); 


/****************************************************************** 
display end
*******************************************************************/
/*******************************我的*****************************/
logic [4:0] cat_cnt_x;
logic [4:0] cat_cnt_y;
logic [10:0] cat_offset_x;
logic [10:0] cat_offset_y;
logic [10:0] cat_addr;
logic [15:0] explosion_addr;
logic in_box;

assign cat_cnt_x = ({1'b0,hcount[10:1]}-cat_mat_s_x+1)>>5;
assign cat_cnt_y = ({1'b0,vcount}-cat_mat_s_y)>>5;
assign cat_offset_x = ({1'b0,hcount[10:1]}-cat_mat_s_x[10:0]+1) & 11'h1f;
assign cat_offset_y = ({1'b0,vcount}-cat_mat_s_y[10:0]) & 11'h1f;
assign cat_addr = {cat_ani_state, cat_offset_y[4:0],cat_offset_x[4:0]};
//assign cat_addr = {1'd0,cat_offset_y[4:0],cat_offset_x[4:0]};

assign in_box = (hcount[10:1]>=(cat_mat_s_x-1)) && (hcount[10:1]<cat_mat_e_x) && (vcount>=cat_mat_s_y) && (vcount<cat_mat_e_y);
assign display_addr = {16{in_box}} & cat_addr;
assign cat2_addr= {16{in_box}} & cat_addr;
assign cat3_addr= {16{in_box}} & cat_addr;
assign cat4_addr= {16{in_box}} & cat_addr;
assign explode_addr = {16{in_box}} & cat_addr;
//assign explosion_addr= {16{in_box}} & cat_addr;
/*******************************我的*****************************/

/******************************mice***************************/
/* mouse */
	logic [15:0] mouse1_addr, mouse2_addr, mouse3_addr;
	logic [10:0] mouse1_offset_x;
	logic [10:0] mouse1_offset_y;
	logic [10:0] mouse2_offset_x;
	logic [10:0] mouse2_offset_y;
	logic [10:0] mouse3_offset_x;
	logic [10:0] mouse3_offset_y;
	logic in_box_mouse1;
	logic in_box_mouse2;
	logic in_box_mouse3;

	assign mouse1_offset_x = ({1'b0,hcount[10:1]}-mouse1_s_x[10:0]+2) & 11'h1f;
	assign mouse1_offset_y = ({1'b0,vcount}-mouse1_s_y[10:0]) & 11'h1f;
	assign mouse1_addr = {9'd0, mouse1_offset_y[3:0], mouse1_offset_x[2:0]};

	assign mouse2_offset_x = ({1'b0,hcount[10:1]}-mouse2_s_x[10:0]+2) & 11'h1f;
	assign mouse2_offset_y = ({1'b0,vcount}-mouse2_s_y[10:0]) & 11'h1f;
	assign mouse2_addr = {9'd0, mouse2_offset_y[3:0], mouse2_offset_x[2:0]};

	assign mouse3_offset_x = ({1'b0,hcount[10:1]}-mouse3_s_x[10:0]+2) & 11'h1f;
	assign mouse3_offset_y = ({1'b0,vcount}-mouse3_s_y[10:0]) & 11'h1f;
	assign mouse3_addr = {9'd0, mouse3_offset_y[3:0], mouse3_offset_x[2:0]};

	assign in_box_mouse1 = (hcount[10:1]>=(mouse1_s_x-2)) && (hcount[10:1]<mouse1_e_x) && (vcount>=mouse1_s_y) && (vcount<mouse1_e_y);
	assign in_box_mouse2 = (hcount[10:1]>=(mouse2_s_x-2)) && (hcount[10:1]<mouse2_e_x) && (vcount>=mouse2_s_y) && (vcount<mouse2_e_y);
	assign in_box_mouse3 = (hcount[10:1]>=(mouse3_s_x-2)) && (hcount[10:1]<mouse3_e_x) && (vcount>=mouse3_s_y) && (vcount<mouse3_e_y);


	//assign others_explosion_addr = (in_box && (game_status != 2'b00)) ? explosion_addr : others_addr;

	assign bone_s_x = dog_project_x;
	assign bone_s_y = dog_project_y;
	assign bone_e_x = bone_s_x + 16'd8;
	assign bone_e_y = bone_s_y + 16'd24;

	assign mouse1_s_x = cat_project_x[1];
	assign mouse2_s_x = cat_project_x[2];
	assign mouse3_s_x = cat_project_x[3];
	assign mouse1_s_y = cat_project_y[1];
	assign mouse2_s_y = cat_project_y[2];
	assign mouse3_s_y = cat_project_y[3];

	assign mouse1_e_x = mouse1_s_x + 4'd8;
	assign mouse2_e_x = mouse2_s_x + 4'd8;
	assign mouse3_e_x = mouse3_s_x + 4'd8;
	assign mouse1_e_y = mouse1_s_y + 5'd16;
	assign mouse2_e_y = mouse2_s_y + 5'd16;
	assign mouse3_e_y = mouse3_s_y + 5'd16;

	assign cat_mat_s_x = cat_array_pos_x;
	assign cat_mat_s_y = cat_array_pos_y;
	assign cat_mat_e_x = cat_mat_s_x + CAT_MATRIX_SIZE_X;
	assign cat_mat_e_y = cat_mat_s_y + CAT_MATRIX_SIZE_Y;


/*************************************************************/


logic [15:0] level_addr, level_num_addr, score_addr, score_n0_addr, score_n1_addr, score_n2_addr, life1_addr, life2_addr, life3_addr, start_addr;

parameter level_s_x = 10; parameter level_e_x = level_s_x + 96;
parameter level_num_s_x = 110; parameter level_num_e_x = level_num_s_x + 32; 
parameter score_s_x = 400; parameter score_e_x = score_s_x + 96;
parameter score_n0_s_x = 500; parameter score_n0_e_x = score_n0_s_x + 32;
parameter score_n1_s_x = 540; parameter score_n1_e_x = score_n1_s_x + 32;
parameter score_n2_s_x = 580; parameter score_n2_e_x = score_n2_s_x + 32; 
parameter life1_s_x = 200; parameter life1_e_x = life1_s_x + 32;
parameter life2_s_x = 240; parameter life2_e_x = life2_s_x + 32;
parameter life3_s_x = 280; parameter life3_e_x = life3_s_x + 32;
parameter start_s_x = 248; parameter start_e_x = 392 ;
parameter start_s_y = 210; parameter start_e_y = 258 ;
parameter dog_pos_s_y = 430; parameter dog_pos_e_y = 462 ;
parameter mystery_pos_s_y = 50; parameter mystery_pos_e_y = 98 ;
parameter game_over_s_x = 222; parameter game_over_e_x = 418 ;
parameter game_over_s_y = 210; parameter game_over_e_y = 238;



/*******************************我的*****************************/
always_ff @(posedge clk)begin
   if (reset) cat_mat_addr <= 0;
   else begin
      if(cat_offset_x == 31 && in_box && ~hcount[0]) begin
         if(cat_cnt_x == 10) begin
            if(cat_offset_y == 31) cat_mat_addr <= (cat_cnt_y == 4)? 0 : (cat_mat_addr + 1);
            else cat_mat_addr <= cat_mat_addr - 10;
         end
         else cat_mat_addr <= cat_mat_addr + 1;
      end
      else cat_mat_addr <= cat_mat_addr;
   end
end


always_comb begin
   if ((hcount[10:1]>=cat_mat_s_x) && (hcount[10:1]<cat_mat_e_x) && (vcount>=cat_mat_s_y) && (vcount<cat_mat_e_y)) begin
      case(cat_mat_data)
      0: output_data = 8'd0;
      1: output_data = display_data;
      2: output_data = cat2_data;
      3: output_data = cat3_data;
      4: output_data = cat4_data;
      5: output_data = explode_data;
      default: output_data = 8'hff;
      endcase
   end
   else output_data = 8'd0;
end

/*****************************我的**************************************/

logic [15:0] cat_mat_count;
logic [7:0] i_barrier_cnt;
logic [4:0] o_barrier_cnt;
parameter delta_x = 70;
parameter delta_y = 10;

always_ff @(posedge clk)begin
   if (reset) begin
	 cat_mat_count <= 0;
         test <= 0;
         barrier_addr <= 0;
	 i_barrier_cnt <= 0;
	 o_barrier_cnt <= 0;
   end
   else 
      if(~hcount[0]) begin
	  

	      if (hcount[10:1]>=dog_pos_x && hcount[10:1]<(dog_pos_x +32) && vcount>=dog_pos_s_y && vcount<(dog_pos_e_y)) begin
		 test <=1;
		 if (hcount[10:1] == (dog_pos_x+31) && vcount == (dog_pos_e_y-1)) //poodle_addr <= 16'h0000;
		 begin
		    case(dog_ani_state) 
		        4'd0:  poodle_addr <= 16'h0000 ;
		        4'd1:  poodle_addr <= 16'h0400 ;        
		        4'd2:  poodle_addr <= 16'h0800 ;
		        4'd3:  poodle_addr <= 16'h0C00 ;
		        4'd4:  poodle_addr <= 16'h1000 ;
		        4'd5:  poodle_addr <= 16'h1400 ;
		        4'd6:  poodle_addr <= 16'h1800 ;
		        4'd7:  poodle_addr <= 16'h1C00 ;
		        4'd8:  poodle_addr <= 16'h2000 ;
		        4'd9:  poodle_addr <= 16'h2400 ;
		        default:  poodle_addr <= 16'h0000 ;
            	    endcase
		 end
		 else poodle_addr <= poodle_addr + 1;
	      end

	      if (hcount[10:1]>= mystery_pos_x && hcount[10:1]<(mystery_pos_x+48) && vcount>=mystery_pos_s_y && vcount<(mystery_pos_e_y)) begin
		 test <=1;
		 if (hcount[10:1] == (mystery_pos_x+47) && vcount == (mystery_pos_e_y-1)) mystery_addr <= 0;
		 else mystery_addr <= mystery_addr + 1;
	      end

	      if (hcount[10:1]>=game_over_s_x && hcount[10:1]<(game_over_e_x) && vcount>=game_over_s_y && vcount<(game_over_e_y) && ~hcount[0]) begin
		 test <=1;
		 if (hcount[10:1] == (game_over_e_x-1) && vcount == (game_over_e_y-1)) game_over_addr <= 0;
		 else game_over_addr <= game_over_addr + 1;
	      end

	   /*    if (hcount[10:1]>=132 && hcount[10:1]<(164) && vcount>=300 && vcount<(332) && ~hcount[0]) begin
		 test <=1;
		 if (hcount[10:1] == (163) && vcount == (331)) num_addr <= 0;
		 else num_addr <= num_addr + 1;
	      end
 */
		  if (hcount[10:1]>=bone_s_x && hcount[10:1]<bone_e_x && vcount>=bone_s_y && vcount<bone_e_y && ~hcount[0]) begin
		 test <=1;
		 if (hcount[10:1] == bone_e_x-1 && vcount == bone_e_y-1) bone_addr <= 0;
		 else bone_addr <= bone_addr + 1;
	      end


	      if ((hcount[10:1]>=54+delta_x && hcount[10:1]<102+delta_x && vcount>=336+delta_y && vcount<368+delta_y) || (hcount[10:1]>=154+delta_x && hcount[10:1]<202+delta_x && vcount>=336+delta_y && vcount<368+delta_y) || (hcount[10:1]>=254+delta_x && hcount[10:1]<302+delta_x && vcount>=336+delta_y && vcount<368+delta_y) || (hcount[10:1]>=354+delta_x && hcount[10:1]<402+delta_x && vcount>=336+delta_y && vcount<368+delta_y)) begin

	

		     if(i_barrier_cnt == 191 && vcount == 351+delta_y) begin
			o_barrier_cnt <= 0;
			i_barrier_cnt <= 0;
			barrier_mat_addr <= 12;
		     end else if(i_barrier_cnt == 191 && vcount != 351+delta_y) begin
			o_barrier_cnt <= o_barrier_cnt + 1;
			i_barrier_cnt <= 0;
			barrier_mat_addr <= (vcount < 352+delta_y) ? 0 : 12;
		     end else if(i_barrier_cnt[3:0] == 15) begin 
			barrier_mat_addr <= barrier_mat_addr + 1;
			i_barrier_cnt <= i_barrier_cnt + 1;
		     end else begin
			i_barrier_cnt <= i_barrier_cnt + 1;
		     end



              end
 
	            case(barrier_mat_data)
			 1: barrier_addr = {8'h0, o_barrier_cnt[3:0], i_barrier_cnt[3:0]};
			 2: barrier_addr = {8'h1, o_barrier_cnt[3:0], i_barrier_cnt[3:0]};
			 3: barrier_addr = {8'h2, o_barrier_cnt[3:0], i_barrier_cnt[3:0]};
			 4: barrier_addr = {8'h3, o_barrier_cnt[3:0], i_barrier_cnt[3:0]};
			 default: barrier_addr = 16'h3F0;
		    endcase


	     if(hcount[10:1] == 52 && vcount == 336) begin
		barrier_mat_addr <= 0;
		o_barrier_cnt <= 0;
		i_barrier_cnt <= 0;
	     end
 /* level*/
	    if (hcount[10:1]>=level_s_x && hcount[10:1]<(level_e_x) && vcount>=10 && vcount<(42) && ~hcount[0]) begin
		 	test <=1;
		    if (hcount[10:1] == (level_s_x) && vcount == (10)) level_addr <= 16'h0400;
		 	else begin 
				level_addr <= level_addr + 1;
				others_addr <= level_addr;
			end
	    end 
		/* level number */
		if (hcount[10:1]>=level_num_s_x && hcount[10:1]<(level_num_e_x) && vcount>=10 && vcount<(42) && ~hcount[0]) begin
		 	test <=1;
		    if (hcount[10:1] == (110) && vcount == (10)) begin
			case(level)
				0: level_num_addr <= 16'h0000;
				1: level_num_addr <= 16'h0400;
				2: level_num_addr <= 16'h0800;
				3: level_num_addr <= 16'h0C00;
				4: level_num_addr <= 16'h1000;
				5: level_num_addr <= 16'h1400;
				6: level_num_addr <= 16'h1800;
				7: level_num_addr <= 16'h1C00;
				8: level_num_addr <= 16'h2000;
				9: level_num_addr <= 16'h2400;
				default: level_num_addr <= 16'h0000;
			endcase
			end
			else begin 
				level_num_addr <= level_num_addr + 1; 
				num_addr <= level_num_addr;
			end
		end

/* mice */
		if (hcount[10:1]>= mouse1_s_x-2 && hcount[10:1]<mouse1_e_x && vcount>=mouse1_s_y && vcount<mouse1_e_y && ~hcount[0] ) begin
			test <=1;
				others_addr <= mouse1_addr + 16'h1400;
		end 
		else if (hcount[10:1]>= mouse2_s_x-2 && hcount[10:1]<mouse2_e_x && vcount>=mouse2_s_y && vcount<mouse2_e_y && ~hcount[0] ) begin
			test <=1;
				others_addr <= mouse2_addr + 16'h1400;
		end 
		else if (hcount[10:1]>= mouse3_s_x-2 && hcount[10:1]<mouse3_e_x && vcount>=mouse3_s_y && vcount<mouse3_e_y && ~hcount[0] ) begin
			test <=1;
				others_addr <= mouse3_addr + 16'h1400;
		end


/* lives */
		if (hcount[10:1]>= life1_s_x && hcount[10:1]<(life1_e_x) && vcount>=10 && vcount<(42) && ~hcount[0] && life>0 && life <4) begin
			test <=1;
		    if (hcount[10:1] == (life1_s_x) && vcount == (10)) life1_addr <= 16'h1000;
		 	else begin
				life1_addr <= life1_addr + 1; 
				others_addr <= life1_addr;
			end 
		end

		if (hcount[10:1]>= life2_s_x && hcount[10:1]<(life2_e_x) && vcount>=10 && vcount<(42) && ~hcount[0] && life>1 && life <4) begin
			test <=1;
		    if (hcount[10:1] == (life2_s_x) && vcount == (10)) life2_addr <= 16'h1000;
		 	else begin
				life2_addr <= life2_addr + 1; 
				others_addr <= life2_addr;
			end 
			end

		if (hcount[10:1]>= life3_s_x && hcount[10:1]<(life3_e_x) && vcount>=10 && vcount<(42) && ~hcount[0] && life == 3) begin
			test <=1;
		    if (hcount[10:1] == (life3_s_x) && vcount == (10)) life3_addr <= 16'h1000;
		 	else begin
				life3_addr <= life3_addr + 1; 
				others_addr <= life3_addr;
			end 
			end

/* score */
		if (hcount[10:1]>= score_s_x && hcount[10:1]<(score_e_x) && vcount>=10 && vcount<(42) && ~hcount[0]) begin
		 	test <=1;
		    if (hcount[10:1] == (score_s_x) && vcount == (10)) score_addr <= 16'h1480;
		 	else begin
				score_addr <= score_addr + 1; 
				others_addr <= score_addr;
			end 
			end

		/* score num 0*/
		if (hcount[10:1]>=score_n0_s_x && hcount[10:1]<(score_n0_e_x) && vcount>=10 && vcount<(42) && ~hcount[0]) begin
		 	test <=1;
		    if (hcount[10:1] == (score_n0_s_x) && vcount == (10)) begin
			case(score[3:0])
				0: score_n0_addr <= 16'h0000;
				1: score_n0_addr <= 16'h0400;
				2: score_n0_addr <= 16'h0800;
				3: score_n0_addr <= 16'h0C00;
				4: score_n0_addr <= 16'h1000;
				5: score_n0_addr <= 16'h1400;
				6: score_n0_addr <= 16'h1800;
				7: score_n0_addr <= 16'h1C00;
				8: score_n0_addr <= 16'h2000;
				9: score_n0_addr <= 16'h2400;
				default: score_n0_addr <= 16'h0000;
			endcase
			end
		 	else begin 
				score_n0_addr <= score_n0_addr + 1;
				num_addr <= score_n0_addr;
			end
		end
		/* score num 1*/
		if (hcount[10:1]>=score_n1_s_x && hcount[10:1]<(score_n1_e_x) && vcount>=10 && vcount<(42) && ~hcount[0]) begin
		 	test <=1;
		    if (hcount[10:1] == (score_n1_s_x) && vcount == (10)) begin
			case(score[7:4])
				0: score_n1_addr <= 16'h0000;
				1: score_n1_addr <= 16'h0400;
				2: score_n1_addr <= 16'h0800;
				3: score_n1_addr <= 16'h0C00;
				4: score_n1_addr <= 16'h1000;
				5: score_n1_addr <= 16'h1400;
				6: score_n1_addr <= 16'h1800;
				7: score_n1_addr <= 16'h1C00;
				8: score_n1_addr <= 16'h2000;
				9: score_n1_addr <= 16'h2400;
				default: score_n1_addr <= 16'h0000;
			endcase
			end
		 	else begin 
				score_n1_addr <= score_n1_addr + 1;
				num_addr <= score_n1_addr;
		end
		end

		/* score num 2*/
		if (hcount[10:1]>=score_n2_s_x && hcount[10:1]<(score_n2_e_x) && vcount>=10 && vcount<(42) && ~hcount[0]) begin
		 	test <=1;
		    if (hcount[10:1] == (score_n2_s_x) && vcount == (10)) begin
			case(score[11:8])
				0: score_n2_addr <= 16'h0000;
				1: score_n2_addr <= 16'h0400;
				2: score_n2_addr <= 16'h0800;
				3: score_n2_addr <= 16'h0C00;
				4: score_n2_addr <= 16'h1000;
				5: score_n2_addr <= 16'h1400;
				6: score_n2_addr <= 16'h1800;
				7: score_n2_addr <= 16'h1C00;
				8: score_n2_addr <= 16'h2000;
				9: score_n2_addr <= 16'h2400;
				default: score_n2_addr <= 16'h0000;
			endcase
			end
		 	else begin 
				score_n2_addr <= score_n2_addr + 1;
				num_addr <= score_n2_addr;
		end
		end

/* Start */
	if (hcount[10:1]>= start_s_x && hcount[10:1]< start_e_x && vcount>= start_s_y && vcount< start_e_y && ~hcount[0] && (game_status == 2'b00)) begin
		test <=1;
		if (hcount[10:1] == (start_s_x) && vcount == (start_s_y)) start_addr <= 16'h2080;
		else begin
			start_addr <= start_addr + 1; 
			others_addr <= start_addr;
		end 
		end

	end
end



/****************************************************************** 
     VGA Display combo logic
*******************************************************************/

   always_comb begin
      {VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0};
      if (VGA_BLANK_n )
         
	/* Start */
	 if (hcount[10:1]>=start_s_x && hcount[10:1]< start_e_x && vcount>= start_s_y && vcount< start_e_y && others_data != 8'd0 && game_status == 2'b00) 
            	{VGA_R, VGA_G, VGA_B} = {{others_data[7:5], 5'd0}, {others_data[4:2], 5'd0}, {others_data[1:0], 6'd0}} ;
	
	/* Game Over */
         else if (hcount[10:1]>=game_over_s_x && hcount[10:1]<(game_over_e_x) && vcount>=game_over_s_y && vcount<(game_over_e_y) && game_over_data != 8'd0 && game_status == 2'b10) 
            {VGA_R, VGA_G, VGA_B} = {{game_over_data[7:5], 5'd0}, {game_over_data[4:2], 5'd0}, {game_over_data[1:0], 6'd0}}; 

	/* Bone */
	 else if (hcount[10:1]>=bone_s_x && hcount[10:1]<bone_e_x && vcount>=bone_s_y && vcount<bone_e_y && bone_data != 8'd0 && dog_bullet) 
            {VGA_R, VGA_G, VGA_B} = {{bone_data[7:5], 5'd0}, {bone_data[4:2], 5'd0}, {bone_data[1:0], 6'd0}};


	/* Mouse 1 */
	else if (((hcount[10:1]>=(mouse1_s_x)) && (hcount[10:1]<mouse1_e_x) && (vcount>=mouse1_s_y) && (vcount<mouse1_e_y) && others_data != 8'd0) && cat_bullet[1] ) 
    	{VGA_R, VGA_G, VGA_B} = {{others_data[7:5], 5'd0}, {others_data[4:2], 5'd0}, {others_data[1:0], 6'd0}}; 

	/* Mouse 2 */
	else if (((hcount[10:1]>=(mouse2_s_x)) && (hcount[10:1]<mouse2_e_x) && (vcount>=mouse2_s_y) && (vcount<mouse2_e_y) && others_data != 8'd0) && cat_bullet[2] ) 
			{VGA_R, VGA_G, VGA_B} = {{others_data[7:5], 5'd0}, {others_data[4:2], 5'd0}, {others_data[1:0], 6'd0}}; 		

	/* Mouse 3 */
	else if (((hcount[10:1]>=(mouse3_s_x)) && (hcount[10:1]<mouse3_e_x) && (vcount>=mouse3_s_y) && (vcount<mouse3_e_y) && others_data != 8'd0) && cat_bullet[3] ) 
    	{VGA_R, VGA_G, VGA_B} = {{others_data[7:5], 5'd0}, {others_data[4:2], 5'd0}, {others_data[1:0], 6'd0}}; 

	/* Cat Matrix */
	 else if ((hcount[10:1]>=cat_mat_s_x) && (hcount[10:1]<cat_mat_e_x) && (vcount>=cat_mat_s_y) && (vcount<cat_mat_e_y) && output_data != 8'd0 ) 
		    {VGA_R, VGA_G, VGA_B} = {{output_data[7:5], 5'd0}, {output_data[4:2], 5'd0}, {output_data[1:0], 6'd0}};

	/* Mystery */
         else if (hcount[10:1]>= mystery_pos_x && hcount[10:1]<(mystery_pos_x+48) && vcount>=mystery_pos_s_y && vcount<(mystery_pos_e_y) && mystery_data != 8'd0 && mystery_vis) 
            {VGA_R, VGA_G, VGA_B} = {{mystery_data[7:5], 5'd0}, {mystery_data[4:2], 5'd0}, {mystery_data[1:0], 6'd0}};  

	/* Poodle */
         else if (hcount[10:1]>=dog_pos_x && hcount[10:1]<(dog_pos_x + 32) && vcount>=dog_pos_s_y && vcount<(dog_pos_e_y) && poodle_data != 8'd0) 
            {VGA_R, VGA_G, VGA_B} = {{poodle_data[7:5], 5'd0}, {poodle_data[4:2], 5'd0}, {poodle_data[1:0], 6'd0}}; 

	/* Explosion */
        // else if (hcount[10:1]>=532 && hcount[10:1]<(564) && vcount>=400 && vcount<(432) && others_data != 8'd0) 
        //    {VGA_R, VGA_G, VGA_B} = {{others_data[7:5], 5'd0}, {others_data[4:2], 5'd0}, {others_data[1:0], 6'd0}}; 


	 /* Barriers */
         else if (((hcount[10:1]>=54+delta_x && hcount[10:1]<102+delta_x && vcount>=336+delta_y && vcount<368+delta_y) || (hcount[10:1]>=154+delta_x && hcount[10:1]<202+delta_x && vcount>=336+delta_y && vcount<368+delta_y) || (hcount[10:1]>=254+delta_x && hcount[10:1]<302+delta_x && vcount>=336+delta_y && vcount<368+delta_y) || (hcount[10:1]>=354+delta_x && hcount[10:1]<402+delta_x && vcount>=336+delta_y && vcount<368+delta_y)) && barrier_data != 8'd0) 
            {VGA_R, VGA_G, VGA_B} = {{barrier_data[7:5], 5'd0}, {barrier_data[4:2], 5'd0}, {barrier_data[1:0], 6'd0}}; 
	
	/* level*/
         else if ((hcount[10:1]>=level_s_x && hcount[10:1]<(level_e_x) && vcount>=10 && vcount<(42) && others_data != 8'd0) ) 
            {VGA_R, VGA_G, VGA_B} = {{others_data[7:5], 5'd0}, {others_data[4:2], 5'd0}, {others_data[1:0], 6'd0}}; 

	/* level number*/
         else if (hcount[10:1]>=level_num_s_x && hcount[10:1]<(level_num_e_x) && vcount>=10 && vcount<(42) && num_data != 8'd0) 
            {VGA_R, VGA_G, VGA_B} = {{num_data[7:5], 5'd0}, {num_data[4:2], 5'd0}, {num_data[1:0], 6'd0}};
	
	/* score */
	else if (hcount[10:1]>=score_s_x && hcount[10:1]<(score_e_x) && vcount>=10 && vcount<(42) && others_data != 8'd0) 
	    {VGA_R, VGA_G, VGA_B} = {{others_data[7:5], 5'd0}, {others_data[4:2], 5'd0}, {others_data[1:0], 6'd0}};

	/* score numbers */
	else if (((hcount[10:1]>=score_n0_s_x && hcount[10:1]<score_n0_e_x) || (hcount[10:1]>=score_n1_s_x && hcount[10:1]<score_n1_e_x) ||
				(hcount[10:1]>=score_n2_s_x && hcount[10:1]<score_n2_e_x)) && vcount>=10 && vcount<(42)   && num_data != 8'd0) 
	    {VGA_R, VGA_G, VGA_B} = {{num_data[7:5], 5'd0}, {num_data[4:2], 5'd0}, {num_data[1:0], 6'd0}};
	
	/* lives */
	else if (((hcount[10:1]>=life1_s_x && hcount[10:1]<(life1_e_x) && life>0 && life <4) || (hcount[10:1]>=life2_s_x && hcount[10:1]<(life2_e_x) && life>1 && life <4)||
 				(hcount[10:1]>=life3_s_x && hcount[10:1]<(life3_e_x) && life == 3)) && vcount>=10 && vcount<(42) && others_data != 8'd0) 
	    {VGA_R, VGA_G, VGA_B} = {{others_data[7:5], 5'd0}, {others_data[4:2], 5'd0}, {others_data[1:0], 6'd0}};

	


         else
         {VGA_R, VGA_G, VGA_B} = {background_r, background_g, background_b};
   end


/****************************************************************** 
     Avalon Bus start
*******************************************************************/
   assign eof_irq =irq;
   always_ff @(posedge clk) begin

    if((hcount[10:0] == 0) && (vcount[9:0] == 463)) begin
	irq <= 1'b1;
	frame_cnt <= frame_cnt + 1;
    end else if ((hcount[10:0] >= 0) && (hcount[10:0] <= 1280) && (vcount[9:0] >= 0) && (vcount[9:0] < 463)) begin
       frame_active <= 1;
    end else begin
       frame_active <= 0;
    end

     if (reset) begin
	background_r <= 8'h0;
	background_g <= 8'h0;
	background_b <= 8'h0;

	irq <= 1'b0;

        play <= 0;
        play_cat <=0;
        play_dog <= 0;
        delay <= 3;
/*
	bone_s_x <= 16'd320;
	bone_s_y <= 16'd400;
	bone_e_x <= bone_s_x + 16'd8;
	bone_e_y <= bone_s_y + 16'd24;

	mouse1_s_x <= 16'd160;
	mouse1_s_y <= 16'd120;

	mouse1_e_x <= mouse1_s_x + 16'd8;
	mouse1_e_y <= mouse1_s_y + 16'd17;

	mouse2_s_x <= 16'd320;
	mouse2_s_y <= 16'd120;

	mouse2_e_x <= mouse2_s_x + 16'd8;
	mouse2_e_y <= mouse2_s_y + 16'd17;

	mouse3_s_x <= 16'd480;
	mouse3_s_y <= 16'd120;

	mouse3_e_x <= mouse3_s_x + 16'd8;
	mouse3_e_y <= mouse3_s_y + 16'd17;

	cat_mat_s_x <= 16'd40;
	cat_mat_s_y <= 16'd120;

	cat_mat_e_x <= 16'd40+CAT_MATRIX_SIZE_X;
	cat_mat_e_y <= 16'd120+CAT_MATRIX_SIZE_Y;

	cat_mat_e_minus_1_x <= 16'd40+CAT_MATRIX_SIZE_X-1;
    cat_mat_e_minus_1_y <= 16'd120+CAT_MATRIX_SIZE_Y-1;
*/

     end else if (chipselect && write) begin
       case (address)
	// 7'h0 : background_r <= writedata;
	// 7'h1 : background_g <= writedata;
	// 7'h2 : background_b <= writedata;
	/* 7'h4 : begin
		    cat_mat_s_x <= writedata;
		    cat_mat_e_x <= writedata + CAT_MATRIX_SIZE_X;
		    cat_mat_e_minus_1_x <= writedata + CAT_MATRIX_SIZE_X - 1;
		end
         7'h6 : begin
		    cat_mat_s_y <= writedata;
		    cat_mat_e_y <= writedata + CAT_MATRIX_SIZE_Y;
		    cat_mat_e_minus_1_y <= writedata + CAT_MATRIX_SIZE_Y - 1;
		end*/

	 7'h29: begin
		 c_write <= 1;
                 c_cs <= 1;
		 c_write_addr <= writedata[15:4];
		 c_writedata <= writedata[3:0];
                 end

         7'h2A : begin
                   play <= writedata[5]; 
                   vol[3:0] <= writedata[3:0];
                 end
         7'h2B : begin
                   play_cat <= writedata[5]; 
                   vol_cat[3:0] <= writedata[3:0];
                 end
         7'h2C : begin 
                   play_dog <= writedata[5]; 
                   vol_dog[3:0] <= writedata[3:0];
                 end
         7'h2D : delay <= writedata;


	 7'h2E: begin
		 b_write <= 1;
                 b_cs <= 1;
		 b_write_addr <= writedata[15:4];
		 b_writedata <= writedata[3:0];
                 end
         7'h2F : irq <= 0;
       endcase
     end else if(chipselect && !write && address == 7'h30) begin
	readdata <= 16'hD065;
     end else if(chipselect && !write && address == 7'h31) begin
	readdata <= {frame_active, frame_cnt[14:0]};
     end else begin
         b_write <= 0;
         b_cs <= 0;
	 c_write <= 0;
         c_cs <= 0;
     end


   end

endmodule



/****************************************************************** 
     VGA counter module
*******************************************************************/

module vga_counters(
 input logic 	     clk50, reset,
 output logic [10:0] hcount,  // hcount[10:1] is pixel column
 output logic [9:0]  vcount,  // vcount[9:0] is pixel row
 output logic VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC +
                            HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC +
                            VBACK_PORCH; // 525

   logic endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;
       
   logic endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) &
		      !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1'b0; // For putting sync on the green signal; unused
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: rising edge sensitive
   
endmodule



  
