import csv
import math
import time

MAX_DAYS = 1000
DAYS_30 = 30
SHORT_TERM = 10
LONG_TERM = 30

class StockRecord:
    def __init__(self, date, open_price, high, low, close, volume, dividends, stock_splits):
        self.date = date
        self.open = open_price  # Renamed parameter to avoid conflict with built-in function
        self.high = high
        self.low = low
        self.close = close
        self.volume = volume
        self.dividends = dividends
        self.stock_splits = stock_splits
        self.price_30_days_ago = 0.0
        self.price_momentum = 0.0
        self.average_volume_short_term = 0.0
        self.average_volume_long_term = 0.0
        self.volume_factor = 0.0
        self.volume_ratio = 0.0
        self.daily_returns = 0.0
        self.volatility_30d = 0.0
        self.rsi = 0.0

def moving_average(values, start, end):
    total = sum(values[start:end + 1])
    count = end - start + 1
    return total / count if count > 0 else 0.0

def read_csv(filename):
    records = []
    with open(filename, 'r') as file:
        reader = csv.reader(file)
        next(reader)  # Skip header
        for row in reader:
            if len(records) >= MAX_DAYS:
                break
            date, open_price, high, low, close, volume, dividends, stock_splits = row
            record = StockRecord(date, float(open_price), float(high), float(low), float(close), int(volume), float(dividends), float(stock_splits))
            records.append(record)
    return records

def calculate_factors(records):
    n = len(records)
    gains = [0] * n
    losses = [0] * n
    volumes = [record.volume for record in records]
    closes = [record.close for record in records]
    sum_returns = 0.0

    for i in range(1, n):
        records[i].daily_returns = (records[i].close - records[i - 1].close) / records[i - 1].close
        sum_returns += abs(records[i].daily_returns)
        gains[i] = max(0, records[i].close - records[i - 1].close)
        losses[i] = max(0, records[i - 1].close - records[i].close)

        if i >= DAYS_30:
            records[i].price_30_days_ago = records[i - DAYS_30].close
            records[i].price_momentum = (records[i].close - records[i].price_30_days_ago) / records[i].price_30_days_ago
            records[i].average_volume_long_term = moving_average(volumes, i - LONG_TERM + 1, i)
            records[i].volatility_30d = math.sqrt(sum_returns / DAYS_30)
            sum_returns -= abs(records[i - DAYS_30 + 1].daily_returns)

        if i >= SHORT_TERM:
            records[i].average_volume_short_term = moving_average(volumes, i - SHORT_TERM + 1, i)

        if records[i].average_volume_long_term > 0:
            records[i].volume_factor = records[i].average_volume_short_term / records[i].average_volume_long_term
            records[i].volume_ratio = records[i].volume / records[i].average_volume_long_term

    for i in range(DAYS_30, n):
        avg_gain = moving_average(gains, i - DAYS_30 + 1, i)
        avg_loss = moving_average(losses, i - DAYS_30 + 1, i)
        rs = avg_loss / avg_gain if avg_gain != 0 else 0
        records[i].rsi = 100.0 - (100.0 / (1.0 + rs)) if rs != 0 else 100.0

def write_csv(filename, records):
    with open(filename, 'w', newline='') as file:
        writer = csv.writer(file)
        writer.writerow(["Date", "Close", "Volume", "Price Momentum", "Volume Factor", "Volume Ratio", "Volatility 30D", "RSI"])
        for record in records:
            writer.writerow([record.date, record.close, record.volume, record.price_momentum, record.volume_factor, record.volume_ratio, record.volatility_30d, record.rsi])

def main():
    

    records = read_csv("/Users/luweitao/Downloads/AFRM.csv")
    start_time = time.time()
    calculate_factors(records)
    

    end_time = time.time()
    print(f"Time taken by main: {end_time - start_time} seconds")
    write_csv("output_222.csv", records)
if __name__ == "__main__":
    main()
