/* * Device driver for the VGA video generator
 *
 * A Platform device implemented using the misc subsystem
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod vga_ball.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree vga_ball.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "vga_ball.h"

#define DRIVER_NAME "vga_ball"

/* Device registers */
#define CHAR_CODE(x) ((x))
#define ACT_VALUE1(x) ((x)+2)
#define ACT_VALUE2(x) ((x)+4)
#define IC1(x) ((x)+6)
#define IC2(x) ((x)+8)
#define IC3(x) ((x)+10)
#define SIGN(x) ((x)+12)

#define OPEN(x) ((x)+14)
#define HIGH(x) ((x)+16)
#define LOW(x) ((x)+18)
#define CLOSE(x) ((x)+20)
#define VOLUME(x) ((x)+22)

#define MOMENTUM(x) ((x)+24)
#define VOLUMEOUT(x) ((x)+26)
#define RSI(x) ((x)+28)
#define MA(x) ((x)+30)
#define DONE(x) ((x)+32)

#define FAC1(x) ((x)+34)
#define FAC2(x) ((x)+36)
#define CURSOR(x) ((x)+38)
/*
 * Information about our device
 */
struct vga_ball_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
	mem memory;
} dev;
/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
 */

static void read_output_factors(mem *memory) {
	memory->momentum = ioread16(MOMENTUM(dev.virtbase));
	memory->volumeOut = ioread16(VOLUMEOUT(dev.virtbase));
	memory->rsi = ioread16(RSI(dev.virtbase));
	memory->ma = ioread16(MA(dev.virtbase));
}


static void write_character(mem *memory) {
	iowrite16(memory->chars, CHAR_CODE(dev.virtbase));
	iowrite16(memory->cursor, CURSOR(dev.virtbase));
	dev.memory = *memory;
}
static void write_values(mem *memory) {
	iowrite16(memory->actual1, ACT_VALUE1(dev.virtbase));
	iowrite16(memory->actual2, ACT_VALUE2(dev.virtbase));
	iowrite16(memory->fac1,FAC1(dev.virtbase));
	iowrite16(memory->fac2, FAC2(dev.virtbase));
	iowrite16(memory->sign, SIGN(dev.virtbase));
	dev.memory = *memory;
}
static void write_ic(mem *memory) {
	iowrite16(memory->ic1, IC1(dev.virtbase));
	iowrite16(memory->ic2, IC2(dev.virtbase));
	iowrite16(memory->ic3, IC3(dev.virtbase));

	dev.memory = *memory;
}
static void write_factors(mem *memory) {
	iowrite16(memory->open, OPEN(dev.virtbase));
	iowrite16(memory->high, HIGH(dev.virtbase));
	iowrite16(memory->low, LOW(dev.virtbase));
	iowrite16(memory->close, CLOSE(dev.virtbase));
	iowrite16(memory->volume, VOLUME(dev.virtbase));
	dev.memory = *memory; 
}

static void read_finish(mem *memory){
	memory->done = ioread16(DONE(dev.virtbase));
}

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long vga_ball_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	calculator vla;
	switch (cmd) {
	case VGA_BALL_WRITE_CHARACTER:
		if (copy_from_user(&vla, (calculator *) arg,
						sizeof(calculator))) {
			return -EACCES;
		}
		write_character(&vla.memory);
		break;

	case VGA_BALL_WRITE_VALUES:
		if (copy_from_user(&vla, (calculator *) arg,
						sizeof(calculator))) {
			return -EACCES;
		}
		write_values(&vla.memory);
		break;

	case VGA_BALL_WRITE_IC:
		if (copy_from_user(&vla, (calculator *) arg,
						sizeof(calculator))) {
			return -EACCES;
		}
		write_ic(&vla.memory);
		break;
	case CALCULATOR_WRITE_VALUES:
		if (copy_from_user(&vla, (calculator *) arg,
						sizeof(calculator))) {
			return -EACCES;
		}
		write_factors(&vla.memory);
		break;
	case CALCULATOR_READ_VALUES:
		read_output_factors(&vla.memory);
		if (copy_to_user((calculator *) arg, &vla,
				 sizeof(calculator)))
			return -EACCES;
		
		break;
	case CALCULATOR_READ_DONE:
		read_finish(&vla.memory);
		if (copy_to_user((calculator *) arg, &vla,
				 sizeof(calculator)))
			return -EACCES;
		
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations vga_ball_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = vga_ball_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice vga_ball_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &vga_ball_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init vga_ball_probe(struct platform_device *pdev)
{
        // vga_ball_color_t beige = { 0x00, 0x00, 0x00 };
	int ret;

	/* Register ourselves as a misc device: creates /dev/vga_ball */
	ret = misc_register(&vga_ball_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
        
	/* Set an initial color */
        // write_background(&beige);

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&vga_ball_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int vga_ball_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&vga_ball_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id vga_ball_of_match[] = {
	{ .compatible = "csee4840,vga_ball-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, vga_ball_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver vga_ball_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(vga_ball_of_match),
	},
	.remove	= __exit_p(vga_ball_remove),
};

/* Called when the module is loaded: set things up */
static int __init vga_ball_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&vga_ball_driver, vga_ball_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit vga_ball_exit(void)
{
	platform_driver_unregister(&vga_ball_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(vga_ball_init);
module_exit(vga_ball_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("VGA ball driver");
