/*
 * Userspace program that communicates with the vga_ball device driver
 * through ioctls
 *
 * Stephen A. Edwards
 * Columbia University
 */

#include <stdio.h>
#include "vga_ball.h"
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>

#include <libusb-1.0/libusb.h>
#include "usbkeyboard.h"

#include "fm_model.h"
#include "matrix.h"
#include <math.h>

#define MIN_RANDOM 0
#define MAX_RANDOM 340
#define COLUMNS 5
#define FRACTIONAL_BITS 8
float fixed_to_float(int16_t fixed_number) {
    return fixed_number / (float)(1 << FRACTIONAL_BITS);
}

int vga_ball_fd;

void set_ball_character(const mem *c) {
  calculator vla;
  vla.memory = *c;
  if(ioctl(vga_ball_fd, VGA_BALL_WRITE_CHARACTER, &vla)) {
    perror("ioctl(VGA_BALL_SET_character) failed");
    return;
  }
}

void set_ball_values(const mem *v) {
  calculator vla;
  vla.memory = *v;
  //printf("VLA read: %d , %d, %d, %d, %d\n", vla.memory.ic1, vla.memory.ic2, vla.memory.ic3, vla.memory.actual1, vla.memory.actual2);
  if(ioctl(vga_ball_fd, VGA_BALL_WRITE_VALUES, &vla)) {
    perror("ioctl(VGA_BALL_SET_values) failed");
    return;
  }
}

void set_IC(const mem *v) {
  calculator vla;
  vla.memory = *v;
  //printf("VLA read: %d , %d, %d, %d, %d\n", vla.memory.ic1, vla.memory.ic2, vla.memory.ic3, vla.memory.actual1, vla.memory.actual2);
  if(ioctl(vga_ball_fd, VGA_BALL_WRITE_IC, &vla)) {
    perror("ioctl(VGA_BALL_SET_IC) failed");
    return;
  }
}

void set_factors(const mem *c) {
  calculator vla;
  vla.memory = *c;
  if(ioctl(vga_ball_fd, CALCULATOR_WRITE_VALUES, &vla)) {
    perror("ioctl(calculator_SET_factors) failed");
    return;
  }
}

mem get_factors() {
  calculator res;
  if(ioctl(vga_ball_fd, CALCULATOR_READ_VALUES, &res)) {
    perror("ioctl(calculator_read_res) failed");
    return;
  }
  return res.memory;
}

int get_done() {
  calculator res;
  if(ioctl(vga_ball_fd, CALCULATOR_READ_DONE, &res)) {
    perror("ioctl(calculator_read_done) failed");
    return 0;
  }
  return res.memory.done;
}

//----------for keyboard
struct libusb_device_handle *keyboard;
uint8_t endpoint_address;

struct libusb_device_handle *openkeyboard(uint8_t *endpoint_address);
void pressKeyAndPrint(struct libusb_device_handle *keyboard, mem *m, int *status, int *cur,  FMModel *model,  int *row_num, int**data);

// Keycode to ASCII mapping for a basic US keyboard layout without Shift key.
// This table includes the basic alphabet, numbers, and now the arrows, Enter, and Space.
char keycode_to_ascii[256] = {
    [0x04] = 'a', [0x05] = 'b', [0x06] = 'c', [0x07] = 'd',
    [0x08] = 'e', [0x09] = 'f', [0x0A] = 'g', [0x0B] = 'h',
    [0x0C] = 'i', [0x0D] = 'j', [0x0E] = 'k', [0x0F] = 'l',
    [0x10] = 'm', [0x11] = 'n', [0x12] = 'o', [0x13] = 'p',
    [0x14] = 'q', [0x15] = 'r', [0x16] = 's', [0x17] = 't',
    [0x18] = 'u', [0x19] = 'v', [0x1A] = 'w', [0x1B] = 'x',
    [0x1C] = 'y', [0x1D] = 'z', [0x1E] = '1', [0x1F] = '2',
    [0x2C] = ' ',    // Space
    [0x28] = '\n',   // Enter
    [0x4F] = '>',    // Right arrow
    [0x50] = '<',    // Left arrow
    [0x51] = 'v',    // Down arrow
    [0x52] = '^',    // Up arrow
    // Add more mappings as needed
};

void pressKeyAndPrint(struct libusb_device_handle *keyboard, mem *m, int *status, int *cur, FMModel *model, int *row_num, int**data) {
    struct usb_keyboard_packet packet;
    int transferred;
    
    int res = 0;
    if (libusb_interrupt_transfer(keyboard, endpoint_address,
                                  (unsigned char *)&packet, sizeof(packet),
                                  &transferred, 5000) == 0 && transferred == sizeof(packet)) {
        if (packet.keycode[0] != 0) {
            char key = keycode_to_ascii[packet.keycode[0]];
	    if(key=='\n') printf("Key Pressed: enter\n");
            else printf("Key Pressed: %c\n", key);
            if (key != '\0') {
		            if(key == 'v'){
		  
                  if(*cur==9||*cur==10){
                      *cur = 0;
                  }
                  else{
                     *cur+=1;
                  }
                }
                else if(key == '^'){
                  if(*cur==0){
                      *cur = 9;
                  }
                  else{
                     *cur-=1;
                  }
                }else if(key=='<'||key=='>'){
                    if(*cur==9){
                        *cur=10;
                    }else{
                        *cur=9;
                    }
		            }
                else if(key=='\n'){ //press enter key
                    if(*cur == 9 ){
                        run_model(m, model, status);
                        set_IC(m);
                        printf("Key Pressed: %c\n", key);
                    }
                    else if( *cur ==10){
			
                        run_test(m, model, status, row_num, &data);
			*row_num = (rand()%(MAX_RANDOM-MIN_RANDOM+1))+MIN_RANDOM;
                        set_ball_values(m);
                        printf("Key Pressed: %c\n", key);
                        
                    }
                    else{  clock_t start_time = clock();
                        status[*cur] = status[*cur] ^ 1;
                    }
                }
                for(int i=10;i>=0;i--){
                  if(i==*cur){
                    res <<= 1;
                    res |= 1;
                  }
                  else{
                    res <<= 1;
                    res |= status[i];
                  }
                }
                m->chars = res;
                m->cursor = *cur;
                printf("chars:%d current cur %d\n", m->chars, m->cursor);
            } else {
                printf("Unknown Key: %02x\n", packet.keycode[0]);
            }
        }
    }
}

//------------------model------------
// Function to load data from a CSV file
void load_data(const char *filename, Matrix *features, Matrix *labels) {
  FILE *file = fopen(filename, "r");
  if (!file) {
    perror("Failed to open file");
    exit(EXIT_FAILURE);
  }

  int count = 0; // A counter to track how many rows have been successfully read
  double check; // A variable to check fscanf output temporarily

  // Read the file line by line
  for (int i = 0; i < features->rows; i++) {
    int result = fscanf(file, "%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf",
                        &features->data[i][0], &features->data[i][1],
                        &features->data[i][2], &features->data[i][3],
                        &features->data[i][4], &features->data[i][5],
                        &features->data[i][6], &features->data[i][7],
                        &features->data[i][8], &labels->data[i][0]);
    if (result == EOF) {
      printf("Reached EOF unexpectedly.\n");
      break; // Stop reading if we reach end of file prematurely
    } else if (result != 10) {
      printf("Incorrect number of items read: %d\n", result);
      continue; // Skip this line if the data was incorrect
    }
    count++;
    //printf("Row %d - Last Feature: %f, Label: %f\n", i, features->data[i][8], labels->data[i][0]);
  }
  printf("Total rows read: %d\n", count);

  fclose(file);
}


void run_model(mem *m, FMModel *model, int *status) {
  // Assume data has 100 rows and we are using 5 features
  int rows = 340;       // total number of data points
  int cols = 9;         // number of features
  int valid_rows = 10; // 10% for validation
  int train_rows = 320; // 10% for training

  // Creating matrices for features and labels
  Matrix features = create_matrix(rows, cols);
  Matrix labels = create_matrix(rows, 1);

  // Load data from a pre-processed file
  load_data("processed_AFRM.csv", &features, &labels);
  double *means = malloc(cols * sizeof(double));
  double *stddevs = malloc(cols * sizeof(double));
  compute_means(features, means);
  compute_stddevs(features, means, stddevs);
  standardize_features(features, means, stddevs);

  // Splitting data into train and test manually
  Matrix features_train = create_matrix(train_rows, cols);
  Matrix labels_train = create_matrix(train_rows, 1);
  Matrix features_test = create_matrix(rows - train_rows-valid_rows, cols);
  Matrix labels_test = create_matrix(rows - train_rows-valid_rows, 1);
  Matrix features_valid = create_matrix(valid_rows, cols);
  Matrix labels_valid = create_matrix(valid_rows, 1);

  // Copying data to train and test matrices
  for (int i = 0; i < train_rows; i++) {
    for (int j = 0; j < cols; j++) {
      features_train.data[i][j] = features.data[i][j];
    }
    labels_train.data[i][0] = labels.data[i][0];
  }
  // Copying data to test matrices
  for (int i = train_rows; i < rows-valid_rows; i++) {
    for (int j = 0; j < cols; j++) {
      features_test.data[i - train_rows][j] = features.data[i][j];
    }
    labels_test.data[i - train_rows][0] = labels.data[i][0];
  }

    // Copying data to valid
  for (int i = rows-valid_rows; i < rows; i++) {
    for (int j = 0; j < cols; j++) {
      features_valid.data[i - rows+valid_rows][j] = features.data[i][j];
    }
    labels_valid.data[i - rows+valid_rows][0] = labels.data[i][0];
  }

  
  fit(model, features_train, labels_train, 9, 0.0005,20, 30, 0.95);
  // Parameters are void fit(FMModel *model, Matrix X, Matrix y, int
  // feature_potential, double learning rate, int epoch , int batch_size, double
  // learning_rate_decay_rate); Predict
  
  double *predictions = predict_active(model, features_test,labels_test,status);

  // Print predictions
  for (int i = 0; i < rows - train_rows-valid_rows; i++) {
    printf("Factor_Value: %f, Actual: %f\n", predictions[i] ,
           labels_test.data[i][0]);
  }
  double *valid_predictions = predict_active(model, features_valid,labels_valid,status);

  // Print predictions
  for (int i = 0; i < features_valid.rows; i++) {
    printf("Factor_Value: %f, Actual_Valid: %f\n", valid_predictions[i] ,
           labels_valid.data[i][0]);
  }


  int ic_int;

  double ic =
  pearson_correlation(predictions, labels_test.data, rows - train_rows - valid_rows);
  printf("Information Coefficient (IC): %f\n", ic);
 //tranfer IC value bit by bit
  ic_int = round(fabs(ic) * 1000);
  m->ic1 = ic_int / 100;
  m->ic2 = (ic_int - m->ic1 * 100) / 10;
  m->ic3 = ic_int % 10;
  // Cleanup
  // free_matrix(features);
  // free_matrix(labels);
  // free_matrix(features_train);
  // free_matrix(labels_train);
  // free_matrix(features_test);
  // free_matrix(labels_test);
  // free(predictions);
}


void run_test(mem *m, FMModel *model, int *status, int *row_num, int data[][COLUMNS]) {
	m->sign |= 1<<2;
       	m->open = data[*row_num][0];
        m->high = data[*row_num][1];
        m->low = data[*row_num][2];
        m->close = data[*row_num][3];
        m->volume = data[*row_num][4];

        float momentum, volumeOut, rsi=0;
  	clock_t start_time = clock();
        set_factors(m);
	mem res;
         while(1){
	
      		int temp = get_done();
	  	      if(temp){
               	res = get_factors();
               momentum = fixed_to_float(res.momentum);
               volumeOut = fixed_to_float(res.volumeOut);
               rsi = fixed_to_float(res.rsi);
            clock_t end_time = clock();
  	    double elapsed_time = (double)(end_time - start_time) / CLOCKS_PER_SEC; 
	     printf("Time taken for calculator: %f seconds\n", elapsed_time);  
            break;

            }
            // usleep(200);
    
      }

  // should add an input of active_features here.
  // Assume data has 100 rows and we are using 5 features
  int rows = 340;       // total number of data points
  int cols = 9;         // number of features
  int valid_rows = 10; // 10% for validation
  int train_rows = 320; // 10% for training

  Matrix features = create_matrix(rows, cols);
  Matrix labels = create_matrix(rows, 1);

  load_data("processed_AFRM.csv", &features, &labels);
  double *means = malloc(cols * sizeof(double));
  double *stddevs = malloc(cols * sizeof(double));
  compute_means(features, means);
  compute_stddevs(features, means, stddevs);
  standardize_features(features, means, stddevs);

  Matrix features_valid = create_matrix(1, cols);
  Matrix labels_valid = create_matrix(1, 1);
    // Copying data to valid
  
    for (int j = 0; j < cols; j++) {
      features_valid.data[0][j] = features.data[*row_num][j];
    }
features_valid.data[0][5] = momentum;
features_valid.data[0][6] = volumeOut;
features_valid.data[0][7] = rsi;
    labels_valid.data[0][0] = labels.data[*row_num][0];
  

  double *valid_predictions = predict_active(model, features_valid,labels_valid,status);

  // Print predictions
  for (int i = 0; i < features_valid.rows; i++) {
    printf("Test_Factor_Value: %f, Actual_Valid: %f\n", valid_predictions[i] ,
           labels_valid.data[i][0]);
  }
  int actual_int;
  int factor_int;
  //factor value
  factor_int = round((valid_predictions[0])*100);
  if (factor_int < 0) {
  m->sign |= 1<<1;
  } 
  else{
    m->sign &= 0b101;

  }

  factor_int = fabs(factor_int);
  m->fac1 = factor_int / 10;
  m->fac2 = factor_int % 10;

  // actual return
  actual_int = round(labels_valid.data[0][0] * 100);
  if (actual_int < 0) {
  m->sign |= 1;
  } 
  else{
    m->sign &= 0b110;
  }

  actual_int = fabs(actual_int);
  m->actual1 = actual_int / 10;
  m->actual2 = actual_int % 10;

  // Cleanup
  free_matrix(features_valid);
  free_matrix(labels_valid);
  free(valid_predictions);
}

// Function to read data from the file and store it in a 2D array
void read_and_store_data(const char *filename, int start_row, int num_rows, int data[][COLUMNS]) {
    FILE *file = fopen(filename, "r");  // Open the CSV file for reading
    if (file == NULL) {
        perror("Failed to open file");
        exit(EXIT_FAILURE);
    }

    char buffer[1024];  // Buffer to store each line of the file
    int row_count = 0;  // Overall row counter
    int index = 0;  // Index for rows within the desired range

    // Read lines from file until the end is reached
    while (fgets(buffer, sizeof(buffer), file)) {
        row_count++;  // Increment the row counter for each line read

        // Check if the current row is within the desired range
        if (row_count >= start_row && row_count < start_row + num_rows) {
            // Parse the integers from the line
            if (sscanf(buffer, "%d,%d,%d,%d,%d", &data[index][0], &data[index][1], &data[index][2], &data[index][3], &data[index][4]) == 5) {
                // printf("Row %d stored: %d, %d, %d, %d, %d\n", row_count, data[index][0], data[index][1], data[index][2], data[index][3], data[index][4]);
                index++;  // Move to the next row in the array
            } else {
                printf("Error in row %d: Line format incorrect\n", row_count);
            }
        }

        // Stop reading once we reach the end of the required range
        if (row_count >= start_row + num_rows - 1) {
            break;
        }
    }

    fclose(file);  // Close the file
}


int main(int argc, char *argv[])
{
  // vga_ball_arg_t vla;
  // default position and radius
  mem memory;
  calculator vla;
  memory.chars = 0;
  memory.actual1 = 0;
  memory.actual2 = 0;
  memory.ic1 = 0;
  memory.ic2 = 0;
  memory.ic3 = 0;
  memory.sign = 0;
  memory.open = 16472;
  memory.high = 16537;
  memory.low = 16362;
  memory.close = 16382;
  memory.volume = 3451;
  memory.momentum = 0;
  memory.volumeOut = 0;
  memory.rsi = 0;
  memory.ma = 0;
  memory.done = 0;
  memory.fac1 = 0;
  memory.fac2 = 0;
  memory.cursor = 0;
  static const char filename[] = "/dev/vga_ball";
  int status[11] = {0,0,0,0,0,0,0,0,0,0,0}; // feature 0-9+ train +test
  int cur = 0;
  int rowNum =(rand()%(MAX_RANDOM-MIN_RANDOM+1))+MIN_RANDOM;


  // Initialize the model
  FMModel model; 
  // for keyboard
  if (libusb_init(NULL) != LIBUSB_SUCCESS) {
        fprintf(stderr, "Failed to initialize libusb\n");
        return -1;
    }

    keyboard = openkeyboard(&endpoint_address);
    if (keyboard == NULL) {
        fprintf(stderr, "Failed to open keyboard\n");
        libusb_exit(NULL);
        return -1;
    }

  printf("VGA ball Userspace program started\n");

  if ( (vga_ball_fd = open(filename, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", filename);
    return -1;
  }
// ------------------- for model-----------------
FILE *file = fopen("output.csv", "w");
    if (file == NULL) {
        perror("Failed to open file");
        return -1;
    }

//------------init
  printf("initial state: ");
 
  set_ball_character(&memory);
  set_ball_values(&memory);
  set_IC(&memory);

//-----------read data from csv file
    const int ROWS = 340;  
    int data[ROWS][COLUMNS];  
    int start_row = 0;  // set start row num
    int num_rows = ROWS;

    // set csv file
    read_and_store_data("AFRM.csv", start_row, num_rows, data);

  clock_t start_time1 = clock();
     for (int i = 0; i < 30; i++) {
         memory.open = data[i][0];
         memory.high = data[i][1];
         memory.low = data[i][2];
         memory.close = data[i][3];
         memory.volume = data[i][4];


	
         set_factors(&memory);
         while(1){
	     //usleep(500000);
   	     int temp = get_done();
    	     if(temp){
                  mem res = get_factors();
		  break;
   	     }
         }
    }
 clock_t end_time1 = clock();
  	    double elapsed_time = (double)(end_time1 - start_time1) / CLOCKS_PER_SEC; 
	     printf("Time taken for calculator days: %f seconds\n", elapsed_time);  
    
    while (1)
    {
       pressKeyAndPrint(keyboard, &memory, status, &cur, &model, &rowNum, data);
       set_ball_character(&memory);
       usleep(10000);
    }
    

    
    printf("VGA BALL Userspace program terminating\n");
    return 0;
}
