/* Wenbo Liu
 * Columbia University
 */

module calculator(input logic        clk,
	        input logic 	   reset,
		input logic [15:0]  writedata,
		input logic 	   write, read,
		input 		   chipselect,
		input logic [4:0]  address,
        output logic [15:0] readdata
);

    logic write_successful, write_price, write_volume;
    logic [15:0] reg_array[30][10]; //30 groups, 10 for each group, 16 bits for each reg
    logic [4:0] reg_idx;
    logic [3:0] reg_within_idx;    // Index within a group
    logic [15:0] sum_volume, sum_price;
    logic [15:0] sum_gain, sum_loss;
    logic [3:0] isValid; // 1000: momentum; 0100: volume factor, etc.. Only when it is 1111 all the factors will be available
    //output
    logic [15:0] momentum, volume, rsi, ma, done;

    // DIV 1, to calculate momentum
    logic start_div1, busy_div1, done_div1, valid_div1, dbz_div1, ovf_div1;
    logic signed [15:0] div_result1;
    logic signed [15:0] locked_dividend1, locked_divisor1;  // Locked values for division
    // Instance of the div module
    div #(.WIDTH(16), .FBITS(8)) div_instance1(
        .clk(clk),
        .rst(reset),
        .start(start_div1),
        .busy(busy_div1),
        .done(done_div1),
        .valid(valid_div1),
        .dbz(dbz_div1),
        .ovf(ovf_div1),
        .a(locked_dividend1),
        .b(locked_divisor1),
        .val(div_result1)
    );

    // DIV2, to calculate volume factor
    logic start_div2, busy_div2, done_div2, valid_div2, dbz_div2, ovf_div2;
    logic signed [15:0] div_result2;
    logic signed [15:0] locked_dividend2, locked_divisor2;  // Locked values for division
    // Instance of the div module
    div #(.WIDTH(16), .FBITS(8)) div_instance2(
        .clk(clk),
        .rst(reset),
        .start(start_div2),
        .busy(busy_div2),
        .done(done_div2),
        .valid(valid_div2),
        .dbz(dbz_div2),
        .ovf(ovf_div2),
        .a(locked_dividend2),
        .b(locked_divisor2),
        .val(div_result2)
    );

    // DIV3, to calculate MA
    logic start_div3, busy_div3, done_div3, valid_div3, dbz_div3, ovf_div3;
    logic signed [15:0] div_result3;
    logic signed [15:0] locked_dividend3, locked_divisor3;  // Locked values for division
    // Instance of the div module
    div #(.WIDTH(16), .FBITS(8)) div_instance3(
        .clk(clk),
        .rst(reset),
        .start(start_div3),
        .busy(busy_div3),
        .done(done_div3),
        .valid(valid_div3),
        .dbz(dbz_div3),
        .ovf(ovf_div3),
        .a(locked_dividend3),
        .b(locked_divisor3),
        .val(div_result3)
    );   

    // DIV4, to calculate sum_gain/sum_loss
    logic start_div4, busy_div4, done_div4, valid_div4, dbz_div4, ovf_div4;
    logic signed [15:0] div_result4;
    logic signed [15:0] locked_dividend4, locked_divisor4;  // Locked values for division
    // Instance of the div module
    div #(.WIDTH(16), .FBITS(8)) div_instance4(
        .clk(clk),
        .rst(reset),
        .start(start_div4),
        .busy(busy_div4),
        .done(done_div4),
        .valid(valid_div4),
        .dbz(dbz_div4),
        .ovf(ovf_div4),
        .a(locked_dividend4),
        .b(locked_divisor4),
        .val(div_result4)
    );     

    // DIV5, to calculate RSI
    logic start_div5, busy_div5, done_div5, valid_div5, dbz_div5, ovf_div5;
    logic signed [15:0] div_result5;
    logic signed [15:0] locked_dividend5, locked_divisor5;  // Locked values for division
    // Instance of the div module
  div #(.WIDTH(16), .FBITS(8)) div_instance5(
        .clk(clk),
        .rst(reset),
        .start(start_div5),
        .busy(busy_div5),
        .done(done_div5),
        .valid(valid_div5),
        .dbz(dbz_div5),
        .ovf(ovf_div5),
        .a(locked_dividend5),
        .b(locked_divisor5),
        .val(div_result5)
    );      

    logic [15:0] diff;
    logic  write_diff;

    // logic [15:0] momentum_tmp;
    // assign momentum_tmp = reg_array[reg_idx][7];

    always_ff @(posedge clk)  begin
        if (reset) begin
            reg_idx <= 0;
            sum_price = 0;
            sum_volume = 0;
            // price_start <= 0;
            sum_gain = 0;
            sum_loss = 0;
            start_div1 <= 0;
            start_div2 <= 0; 
            start_div3 <= 0; 
            start_div4 <= 0; 
            start_div5 <= 0; 
            isValid <= 0;
            write_price = 0;
            write_successful = 0;
            write_volume = 0;
            momentum = 0;
            volume = 0;
            rsi = 0;
            ma = 0;
            done = 0;
            reg_within_idx <= 0;
	    //readdata <= 0;
            for (int group = 0; group < 30; group++) begin
                for (int reg_num = 0; reg_num < 10; reg_num++) begin
                    reg_array[group][reg_num] <= 16'd0;  
                end
            end

        end 
        else if (chipselect && read) begin
            case (address) 
                5'hc : begin
                    readdata <= momentum;
                end
                5'hd : begin
                    readdata <= volume;
                end
                5'he : begin
                    readdata <= rsi;
                end
                5'hf : begin
                    readdata <= ma;
                end
                5'h10 : begin
                    readdata <= done;
                end
            endcase
        end
        else if (chipselect && write) begin
            done = 0;
            if(reg_within_idx < 5) begin
                case (address)
                    5'h7 : begin 
                        reg_array[reg_idx][0] <= writedata; //open
                        write_successful = 1;
                    end
                    5'h8 : begin
                        reg_array[reg_idx][1] <= writedata; //high
                        write_successful = 1;
                    end
                    5'h9 : begin
                        reg_array[reg_idx][2] <= writedata; //low
                        write_successful = 1;
                    end
                    5'ha : begin
                        reg_array[reg_idx][3] <= writedata; //close
                        write_successful = 1;
                        write_price = 1;
                    end
                    5'hb : begin
                        reg_array[reg_idx][4] <= writedata; // volume
                        write_successful = 1;
                        write_volume = 1;
                    end
                endcase
            end
            if(write_successful) begin
                reg_within_idx <= reg_within_idx + 1;
                write_successful = 0;
            end
        end
        if(reg_array[reg_idx][4] != 0 && write_volume) begin
            sum_volume = sum_volume + reg_array[reg_idx][4];
            write_volume = 0;
        end
        if(reg_array[reg_idx][3] != 0 && write_price) begin
            write_price = 0;
            sum_price = sum_price + reg_array[reg_idx][3];
            // if(reg_idx == 0) diff = 0;
            // else diff = reg_array[reg_idx][3] - reg_array[reg_idx-1][3];
            // if(diff < 0) sum_loss = sum_loss - diff;
            // else sum_gain = sum_gain + reg_array[reg_idx][9];
            if(reg_idx != 0) begin
                if(reg_array[reg_idx][3] > reg_array[reg_idx-1][3]) begin
                    sum_gain = sum_gain + reg_array[reg_idx][3] - reg_array[reg_idx-1][3];
                end
                else begin
                    sum_loss = sum_loss + reg_array[reg_idx-1][3] - reg_array[reg_idx][3];
                end
            end
            write_diff = 1;
        end
        if (write_diff && !reg_array[reg_idx][9]) begin
            // reg_array[reg_idx][9] <= diff;
            if(reg_array[reg_idx][3] > reg_array[reg_idx-1][3]) begin
                reg_array[reg_idx][9] <= reg_array[reg_idx][3] - reg_array[reg_idx-1][3];
            end
            else begin
                reg_array[reg_idx][9] <= reg_array[reg_idx-1][3] - reg_array[reg_idx][3];
            end
            write_diff = 0;
        end
        if (done_div1) begin
            if(reg_array[reg_idx][5] == 0) begin
                reg_array[reg_idx][5] <= div_result1 - (16'b1 << 8);  // Store the momentum
            end
            start_div1 <= 0;
            isValid <= isValid | 4'b1000;
            reg_within_idx <= reg_within_idx + 1;
            // done_div1 <= 0;
        end
        if (done_div2) begin
            if(reg_array[reg_idx][6] == 0) begin
                reg_array[reg_idx][6] <= div_result2; // Store the volume factor
            end
            start_div2 <= 0;
            isValid <= isValid | 4'b0100;
            reg_within_idx <= reg_within_idx + 1;
            // done_div2 <= 0;
        end
        if (done_div4) begin
            locked_dividend5 <= 16'd100 << 8;
            locked_divisor5 <= div_result4 + (16'b1 << 8);
            start_div5 <= 1;
            start_div4 <= 0;
            // done_div4 <= 0;
        end
        if (done_div5) begin
            if(reg_array[reg_idx][7] == 0) begin
                reg_array[reg_idx][7] <= (16'd100 << 8) - div_result5; // Store the RSI
            end
            start_div5 <= 0;
            isValid <= isValid | 4'b0010;
            reg_within_idx <= reg_within_idx + 1;
            // done_div5 <= 0;
        end

        if (done_div3) begin
            if(reg_array[reg_idx][8] == 0) begin
                reg_array[reg_idx][8] <= div_result3; // Store the MA factor
            end
            start_div3 <= 0;
            isValid <= isValid | 4'b0001;
            reg_within_idx <= reg_within_idx + 1;
            // done_div3 <= 0;
        end

        if (reg_within_idx == 10) begin
            // finished factors calculation
            momentum = reg_array[reg_idx][5];
            volume = reg_array[reg_idx][6];
            rsi = reg_array[reg_idx][7];
            ma = reg_array[reg_idx][8];
            done = 1;

            reg_idx <= reg_idx + 1;
            isValid <= 0;
            reg_within_idx <= 0; // Reset index within group
        end

        if (reg_idx == 30) begin
            sum_price = sum_price - reg_array[0][3];
            sum_volume = sum_volume - reg_array[0][4];
            if(reg_array[0][9] < 0) sum_loss = sum_loss + reg_array[0][9];
            else sum_gain = sum_gain - reg_array[0][9];
            for (int i = 0; i < 29; i++) begin
                reg_array[i] <= reg_array[i+1];
            end
            for (int j = 0; j < 10; j++) begin
                reg_array[29][j] <= 16'd0;
            end
            reg_idx <= 29; // Keep the index at the last group ready for new data
        end
        if(reg_within_idx == 5) begin
            // Start factor calculation
            locked_dividend1 = reg_array[reg_idx][3];  // Close price of current group
            locked_divisor1 = reg_array[0][3];         // Close price of first group
            start_div1 <= 1;  // Start division
            locked_dividend2 = sum_volume;
            locked_divisor2 = {3'b0, reg_idx, 8'b0} + 16'b100000000;
            start_div2 <= 1;
            // Calculate RSI
            locked_dividend4 = sum_gain << 8;
            locked_divisor4 = sum_loss << 8;
            start_div4 <= 1;

            locked_dividend3 = sum_price;
            locked_divisor3 = {3'b0, reg_idx, 8'b0} + 16'b100000000;
            start_div3 <= 1;

            // reg_within_idx <= 0; // Reset index within group
        end
    end
endmodule

//`default_nettype none
//`timescale 1ns / 1ps

module div #(
    parameter WIDTH=16,  // width of numbers in bits (integer and fractional)
    parameter FBITS=8   // fractional bits within WIDTH
    ) (
    input wire logic clk,    // clock
    input wire logic rst,    // reset
    input wire logic start,  // start calculation
    output     logic busy,   // calculation in progress
    output     logic done,   // calculation is complete (high for one tick)
    output     logic valid,  // result is valid
    output     logic dbz,    // divide by zero
    output     logic ovf,    // overflow
    input wire logic signed [WIDTH-1:0] a,   // dividend (numerator)
    input wire logic signed [WIDTH-1:0] b,   // divisor (denominator)
    output     logic signed [WIDTH-1:0] val  // result value: quotient
    );

    localparam WIDTHU = WIDTH - 1;                 // unsigned widths are 1 bit narrower
    localparam FBITSW = (FBITS == 0) ? 1 : FBITS;  // avoid negative vector width when FBITS=0
    localparam SMALLEST = {1'b1, {WIDTHU{1'b0}}};  // smallest negative number

    localparam ITER = WIDTHU + FBITS;  // iteration count: unsigned input width + fractional bits
    logic [$clog2(ITER):0] i;          // iteration counter (allow ITER+1 iterations for rounding)

    logic a_sig, b_sig, sig_diff;      // signs of inputs and whether different
    logic [WIDTHU-1:0] au, bu;         // absolute version of inputs (unsigned)
    logic [WIDTHU-1:0] quo, quo_next;  // intermediate quotients (unsigned)
    logic [WIDTHU:0] acc, acc_next;    // accumulator (unsigned but 1 bit wider)

    // input signs
    always_comb begin
        a_sig = a[WIDTH-1+:1];
        b_sig = b[WIDTH-1+:1];
    end

    // division algorithm iteration
    always_comb begin
        if (acc >= {1'b0, bu}) begin
            acc_next = acc - bu;
            {acc_next, quo_next} = {acc_next[WIDTHU-1:0], quo, 1'b1};
        end else begin
            {acc_next, quo_next} = {acc, quo} << 1;
        end
    end

    // calculation state machine
    enum {IDLE, INIT, CALC, ROUND, SIGN} state;
    always_ff @(posedge clk) begin
        done <= 0;
        case (state)
            INIT: begin
                state <= CALC;
                ovf <= 0;
                i <= 0;
                {acc, quo} <= {{WIDTHU{1'b0}}, au, 1'b0};  // initialize calculation
            end
            CALC: begin
                if (i == WIDTHU-1 && quo_next[WIDTHU-1:WIDTHU-FBITSW] != 0) begin  // overflow
                    state <= IDLE;
                    busy <= 0;
                    done <= 1;
                    ovf <= 1;
                end else begin
                    if (i == ITER-1) state <= ROUND;  // calculation complete after next iteration
                    i <= i + 1;
                    acc <= acc_next;
                    quo <= quo_next;
                end
            end
            ROUND: begin  // Gaussian rounding
                state <= SIGN;
                if (quo_next[0] == 1'b1) begin  // next digit is 1, so consider rounding
                    // round up if quotient is odd or remainder is non-zero
                    if (quo[0] == 1'b1 || acc_next[WIDTHU:1] != 0) quo <= quo + 1;
                end
            end
            SIGN: begin  // adjust quotient sign if non-zero and input signs differ
                state <= IDLE;
                if (quo != 0) val <= (sig_diff) ? {1'b1, -quo} : {1'b0, quo};
                busy <= 0;
                done <= 1;
                valid <= 1;
            end
            default: begin  // IDLE
                if (start) begin
                    valid <= 0;
                    if (b == 0) begin  // divide by zero
                        state <= IDLE;
                        busy <= 0;
                        done <= 1;
                        dbz <= 1;
                        ovf <= 0;
                    end else if (a == SMALLEST || b == SMALLEST) begin  // overflow
                        state <= IDLE;
                        busy <= 0;
                        done <= 1;
                        dbz <= 0;
                        ovf <= 1;
                    end else begin
                        state <= INIT;
                        au <= (a_sig) ? -a[WIDTHU-1:0] : a[WIDTHU-1:0];  // register abs(a)
                        bu <= (b_sig) ? -b[WIDTHU-1:0] : b[WIDTHU-1:0];  // register abs(b)
                        sig_diff <= (a_sig ^ b_sig);  // register input sign difference
                        busy <= 1;
                        dbz <= 0;
                        ovf <= 0;
                    end
                end
            end
        endcase
        if (rst) begin
            state <= IDLE;
            busy <= 0;
            done <= 0;
            valid <= 0;
            dbz <= 0;
            ovf <= 0;
            val <= 0;
        end
    end

    // generate waveform file with cocotb
    `ifdef COCOTB_SIM
    initial begin
        $dumpfile($sformatf("%m.vcd"));
        $dumpvars;
    end
    `endif
endmodule
