{-# LANGUAGE DeriveGeneric, DeriveAnyClass #-}

module Sim (runSimLoop, simResultToByteString, SimResult(..), SimArgs(..)) where
import Scene (Scene, SpawnerConfig(..), updateScene, initScene)

import Control.Parallel.Strategies (NFData)

import Config (SimConfig(..), getSceneConfig, setSceneConfig, getSpawnerConfig)
import Control.DeepSeq (rnf)
import qualified Data.ByteString as B
import qualified Data.ByteString.Char8 as BSC

-- Arguments passed in via command line (not JSON)
data SimArgs = SimArgs {
  numIters :: Int,
  noDisplay :: Bool
}

-- The result of a full simulation
data SimResult = SimResult SimConfig [B.ByteString] 

-- Convert simulation result to a bytestring for file writing
simResultToByteString :: SimResult -> B.ByteString
simResultToByteString (SimResult config sceneBytes) = (BSC.pack (show config)) `BSC.append` (BSC.concat sceneBytes)

instance NFData SimResult where
    rnf (SimResult _ scenes) = rnf scenes


-- Runs simulation and returns printable result
runSimLoop :: SimArgs -> SimConfig -> SimResult
runSimLoop simArgs simConfig = SimResult simConfig byteStringResult
    where 
        byteStringResult =  take (numIters simArgs) sceneGenerator

        sceneGenerator = buildByteResult initialScene updateFunc

        initialScene = initScene (spRandomSeed spawnerConfig) sceneConfig
        updateFunc = updateScene sceneConfig spawnerConfig (noDisplay simArgs)

        sceneConfig = getSceneConfig simConfig
        spawnerConfig = getSpawnerConfig simConfig


-- Builds infinite list of simulation byte result
-- Takes update function that generates the next scene and converts the current scene to bytes
buildByteResult :: Scene -> (Scene -> (Scene, B.ByteString)) -> [B.ByteString]
buildByteResult currScene updateFunc = currBytes : (buildByteResult nextScene updateFunc)
 where
    (nextScene, currBytes) = updateFunc currScene