{-# LANGUAGE DeriveGeneric, DeriveAnyClass #-}

module Scene (Scene, SceneConfig(..), SpawnerConfig(..), generateChunks, updateScene, updateConsecPairsSkipLast,updateCarsChunk, initScene) where

import Car (Car(..), CarConfig(..), initCar, initCarState, updateCarPosVel, updateCarAccel, carWithinBound)

import Data.List (sort)
import Control.Parallel.Strategies (parMap, NFData, rdeepseq)
import GHC.Generics (Generic)
import qualified Data.ByteString as B
import qualified Data.ByteString.Char8 as BSC
import System.Random (randomR, mkStdGen)

-- Configuration for car spawner
data SpawnerConfig = SpawnerConfig {
   spRandomSeed :: Int,
   spSpawnRate :: Int,
   spSpawnSpeed :: Float,
   spConfigFreqs :: [Float],
   spCarConfigs :: [CarConfig] 
} deriving (Show, Eq)

type SceneChunk = [Car]

-- Configuration for initial scene construction
data SceneConfig = SceneConfig {
    scBound :: Float,
    scStepTime :: Float,
    scLanes :: Int,
    scInitConfig :: CarConfig,      
    scInitCarCount :: Int
    } deriving (Show, Eq)

-- Datatype representing a lane of cars processed up until a given timestep
data Lane = Lane {
    lnCars :: [Car],
    lnStep :: Int,
    lnMaxCarId :: Int, -- ID's are per-lane
    lnSeed :: Int,
    lnID :: Int
} deriving (Show, Generic, NFData)  

-- Datatype representing the full scene 
data Scene = Scene {
    sLanes :: [Lane]
 } deriving (Show, Generic, NFData)

-- Initialize a lane with given boundary, car count, and initial car configuration
initLane :: Int -> Int -> Float -> Int -> CarConfig -> Lane
initLane laneID startSeed bound carCount carConfig = Lane initCars 0 carCount startSeed laneID
 where
    initCars = sort $ map (initCarFunc) [1..carCount]
    initCarFunc index = initCar index carConfig (initCarState (posStep * (fromIntegral index)) 35) -- TODO make configurable (starting speed for pre-init cars)
    posStep = bound / (fromIntegral carCount)

initScene :: Int -> SceneConfig -> Scene
initScene startSeed sceneConfig = Scene $ map genLane [0..numLanes]
 where
    numLanes = (scLanes sceneConfig)
    genLane laneID = initLane laneID (startSeed*laneID) (scBound sceneConfig) (scInitCarCount sceneConfig) (scInitConfig sceneConfig)

updateScene :: SceneConfig -> SpawnerConfig -> Bool -> Scene -> (Scene, B.ByteString)
updateScene sceneConfig spawnerConfig noDisplay scene = (nextScene, currBytes)
 where
    nextScene = Scene nextLanes
    currBytes = BSC.concat currLaneBytes

    (nextLanes, currLaneBytes) = unzip $ parMap rdeepseq updateLaneFunc currLanes
    
    updateLaneFunc = updateLane noDisplay sceneConfig spawnerConfig
    currLanes = sLanes scene 

-- Updates a single lane and returns the bytestring representation of the current lane
-- and the next lane 
updateLane :: Bool -> SceneConfig -> SpawnerConfig -> Lane -> (Lane, B.ByteString)
updateLane noDisplay sceneConfig spawnerConfig lane = (nextLane, currLaneBytes)
 where
    nextLane = Lane nextCars nextStep nextMaxID nextSeed (lnID lane)
    nextStep = (lnStep lane) + 1
    
    currLaneBytes =  laneIDBytes `BSC.append` currLaneBytesRaw
    laneIDBytes = BSC.pack ("\n{LaneID = " ++ (show (lnID lane)) ++ ", Step = " ++ (show (lnStep lane)) ++ "}")
    
    (nextCars, currLaneBytesRaw) = processChunks noDisplay deltaT chunks
    deltaT = scStepTime sceneConfig

    chunks = generateChunks 150 $ newCars ++ [dummyCar]
    dummyCar = initCar (-1) (CarConfig 0 0 0 0 0) (initCarState  100 10000000.0) -- really big so definetly out of the way

    (nextMaxID, newCars)
        | shouldSpawn = spawnNewCar (lnMaxCarId lane) nextSeed (spSpawnSpeed spawnerConfig) (spCarConfigs spawnerConfig) (spConfigFreqs spawnerConfig) filteredCars
        | otherwise = (lnMaxCarId lane, filteredCars)
    nextSeed = (lnSeed lane) + 1
    shouldSpawn = nextStep `mod` (spSpawnRate spawnerConfig) == 0
    filteredCars = filterCars (scBound sceneConfig) (lnCars lane)
    

-- Spawn new car, chosing config based on random seed
-- Returns new list and new car's ID
spawnNewCar :: Int -> Int -> Float -> [CarConfig] -> [Float] -> [Car] -> (Int, [Car])
spawnNewCar currMaxID seed startSpeed carConfigs weights cars 
        | null carConfigs || null weights || length carConfigs /= length weights  = (currMaxID, cars)
        | otherwise = (nextID, newCar : cars)
         where 
            nextID = currMaxID + 1
            newCar = initCar nextID randConfig (initCarState startSpeed 0)

            randConfig = selectByWeight rand weights carConfigs  
            
            (rand, _) = randomR (0, totalWeight) gen -- Generate a random number within the range of total weights
            gen = mkStdGen seed
            totalWeight = sum weights

-- TODO test these         
-- Helper function to select an element based on the random number and weights
selectByWeight :: Float -> [Float] -> [a] -> a
selectByWeight rand weights options = selectByWeightHelper rand (scanl1 (+) weights) options

selectByWeightHelper :: Float -> [Float] -> [a] -> a
selectByWeightHelper _ _ [] = error "Empty list, this should not happen"
selectByWeightHelper rand (w:ws) (x:xs)
  | rand <= w  = x
  | otherwise = selectByWeightHelper rand ws xs



-- Take in a bound and filter all cars beyond that bound
filterCars :: Float -> [Car] -> [Car]
filterCars bound cars = filter (carWithinBound bound) cars

processChunks :: Bool -> Float -> [SceneChunk] -> ([Car], B.ByteString)
processChunks noDisplay deltaT chunks = (concat cars, B.concat byteStrings)
  where
    chunkResults = parMap rdeepseq (processChunk noDisplay deltaT) chunks
    (cars, byteStrings) = unzip chunkResults


-- Returns a chunk with one fewer cars
processChunk :: Bool -> Float -> SceneChunk -> (SceneChunk, B.ByteString)
processChunk noDisplay deltaT cars = (nextCars, currBytes)   
 where
    nextCars = updateCarsChunk deltaT cars
    currBytes
        | noDisplay = BSC.pack []
        | otherwise = BSC.pack $ show (init cars)

updateCarsChunk :: Float -> [Car] -> [Car]
updateCarsChunk deltaT chunk = map (updateCarPosVel deltaT) (updateConsecPairsSkipLast updateCarAccel chunk)

updateConsecPairsSkipLast :: (a -> a -> b) -> [a] -> [b]
updateConsecPairsSkipLast _ [] = []
updateConsecPairsSkipLast _ [_] = []
updateConsecPairsSkipLast update (x:xs) = update x (head xs) : updateConsecPairsSkipLast update xs


-- Generate chunks of size n from a list, with one element overlap between each chunk.
generateChunks :: Int -> [a] -> [[a]]
generateChunks _ [] = []
generateChunks n xs 
    | length xs > n = take n xs : generateChunks n (drop (n - 1) xs)
    | otherwise = [xs]
