{-# LANGUAGE DeriveGeneric #-}
{-# LANGUAGE OverloadedStrings #-}

module Config (SimConfig(..),
              parseConfigFile, 
              getSceneConfig,
              setSceneConfig, 
              getSpawnerConfig
              ) where

import Control.Exception (Exception, throwIO)

import Car (CarConfig(..), CarState(..))
import Scene (SceneConfig(..), SpawnerConfig(..))

import GHC.Generics (Generic)
import Data.Aeson (FromJSON, decode, parseJSON, withObject, (.:))
import qualified Data.ByteString.Lazy as B

-- "Parsable" Wrapper types to allow JSON Parsing -- 

-- State of a single car 
newtype ParsCarState = ParsCarState {unwrapCarState :: CarState}
  deriving (Show, Eq)

instance FromJSON ParsCarState where
  parseJSON = withObject "Car State" $ \v -> fmap ParsCarState $ CarState
    <$> v .: "Position"
    <*> v .: "Velocity"
    <*> v .: "Acceleration"

-- Base config for a car 
newtype ParsCarConfig = ParsCarConfig {unwrapCarConfig :: CarConfig}
  deriving (Show, Eq)

instance FromJSON ParsCarConfig where
  parseJSON = withObject "Car Config" $ \v -> fmap ParsCarConfig $ CarConfig
    <$> v .: "Desired Speed (m/s)"
    <*> v .: "Minimum Gap (m)"
    <*> v .: "Desired Time Gap (s)"
    <*> v .: "Max Deceleration (m/s^2)"
    <*> v .: "Max Acceleration (m/s^2)"

-- Car spawning configuration
newtype ParsSpawnerConfig = ParsSpawnerConfig {unwrapSpawnerConfig :: SpawnerConfig}
    deriving (Show, Eq)

instance FromJSON ParsSpawnerConfig where
    parseJSON = withObject "Spawner Config" $ \v -> do
       pRandomSeed <- v.: "Random Seed"
       pSpawnRate <- v.: "Spawn Rate (steps)"
       pSpawnSpeed <- v.: "Spawn Speed (m/s)"
       configFreqs <- v.: "Config Frequencies"
       parsCarConfigs <- v.: "Car Configs"

       let carConfigs = map (unwrapCarConfig) parsCarConfigs
       
       let spawnerConfig = SpawnerConfig pRandomSeed pSpawnRate pSpawnSpeed configFreqs carConfigs
       
       return $ ParsSpawnerConfig $ spawnerConfig

-- Initial scene confguration
newtype ParsSceneConfig = ParsSceneConfig { unwrapSceneConfig :: SceneConfig }
  deriving (Show, Eq)

instance FromJSON ParsSceneConfig where
  parseJSON = withObject "Initial Scene" $ \v -> do
    pMaxPosition <- v.: "Max Position"
    pStepTime <- v .: "Step Time (s)"
    pLanes <- v.: "Lanes"
    initCarConfig <- v.: "Initial Car Config"
    initCarCount <- v.: "Initial Car Count"

    let carConfig = unwrapCarConfig initCarConfig
    let sceneConfig = SceneConfig pMaxPosition pStepTime pLanes carConfig initCarCount
   
    return $ ParsSceneConfig $ sceneConfig

-- Overall simulation configuration (no need for non-parsable version)
data SimConfig = SimConfig {
  simSceneConfig :: ParsSceneConfig,
  simSpawnerConfig :: ParsSpawnerConfig
} deriving (Show, Eq, Generic)

instance FromJSON SimConfig

-- Set the SceneConfig attribute in a SimConfig
setSceneConfig :: SimConfig -> SceneConfig -> SimConfig
setSceneConfig simConfig sceneConfig = simConfig {simSceneConfig = newParsScConfig}
 where 
  newParsScConfig = ParsSceneConfig sceneConfig

-- Retrieve the SceneConfig from a SimConfig
getSceneConfig :: SimConfig -> SceneConfig
getSceneConfig simConfig = unwrapSceneConfig $ simSceneConfig simConfig

-- Retrieve the SpawnerConfig from a SimConfig
getSpawnerConfig :: SimConfig -> SpawnerConfig
getSpawnerConfig simConfig = unwrapSpawnerConfig $ simSpawnerConfig simConfig


data ParseError = ParseError String deriving (Show)
instance Exception ParseError

-- Parse the config file
parseConfigFile :: String -> IO SimConfig
parseConfigFile filePath = do
  fileContent <- B.readFile filePath
  case decode fileContent of
    Just config -> return config
    Nothing -> throwIO $ ParseError $ "Invalid JSON format in: " ++ (show filePath)

