{-# LANGUAGE DeriveGeneric, DeriveAnyClass #-}

module Car (Car(..), CarConfig(..), CarState(..), initCar, initCarState, 
moveCarStateForward, calcPos, calcVel, updateCarPosVel, carWithinBound, 
updateCarAccel, calcAccel, calcFreeAccel, calcBreakingAccel) where

import Control.Parallel.Strategies (NFData)
import GHC.Generics (Generic)
import qualified Data.ByteString as B

data CarState = CarState
  {
    csPos :: Float,
    csVel :: Float,
    csAcc :: Float
  } deriving (Show, Eq)

-- Initialize a car state to the given position
-- with 0 acceleration and velocity
initCarState :: Float -> Float -> CarState
initCarState pos vel = CarState pos vel 0 

-- Constants for a single car
data CarConfig = CarConfig
  {
      ccDesiredSpeed :: Float,
      ccMinGap :: Float,
      ccDesiredTimeGap :: Float,
      ccMaxDecel :: Float,
      ccMaxAccel :: Float
  } deriving (Show, Eq)


data Car = Car
  { cID :: Int,
    cPos :: Float,
    cVel :: Float,
    cAccel :: Float,
    cDesiredSpeed :: Float,
    cMinGap :: Float,
    cDesiredTimeGap :: Float,
    cMaxDecel :: Float,
    cMaxAccel :: Float
  } deriving (Generic, NFData, Show)

initCar :: Int -> CarConfig -> CarState -> Car
initCar carID carConfig carState =
        Car { cID         = carID
        , cPos            = csPos carState
        , cVel            = csVel carState
        , cAccel          = csAcc carState
        , cDesiredSpeed   = ccDesiredSpeed carConfig
        , cMinGap         = ccMinGap carConfig
        , cDesiredTimeGap = ccDesiredTimeGap carConfig
        , cMaxDecel      = ccMaxDecel carConfig
        , cMaxAccel       = ccMaxAccel carConfig
        }

-- Allow sorting cars by location initially
instance Ord Car where
  compare a b = compare (cPos a) (cPos b)

instance Eq Car where
    a == b = cPos a == cPos b

moveCarStateForward :: Float -> CarState -> CarState
moveCarStateForward dist carstate = carstate {csPos = forwardPos}
 where 
  forwardPos = (csPos carstate) + dist

carWithinBound ::  Float -> Car -> Bool
carWithinBound bound car = (cPos car) < bound

updateCarAccel :: Car -> Car -> Car
updateCarAccel car next = car {cAccel = calcAccel curV deltaV curGap car}
  where 
    curV = cVel car
    deltaV = curV - (cVel next)
    curGap = (cPos next) - (cPos car)

-- Calculate the actual acceleration chosen by the driver based on current speed, gap, delta-V, and characteristics
calcAccel :: Float -> Float -> Float -> Car -> Float 
calcAccel curV deltaV curGap car = maxAccel * (freeDesAcc - breakingAccel)
 where
  maxAccel = cMaxAccel car 
  freeDesAcc = calcFreeAccel curV car
  breakingAccel = calcBreakingAccel curV deltaV curGap car

-- Calculate the 'free desired' acceleration; fraction of max the driver would accelerate
-- on an unobstructed road based on current speed and desired speed
calcFreeAccel :: Float -> Car -> Float
calcFreeAccel curV car = 1 - (curV/desV) ^ (4 :: Int)
 where
  desV = cDesiredSpeed car


-- Calculate the 'breaking' acceleration; fraction of max acceleration driver doesn't 
-- use because they are getting too close to the car in front
calcBreakingAccel :: Float -> Float -> Float -> Car -> Float
calcBreakingAccel curV deltaV curGap car = (dynamicDesiredGap / curGap) ^ (2 :: Int)
 where
  dynamicDesiredGap =  (cMinGap car) + (max 0 (ownSpeedFac + relSpeedFac))
  ownSpeedFac = curV * (cDesiredTimeGap car)
  relSpeedFac = (curV * deltaV) / ((2) * (sqrt (maxAccel * maxDecel)))
  maxAccel = cMaxAccel car
  maxDecel = cMaxDecel car

updateCarPosVel :: Float -> Car -> Car
updateCarPosVel deltaT car = car {cPos = (calcPos deltaT car), cVel = (calcVel deltaT car)}

calcPos :: Float -> Car -> Float
calcPos deltaT car = x + v*deltaT + (0.5)*(a)*(deltaT^(2 :: Int))
  where
    x = cPos car
    v = cVel car
    a = cAccel car

calcVel :: Float -> Car -> Float
calcVel deltaT car = max 0 (v + a*deltaT)   
 where
  v = cVel car  
  a = cAccel car
