import cv2
import re
import numpy as np
from PIL import Image, ImageDraw
import random
import sys

def parse_config_line(line):
    bound_str = re.findall(r'scBound = (\d+.\d+)', line)
    lanes_str = re.findall(r'scLanes = (\d+)', line)
    iters_str = re.findall(r'Steps = (\d+)', line)
    print(bound_str, lanes_str, iters_str)
    return int(iters_str[0]), int(lanes_str[0]), float(bound_str[0])

def parse_lane_line(line):
    """ Extracts car positions from a scene line. """
    lane_id = int(re.findall(r'LaneID = (\d+)',line)[0])
    step = int(re.findall(r'Step = (\d+)',line)[0])
    car_ids = re.findall(r'cID = (\d+)', line)
    car_positions = re.findall(r'cPos = (\d+.\d+)', line)
    return (lane_id, step, [(int(car_ids[i]), float(car_positions[i])) for i in range(len(car_positions))])

CAR_LEN = 15
CAR_HEIGHT = 10
LANE_WIDTH = CAR_HEIGHT + 2

def draw_lane_line(lane_id, car_list, car_colors, width=1920, height=1080):
    """ Draws a scene with cars at specified positions with random colors. """
    image = Image.new('RGB', (width, height), color='gray')  # Road background
    draw = ImageDraw.Draw(image)
    
    ypos = LANE_WIDTH*lane_id

    for id, pos in car_list:        
        # Adjust position and size as needed
        upper_left = (pos, ypos)
        lower_right = (pos + CAR_LEN, ypos + CAR_HEIGHT)
        draw.rectangle([upper_left, lower_right], fill=car_colors[id])  # Draw car with random color

    return image
# pos / bound = x / frame_width
# x = (pos / bound) * frame_width
def convert_pos(pos, bound, frame_width):
    return int((pos/bound) * frame_width)

def all_cars_out_of_frame(car_list, frame_width):
    """ Check if all cars are out of the frame. """
    return all(pos > frame_width for _, pos in car_list)

CAR_LEN = 15
CAR_HEIGHT = 10
LANE_WIDTH = CAR_HEIGHT + 2

IMG_WIDTH = 1920
IMG_HEIGHT = 1080

def create_video_from_scenes(scene_file, output_video, dt=0.06, frame_rate=30, frame_width=1920):
    """ Generates a video from the scene descriptions. Stops when all cars are out of frame. """
    # Open the scene file and read lines
    with open(scene_file, 'r') as file:
        config = file.readline()
        lines = file.readlines()
    
    iters, num_lanes, bound = parse_config_line(config)

    # Video Writer setup
    fourcc = cv2.VideoWriter_fourcc(*'mp4v')
    video = cv2.VideoWriter(output_video, fourcc, frame_rate, (frame_width, 1080))

    car_colors = {}
    line_index = 0

    for scene_num in range(0, iters):

        image = Image.new('RGB', (IMG_WIDTH, IMG_HEIGHT), color='gray')  # Road background
        draw = ImageDraw.Draw(image)

        for lane_num in range(0, num_lanes + 1):
            line = lines[line_index]
            line_index = line_index + 1

            lane_id, step, raw_cars_list = parse_lane_line(line)

            assert(lane_id == lane_num)
            assert(step == scene_num)

            car_list = [(id, convert_pos(pos, bound, frame_width)) for id, pos in raw_cars_list]
            
            ypos = (IMG_HEIGHT // 2) + LANE_WIDTH*lane_id

            for id, pos in car_list:
                car_id = (lane_id, id)
                if (car_colors.get(car_id) is None):
                    car_colors[car_id] = (random.randint(0, 255), random.randint(0, 255), random.randint(0, 255))

                # Adjust position and size as needed
                upper_left = (pos, ypos)
                lower_right = (pos + CAR_LEN, ypos + CAR_HEIGHT)
                draw.rectangle([upper_left, lower_right], fill=car_colors[car_id])  # Draw car with random color

        frame = cv2.cvtColor(np.array(image), cv2.COLOR_RGB2BGR)
        # Write the frame to the video for 'dt' seconds
        for _ in range(int(frame_rate * dt)):
            video.write(frame)

    video.release()

def main():
    if len(sys.argv) != 3:
        print("Usage: python3 render.py input-file output-file")
        sys.exit(1)

    input_file = sys.argv[1]
    output_file = sys.argv[2]

    create_video_from_scenes(input_file, output_file)

if __name__ == "__main__":
    main()
