module Main (main) where

import Config (parseConfigFile, 
              SimConfig)

import Sim (runSimLoop, simResultToByteString, SimArgs(..))

import qualified Data.ByteString as B
import qualified Data.ByteString.Char8 as BSC

import System.Exit (exitFailure)
import Control.Exception (catch, IOException)
import Options.Applicative (Parser, strOption, long, short, metavar, help, 
                            option, auto, showDefault, value, execParser,
                            info, (<**>), helper, fullDesc, progDesc, header)

data TrafficSimArgs = TrafficSimArgs
  { configFile :: String,
    outFile :: String,
    numIterations :: Int,
    tsNoDisplay :: Bool
  } deriving Show

arguments :: Parser TrafficSimArgs  
arguments = TrafficSimArgs
  <$> strOption
      ( long "config"
     <> short 'c'
     <> metavar "FILENAME"
     <> help "Path to the configuration file"
      )
  <*> strOption
      ( long "outputfile"
     <> short 'o'
     <> metavar "FILENAME"
     <> help "Path to place output file" 
      )
  <*> option auto
      ( long "iterations"
     <> short 'n'
     <> metavar "INT"
     <> help "Number of iterations to run the simulator"
     <> showDefault
     <> value 10 -- Default to 10 iterations 
      )
  <*> option auto
    (
      long "no-output"
      <> short 'x'
      <> metavar "BOOL"
      <> help "Don't save output (for benchmarking)"
      <> showDefault
      <> value False
    )

main :: IO ()
main = runTrafficSim =<< execParser opts
  where
    opts = info (arguments <**> helper)
                 ( fullDesc
                 <> progDesc "Traffic Simulator Application"
                 <> header "Traffic Simulator - a tool for simulating traffic" )


runTrafficSim :: TrafficSimArgs -> IO ()
runTrafficSim args
        | numIterations args <= 0 = do
            putStrLn "Error: Number of iterations must be a positive integer."
            exitFailure
        | otherwise = do
            simConfig <- parseConfigFile (configFile args) `catch` handleConfigParseError
            
            let simArgs = SimArgs {noDisplay = (tsNoDisplay args),
                                   numIters = (numIterations args)}
            
            let simResult = runSimLoop simArgs simConfig
            let simResultBytes = simResultToByteString simResult
            let numStepBytes = BSC.pack ("{Steps = " ++ (show (numIterations args)) ++ "}")

            B.writeFile (outFile args) $ numStepBytes `BSC.append` simResultBytes

handleConfigParseError :: IOException -> IO SimConfig
handleConfigParseError e = do
  putStrLn $ "Error parsing config file: " ++ show e
  exitFailure  





 