import System.Exit(die)
import System.Environment(getArgs, getProgName)
import System.IO
import Data.Char(isAlpha, isSpace, toLower)
import Data.Map as Map
import Control.DeepSeq (deepseq)

-- removes non-alphabetic characters and converts to lower case
cleanAndSplit :: String -> [String]
cleanAndSplit s = words $ Prelude.map toLower $ Prelude.filter (\x -> isAlpha x || isSpace x) s

-- Map stage of MapReduce
doMap :: [String] -> [(String, Int)]
doMap xs = [(w,1) | w <- xs]

-- Reduce stage of MapReduce
doReduce :: [(String, Int)] -> Map String Int
doReduce xs = Map.fromListWith (+) xs

-- Find most similar word to search word
getClosestWord :: Map String Int -> String -> (String, Int)
getClosestWord wordFreq target = (closestWord, resultInt $ Map.lookup closestWord wordFreq)
    where 
    closestWord = getClosestWord' calcLevenshteinDist (Map.keys wordFreq) target

-- getClosestWord helper function
getClosestWord' :: (String -> String -> (Int, String)) -> [String] -> String -> String
getClosestWord' distFunction listWords target' = snd $ Map.findMin $ Map.fromList $ Prelude.map (distFunction target') listWords

-- Calculate the Levenshtein distance between two strings
calcLevenshteinDist :: String -> String -> (Int, String)
calcLevenshteinDist w1 w2 = (last $ Prelude.foldl transform [0..length w1] w2, w2)
  where
    transform [] _ = []
    transform xs@(x:xs') c = scanl compute (x + 1) (zip3 w1 xs xs')
      where
        compute z (c', x', y) = minimum [y + 1, z + 1, x' + fromEnum (c /= c')]

-- Helper function to convert a Maybe into an Int
resultInt :: Maybe Int -> Int
resultInt r = case r of
    Just x  -> x
    Nothing -> -1

-- Repeatedly get search word from user
getUserInput :: Map String Int -> IO ()
getUserInput wordFreq = do
  putStr "Enter a search word (or 'exit' to quit): "
  hFlush stdout
  target <- getLine

  if target == "exit"
    then putStrLn "Exiting..."
    else do
      putStrLn ("  You entered: \"" ++ target ++ "\"")
      let value = resultInt $ Map.lookup target wordFreq
      if value /= -1
        then putStrLn $ "  count: " ++ show value
        else do
          let (closest, count) = getClosestWord wordFreq target
          putStrLn $ "  search word not found:"
          putStrLn $ "  closest word: \"" ++ closest ++ "\"\n  count: " ++ (show count)
      getUserInput wordFreq


main :: IO ()
main = do
  args <- getArgs
  case args of
    [filename] -> do
      content <- readFile filename
      putStrLn $ "Starting MapReduce word counting..."
      let wordFreq = doReduce $ doMap $ cleanAndSplit content
      wordFreq `deepseq` return ()    -- force computation
      putStrLn $ "MapReduce completed..."
      getUserInput $ wordFreq
    _ -> do 
      pn <- getProgName
      die $ "Usage: " ++ pn ++ " <filename>"