module Biome where

import Codec.Picture
import System.Random (newStdGen, randomRs, split)
import Data.List (minimumBy)
import Data.Ord (comparing)
import Control.Parallel.Strategies (parMap, rdeepseq, rpar, rseq, using, parList)

-- Define a biome and a seed data type
data Biome = Forest | Desert | Ocean deriving (Show, Eq, Enum, Bounded)
data Seed = Seed { seedX :: Int, seedY :: Int, seedBiome :: Biome } deriving (Show, Eq)

-- Generate a random list of seeds
generateSeeds :: Int -> Int -> Int -> IO [Seed]
generateSeeds numSeeds width height = do
  gen <- newStdGen
  let biomes = [minBound .. maxBound] :: [Biome]
      biomeGen = randomRs (0, length biomes - 1) gen
      (xGen, yGen) = split gen
      xs = take numSeeds $ randomRs (0, width - 1) xGen
      ys = take numSeeds $ randomRs (0, height - 1) yGen
      biomeIndexes = take numSeeds biomeGen
  return [Seed x y (biomes !! index) | (x, y, index) <- zip3 xs ys biomeIndexes]

-- Euclidean distance function
distance :: Seed -> Int -> Int -> Double
distance seed x y = sqrt $ fromIntegral ((x - seedX seed) ^ 2 + (y - seedY seed) ^ 2)

-- Function to render the Voronoi diagram
renderVoronoiSeq :: Int -> Int -> [Seed] -> Image PixelRGB8
renderVoronoiSeq width height seeds =
  generateImage pixelRenderer width height
  where
    pixelRenderer x y = biomeToColor $ seedBiome $ closestSeed x y
    closestSeed x y = minimumBy (comparing (\seed -> distance seed x y)) seeds
    biomeToColor biome = case biome of
      Forest -> PixelRGB8 0 255 0     -- Green
      Desert -> PixelRGB8 255 255 0   -- Yellow
      Ocean -> PixelRGB8 0 0 255      -- Blue


renderVoronoiPar :: Int -> Int -> [Seed] -> Image PixelRGB8
renderVoronoiPar width height seeds =
  generateImageParallel pixelRenderer width height
  where
    pixelRenderer x y = biomeToColor $ seedBiome $ closestSeed x y
    closestSeed x y = minimumBy (comparing (\seed -> distance seed x y)) seeds
    biomeToColor biome = case biome of
      Forest -> PixelRGB8 0 255 0     -- Green
      Desert -> PixelRGB8 255 255 0   -- Yellow
      Ocean -> PixelRGB8 0 0 255      -- Blue

generateImageParallel :: (Int -> Int -> PixelRGB8) -> Int -> Int -> Image PixelRGB8
generateImageParallel pixelFunc width height =
  let rows = [ [ pixelFunc x y | x <- [0 .. width - 1] ] | y <- [0 .. height - 1] ]
      parallelRows = rows `using` parList rseq
  in generateImage (\x y -> (parallelRows !! y) !! x) width height


-- generateImageParallel :: Int -> Int -> (Int -> Int -> PixelRGB8) -> Image PixelRGB8
-- generateImageParallel width height pixelFunc =
--   let rows = [ [ pixelFunc x y | x <- [0 .. width - 1] ] | y <- [0 .. height - 1] ]
--       parallelRows = parMap rdeepseq id rows
--   in generateImage (\x y -> (parallelRows !! y) !! x) width height