
{-
A good chunk of the code, mainly just the the actual board structure and game play, is taken and adapted from:
https://github.com/IAmSam42/CS312-Checkers

We have  clearly denoted our code and places taken from the source code.
The bot is our contribution, as is any parallelization.
-}
import Control.Parallel.Strategies
import Data.List (maximumBy)
import Data.Maybe (isNothing)

---  Slightly modified data structures from source code to be Black and Red
data CPiece = Red | RedKing | Black | BlackKing deriving (Eq, Show)
type CBoard = [[Maybe CPiece]]
data CMove = Move (Int,Int) (Int, Int) | Take (Int,Int) (Int, Int) deriving (Eq, Show)

{-
Not in source code:

Functions:
pieceWeight, positionWeight, getHeuristicValueRed, evaluateBoardRed,
getHeuristicValueBlack, evaluateBoardBlack
-}
pieceWeight :: CPiece -> Int
pieceWeight Red = 1
pieceWeight RedKing = 10
pieceWeight Black = 1
pieceWeight BlackKing = 10

positionWeight :: (Int, Int) -> Int
positionWeight (_, y) = y 

getHeuristicValueRed :: CBoard -> Int
getHeuristicValueRed board = evaluateBoardRed board

evaluateBoardRed :: CBoard -> Int
evaluateBoardRed board = redScore - blackScore + captureBonus
  where
    redScore = sum [pieceWeight p * positionWeight (x, y)| x <- [1..8],
     y <- [1..8], Just p <- [getPiece board (x, y)], isRedPiece (Just p)]
    blackScore = sum [pieceWeight p * positionWeight (x, y) | x <- [1..8], 
     y <- [1..8], Just p <- [getPiece board (x, y)], isBlackPiece (Just p)]
    captureBonus = sum [captureValue | possibleMove <- allPossibleMoves board, isCaptureMove possibleMove]

    captureValue = 1000

    isCaptureMove :: CMove -> Bool
    isCaptureMove (Take _ _) = True
    isCaptureMove _ = False

    allPossibleMoves :: CBoard -> [CMove]
    allPossibleMoves b = concatMap (\pos -> listMoves b pos) allPositions

    allPositions = [(x, y) | x <- [1..8], y <- [1..8]]


getHeuristicValueBlack :: CBoard -> Int
getHeuristicValueBlack board = evaluateBoardBlack board

evaluateBoardBlack :: CBoard -> Int
evaluateBoardBlack board = blackScore - redScore + captureBonus
  where
    redScore = sum [pieceWeight p *  positionWeight (x, y)  | 
        x <- [1..8], y <- [1..8], Just p <- [getPiece board (x, y)], isRedPiece (Just p)]
    blackScore = sum [pieceWeight p *  positionWeight (x, y)  | 
        x <- [1..8], y <- [1..8], Just p <- [getPiece board (x, y)], isBlackPiece (Just p)]
    captureBonus = sum [captureValue | possibleMove <- allPossibleMoves board, isCaptureMove possibleMove]

    captureValue = 1000

    isCaptureMove :: CMove -> Bool
    isCaptureMove (Take _ _) = True
    isCaptureMove _ = False

    allPossibleMoves :: CBoard -> [CMove]
    allPossibleMoves b = concatMap (\pos -> listMoves b pos) allPositions
    allPositions = [(x, y) | x <- [1..8], y <- [1..8]]


{- 
From source code, slight modifications to Red and Black and warning fixes

Functions:
makeEmptyBoard, makeStandardBoard, move, movePiece, listMoves, getPiece
updatePiece,setPiece, width, height, allMoves, allTakes, printBoard
-}

makeEmptyBoard :: Int -> Int -> CBoard
makeEmptyBoard w h = [[Nothing | _ <- [1..w]] | _ <- [1..h]]

makeStandardBoard :: CBoard
makeStandardBoard = board where 
    board = foldl (\x y -> setPiece x (Just Black) y) w_board b_positions
    w_board = foldl (\x y -> setPiece x (Just Red) y) initial w_positions
    b_positions = [(x,y) | x <- [1..8], y <- [1,2,3], (x `mod` 2)/=(y `mod` 2)]
    w_positions = [(x,y) | x <- [1..8], y <- [6,7,8], (x `mod` 2)/=(y `mod` 2)]
    initial = makeEmptyBoard 8 8


movePiece :: CBoard -> CMove -> CBoard
movePiece b (Move (x1,y1) (x2,y2)) = setPiece b' p (x2,y2) where
    p = updatePiece b (x1, y1) (x2,y2)
    b' = setPiece b (Nothing) (x1,y1)
movePiece b (Take (x1,y1) (x2,y2)) = setPiece b'' p (x2,y2) where
    p = updatePiece b (x1, y1) (x2, y2)
    b'' = setPiece b' (Nothing) ((quot (x1+x2) 2), (quot (y1+y2) 2))
    b' = setPiece b (Nothing) (x1,y1)


listMoves :: CBoard -> (Int,Int) -> [CMove]
listMoves b (x, y) = case getPiece b (x,y) of
    Just Black -> [Move(x,y) (x2,y2)|(x2,y2)<-allMoves b (x,y), y2>y]
        ++ [Take(x,y)(x2,y2)|(x2, y2)<-allTakes b (Just Black) (x, y)]
    Just Red -> [Move(x,y) (x2,y2)|(x2,y2)<-allMoves b (x,y), y2<y]
        ++ [Take(x,y)(x2,y2)|(x2, y2)<-allTakes b (Just Red) (x, y)]
    Just BlackKing -> [Move(x,y)(x2,y2)|(x2,y2)<-allMoves b (x,y)] ++
        [Take(x,y)(x2,y2)|(x2, y2)<-allTakes b (Just BlackKing) (x, y)]
    Just RedKing -> [Move(x,y)(x2,y2)|(x2,y2)<-allMoves b (x,y)] ++
        [Take(x,y)(x2,y2)|(x2, y2)<-allTakes b (Just RedKing) (x, y)]
    Nothing -> []

getPiece :: CBoard -> (Int,Int) -> Maybe CPiece
getPiece board (x,y) = case (getElem board y) of 
    Nothing -> Nothing
    Just ls -> case (getElem ls x) of
        Nothing -> Nothing
        Just a  -> a


updatePiece :: CBoard -> (Int,Int) -> (Int,Int) -> Maybe CPiece
updatePiece b (x1,y1) (_,y2) = case getPiece b (x1, y1) of
    Just Black -> if y2 == height b then Just BlackKing else Just Black
    Just Red -> if y2 == 1 then Just RedKing else Just Red
    a -> a

setPiece :: CBoard -> Maybe CPiece -> (Int,Int) -> CBoard
setPiece [] _ (_,_) = []
setPiece ((_:l):ls) p (1,1) = (p : l) : ls
setPiece ((x:l):ls) p (w,1) = case (setPiece (l:ls) p ((w-1),1)) of
    (l2:ls2) -> (x:l2):ls2
    []       -> [[x]]
setPiece (l:ls) p (w,h)     = l : (setPiece ls p (w,(h-1)))

width :: CBoard -> Int
width []    = 0
width (h:_) = length h

height :: CBoard -> Int
height = length

getElem :: [a] -> Int -> Maybe a
getElem [] _ = Nothing
getElem (h:_) 1 = Just h
getElem (_:t) n = getElem t (n-1)

allMoves :: CBoard -> (Int, Int) -> [(Int, Int)]
allMoves b (x, y) = [p | p <- onBoard, (getPiece b p) == Nothing] 
    where
        onBoard = [(x', y') | (x', y') <- spaces, x' < width b, y' < height b, x' > 0, y' > 0]
        spaces = [(x + 1, y + 1), (x - 1, y + 1), (x + 1, y - 1), (x - 1, y - 1)]

    
allTakes :: CBoard -> Maybe CPiece -> (Int, Int) -> [(Int, Int)]
allTakes _ Nothing _ = []
allTakes b (Just Black) (px, py) = let
    onBoard = [(nx, ny) | (nx, ny) <- spaces, nx < width b, ny < height b, nx > 0, ny > 0]
    spaces = [(px + 2, py + 2), (px - 2, py + 2)]
    free = filter (\(nx, ny) -> isNothing (getPiece b (nx, ny))) onBoard
    in [(nx, ny) | (nx, ny) <- free,
        let enemyPos = ((px + nx) `div` 2, (py + ny) `div` 2),
        isRedPiece (getPiece b enemyPos)]
allTakes b (Just BlackKing) (px, py) = let
    onBoard = [(nx, ny) | (nx, ny) <- spaces, nx < width b, ny < height b, nx > 0, ny > 0]
    spaces = [(px + 2, py + 2), (px - 2, py + 2), (px + 2, py - 2), (px - 2, py - 2)]
    free = filter (\(nx, ny) -> isNothing (getPiece b (nx, ny))) onBoard
    in [(nx, ny) | (nx, ny) <- free,
        let enemyPos = ((px + nx) `div` 2, (py + ny) `div` 2),
        isRedPiece (getPiece b enemyPos)]
allTakes b (Just Red) (px, py) = let
    onBoard = [(nx, ny) | (nx, ny) <- spaces, nx < width b, ny < height b, nx > 0, ny > 0]
    spaces = [(px + 2, py - 2), (px - 2, py - 2)]
    free = filter (\(nx, ny) -> isNothing (getPiece b (nx, ny))) onBoard
    in [(nx, ny) | (nx, ny) <- free,
        let enemyPos = ((px + nx) `div` 2, (py + ny) `div` 2),
        isBlackPiece (getPiece b enemyPos)]
allTakes b (Just RedKing) (px, py) = let
    onBoard = [(nx, ny) | (nx, ny) <- spaces, nx < width b, ny < height b, nx > 0, ny > 0]
    spaces = [(px + 2, py + 2), (px - 2, py + 2), (px + 2, py - 2), (px - 2, py - 2)]
    free = filter (\(nx, ny) -> isNothing (getPiece b (nx, ny))) onBoard
    in [(nx, ny) | (nx, ny) <- free,
        let enemyPos = ((px + nx) `div` 2, (py + ny) `div` 2),
        isBlackPiece (getPiece b enemyPos)]


printBoard :: CBoard -> IO ()
printBoard b = putStr (showBoard b)

--- Modified from source code to be a better representation of the board
showBoard :: CBoard -> String
showBoard board = indexedBoard
  where
    indexedBoard = unlines $ zipWith (\i row -> pad i ++ " | " 
        ++ concatMap showPieceWithSpace row ++ "|") [1..] board

-- Not in source code
pad :: Int -> String
pad n
  | n < 10 = " " ++ show n
  | otherwise = show n

showPieceWithSpace :: Maybe CPiece -> String
showPieceWithSpace piece = showPiece piece ++ " "


-- Modified from source code
showPiece :: Maybe CPiece -> String
showPiece (Just Red) = "r"
showPiece (Just RedKing) = "R"
showPiece (Just Black) = "b"
showPiece (Just BlackKing) = "B"
showPiece (Nothing) = "-"
gameStart :: IO ()
gameStart = do
   putStrLn "+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"
   putStrLn "Welcome to our CheckerBot!"
   putStrLn "+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"
   putStrLn "If your valid moves are listed as [], you have tried to move a piece you cannot."
   putStrLn "In that situation, please just use that same input when prompted for where you"
   putStrLn  "want to move the piece."
   putStrLn "+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"
   putStrLn "Game Begins: Red Starts"
   printBoard makeStandardBoard


{-
Not in source code 

Functions:
getAllBlackPieces, getAllRedPieces, isBlackPiece, isRedPiece, playGame, areMovesAvailable, 
minimaxRed,  maximizeRed, minimizeRed, minimaxBlack, maximizeBlack, minimizeBlack,
endGameScore, performRedMove, performBlackMove, isGameOver
-}
getAllBlackPieces :: CBoard -> [(Int, Int)]
getAllBlackPieces board = [(x, y) | x <- [1..8], y <- [1..8], isBlackPiece (getPiece board (x, y))]


getAllRedPieces :: CBoard -> [(Int, Int)]
getAllRedPieces board = [(x, y) | x <- [1..8], y <- [1..8], isRedPiece (getPiece board (x, y))]


isBlackPiece :: Maybe CPiece -> Bool
isBlackPiece (Just Black) = True
isBlackPiece (Just BlackKing) = True
isBlackPiece _ = False


isRedPiece :: Maybe CPiece -> Bool
isRedPiece (Just Red) = True
isRedPiece (Just RedKing) = True
isRedPiece _ = False


playGame :: CBoard -> IO ()
playGame board = do
    putStrLn "Red's Turn:"
    redMovesAvailable <- areMovesAvailable board Red
    if not redMovesAvailable
        then putStrLn "Game Over! Red has no moves left. Black wins!" >> return ()
        else do
            redBoard <- performRedMove board
            printBoard redBoard
            blackMovesAvailable <- areMovesAvailable redBoard Black
            if not blackMovesAvailable
                then putStrLn "Game Over! Black has no moves left. Red wins!" >> return ()
                else do
                    putStrLn "Black's Turn:"
                    blackBoard <- performBlackMove redBoard
                    printBoard blackBoard
                    playGame blackBoard

areMovesAvailable :: CBoard -> CPiece -> IO Bool
areMovesAvailable board player = 
    return . not . null $ concatMap (\pos -> listMoves board pos) positions
    where
        positions = case player of
            Red -> getAllRedPieces board
            Black -> getAllBlackPieces board
            _ -> []  
        

minimaxRed :: CBoard -> Int -> CMove
minimaxRed board depth = bestMove where
    (_, bestMove) = maximizeRed depth board

maximizeRed :: Int -> CBoard -> (Int, CMove)
maximizeRed 0 board = (evaluateBoardRed board, Move (0, 0) (0, 0))
maximizeRed depth board
    | null redMoves = (endGameScore board, Move (0, 0) (0, 0))  
    | depth >= 4 = maximumBy (\(score1, _) (score2, _) -> compare score1 score2)
                  $ parMap rpar (\move -> (minimizeRed (depth - 1) (movePiece board move), move)) redMoves
    | otherwise = maximumBy (\(score1, _) (score2, _) -> compare score1 score2)
                  $ map (\move -> (minimizeRed (depth - 1) (movePiece board move), move)) redMoves
  where
    redMoves = concatMap (\(x, y) -> listMoves board (x, y)) $ getAllRedPieces board

minimizeRed :: Int -> CBoard -> Int
minimizeRed 0 board = evaluateBoardRed board
minimizeRed depth board
    | null blackMoves = endGameScore board
    | depth >= 4 = minimum $ parMap rpar (\move -> fst $ maximizeRed (depth - 1) (movePiece board move)) blackMoves
    | otherwise = minimum $ map (\move -> fst $ maximizeRed (depth - 1) (movePiece board move)) blackMoves
  where
    blackMoves = concatMap (\(x, y) -> listMoves board (x, y)) $ getAllBlackPieces board


minimaxBlack :: CBoard -> Int -> CMove
minimaxBlack board depth = bestMove where
    (_, bestMove) = maximizeBlack depth board

maximizeBlack :: Int -> CBoard -> (Int, CMove)
maximizeBlack 0 board = (evaluateBoardBlack board, Move (0, 0) (0, 0))
maximizeBlack depth board
    | null blackMoves = (endGameScore board, Move (0, 0) (0, 0)) 
    | depth >= 4  = maximumBy (\(score1, _) (score2, _) -> compare score1 score2)
                  $ parMap rpar (\move -> (minimizeBlack (depth - 1) (movePiece board move), move)) blackMoves
    | otherwise = maximumBy (\(score1, _) (score2, _) -> compare score1 score2)
                  $ map (\move -> (minimizeBlack (depth - 1) (movePiece board move), move)) blackMoves
  where
    blackMoves = concatMap (\(x, y) -> listMoves board (x, y)) $ getAllBlackPieces board

minimizeBlack :: Int -> CBoard -> Int
minimizeBlack 0 board = evaluateBoardBlack board
minimizeBlack depth board
    | depth >= 4 = minimum $ parMap rpar (\move -> fst $ maximizeBlack (depth - 1) (movePiece board move)) redMoves
    | otherwise = minimum $ map (\move -> fst $ maximizeBlack (depth - 1) (movePiece board move)) redMoves
  where
    redMoves = concatMap (\(x, y) -> listMoves board (x, y)) $ getAllRedPieces board


endGameScore :: CBoard -> Int
endGameScore board
    | isGameOver board && null (getAllRedPieces board) = 10000  -- black win case
    | isGameOver board && null (getAllBlackPieces board) = -10000  -- red win case
    | otherwise = evaluateBoardRed board  -- no winner yet


performRedMove :: CBoard -> IO CBoard
performRedMove board = do
    let redMove = minimaxRed board 5
    putStrLn $ "Red chose: " ++ show redMove
    let newBoard = movePiece board redMove
    putStrLn $ "Heuristic Value after Red's Move: " ++ show (getHeuristicValueRed newBoard)
    return newBoard

performBlackMove :: CBoard -> IO CBoard
performBlackMove board = do
    let blackMove = minimaxBlack board 5
    putStrLn $ "Black chose: " ++ show blackMove
    let newBoard = movePiece board blackMove
    putStrLn $ "Heuristic Value after Black's Move: " ++ show (getHeuristicValueBlack newBoard)
    return newBoard


isGameOver :: CBoard -> Bool
isGameOver board = null (getAllRedPieces board) || null (getAllBlackPieces board)

main :: IO()
main = do
   gameStart
   playGame makeStandardBoard
   putStrLn "End"
