#include <stdio.h>
#include <math.h>
#include "vga_ball.h"
#include "sift.h"
#include "sift_utils.h"
#include <stdlib.h>
#include <assert.h>
#include "keypoints.h"

void compute_DoG(Img *imgA, Img *imgB, DoG *dog) { // assumes imgA->array and imgB->array are within range [0,255]
    printf("Computing DoG for octave %d blur_idx %d\n", imgA->octave, imgA->blur_idx);
    int height = dog->height;
    int width = dog->width;

    int img_size = height * width;
    int heightA_diff = imgA->height - height;
    int heightB_diff = imgB->height - height;
    assert(heightA_diff > 0);
    assert(heightA_diff > heightB_diff);
    int widthA_diff = imgA->width - width;
    int widthB_diff = imgB->width - width;
    assert(widthA_diff > 0);
    assert(widthA_diff > widthB_diff);
    if (dog->img2->blur_idx == 4) assert(heightB_diff == 0 && widthB_diff == 0);
    assert(widthA_diff % 2 == 0 && widthB_diff % 2 == 0);
    assert(heightA_diff % 2 == 0 && heightB_diff % 2 == 0);
    
    int idx_startA = imgA->width * (heightA_diff / 2) + (widthA_diff / 2); // top left pix of cropped A
    int idx_startB = imgB->width * (heightB_diff / 2) + (widthB_diff / 2); // top left pix of cropped B
     
    int idxA = idx_startA;
    int idxB = idx_startB;
    for (int i=0; i<img_size; i++) {
        // printf("height_diffA %d, height_diffB %d, widthA_diff %d, widthB_diff %d\n", heightA_diff, heightB_diff, widthA_diff, widthB_diff);
        // printf("idxStartA %d, idxStartb %d, idxA %d, idxB %d\n", idx_startA, idx_startB, idxA, idxB);
        int row_DoG = (i / width);
        dog->array[i] = ((double)(imgB->array[idxB] - imgA->array[idxA]) / 255);
        if (i % width == (width-1)) {
           idxA += widthA_diff;
           idxB += widthB_diff;
        }
        idxA++; 
        idxB++;
    }
    printf("Finished Computing DoG for octave %d blur_idx %d\n", imgA->octave, imgA->blur_idx);
}

void get_kernel(double *arr, double sigma) {
    int size = get_kernel_size_from_sigma(sigma);
    double constant = 1 / (2 * 3.14159265 * pow(sigma, 2));
    double arr_sum = 0;
    for (int r=0; r<size; r++) {
        for (int c=0; c<size; c++) {
            double val = constant * exp((pow(r - size/2, 2) + pow(c - size/2, 2)) / (2 * pow(sigma, 2)));
            arr[r * size + c] = val;
            arr_sum += val;
        }
    }
    for (int i=0; i<size*size; i++)
        arr[i] /= arr_sum;

}

int get_octave_im_dim(int orig_dim, int octave_num) {
	return (orig_dim >> octave_num) + check_bit(orig_dim, octave_num-1);
}

int check_bit(int num, int i) {
    if (i<0) return 0;
    int mask = 1 << i;
    return (num & mask) != 0;
}

int get_kernel_size_from_sigma(double sigma) {
    return 2 * ceil(3 * sigma) + 1;
}

void get_sigmas(double *arr, double root) {
    for (int i=0; i<NUM_BLURS; i++) {
        arr[i] = pow(2, ((i+1) / root));
    }
}

void mark_keypoints_on_imag(char *img_array, Keypoints *keypoints, int octave_num) {
    //unsigned char kps[IMAGE_SIZE] = {0};
    // clear data from the marker value 
    // --> this makes the black pixels slightly brighter
    if (octave_num == 0) {
	    for (int i=0; i < IMAGE_SIZE; i++) {
		if (img_array[i] == 0)
		    img_array[i]++;
	    }
    }
    // mark the pixels around the kp with the flag to show them as RED on the VGA board
    for (int i=0; i < keypoints->count; i++) {
        Kp *kp = keypoints->kp_list[i];
        if (!kp->is_valid)
          continue;
        if (kp->DoG_idx / (NUM_BLURS-1) != octave_num) continue;
        // int arr_index = (int) ((kp->y+12) * IMAGE_WIDTH + (kp->x+12));
        int arr_index = round(kp->kp_precise->real_y * IMAGE_WIDTH + kp->kp_precise->real_x);
        /* Do we want to do this for specific DoGs or just for all found kps? */
        // if (kp->DoG_idx == img->)
        int row = arr_index / IMAGE_WIDTH;
        int col = arr_index % IMAGE_WIDTH;

        img_array[arr_index] = 0;
        if (col < IMAGE_WIDTH-1) // not the last pixel on the line
            img_array[arr_index + 1] = 0;
        if (col > 0) // not the first pixel on the line
            img_array[arr_index - 1] = 0;

        // not on the last line of the image
        if (row < IMAGE_HEIGHT - 1) {
            img_array[arr_index + IMAGE_WIDTH] = 0;
            if (col > 0)
                img_array[arr_index + IMAGE_WIDTH - 1] = 0;
            if (col < IMAGE_WIDTH-1)
                img_array[arr_index + IMAGE_WIDTH + 1] = 0;
        }

        // not on the first line
        if (row > 1) {
            img_array[arr_index - IMAGE_WIDTH] = 0;
            if (col > 0)
                img_array[arr_index - IMAGE_WIDTH - 1] = 0;
            if (col < IMAGE_WIDTH-1)
                img_array[arr_index - IMAGE_WIDTH + 1] = 0;
        }
    }
}

void read_DoG_py(Img *imgA, Img *imgB, DoG *dog) {
    int img_size = dog->height * dog->width; // imgB size will be smaller than imgA, since imgA uses a smaller kernel
    
    for (int octave_num = 0; octave_num < NUM_OCTAVES; octave_num++) {
    	for (int blur_idx = 0; blur_idx < NUM_BLURS-1; blur_idx++) {
	    char file_name[100] = {0};
	    sprintf(file_name, "../../DoG_out_py/DoG_out_py_%d_%d", octave_num, blur_idx);
	    
	    FILE *file = fopen(file_name, "r");
	    if (file == NULL) {
		printf("Cannot open file \n");
		return;
	    }

	    int i = 0;
	    while (i < img_size && fscanf(file, "%lf\n", &dog->array[i]) == 1) {
		i++;
	    }
	    fclose(file);
	    }
    }
}
