/* * Device driver for the VGA video generator
 *
 * A Platform device implemented using the misc subsystem
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod vga_ball.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree vga_ball.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "vga_ball.h"

#define DRIVER_NAME "vga_ball"

/* Device registers */
#define BG_RED(x) (x)
#define BG_GREEN(x) ((x)+1)
#define BG_BLUE(x) ((x)+2)
#define notused(x) ((x)+3)

/*new device register */
#define fallingSprite(x) ((x)+4)
//#define back0(x) (x)
#define fallv(x) ((x)+87)
#define fallh(x) ((x)+91)
#define backN(x) ((x)+96)
#define nextS(x) ((x)+100)
//#define fallspi(x) ((x)+87)
//#define backnext(x) ((x)+88)
/*
#define cScore(x) ((x)+24)
#define hScore(x) ((x)+25)
#define musicSound(x) ((x)+26)
#define musicEN(x) ((x)+27)
#define fallingOri(x) ((x)+28)
*/

/*
 * Information about our device
 */
struct vga_ball_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
        vga_ball_color_t background;
	tetromino_location_t tetro_loco;
	tetromino_ori_t ori; 
	tetromino_choice_t choice;
	tetromino_choice_t next;
	tetromino_row_t row;
} dev;

/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
 */
static void write_background(vga_ball_color_t *background)
{
	iowrite8(background->red, BG_RED(dev.virtbase) );
	iowrite8(background->green, BG_GREEN(dev.virtbase) );
	iowrite8(background->blue, BG_BLUE(dev.virtbase) );
	iowrite8(0x00,BG_BLUE(dev.virtbase) + 1);
	dev.background = *background;

}

static void write_choice(tetromino_choice_t *choice){
	iowrite8(choice->tetro, fallingSprite(dev.virtbase)); //what gets sent first 
	iowrite8(0x00, fallingSprite(dev.virtbase)+ 1); //is in the register at bits [7:0]
	iowrite8(0x00, fallingSprite(dev.virtbase) + 2);
	iowrite8(0x00, fallingSprite(dev.virtbase)+ 3);
	//iowrite8(choice->tetro, fallingSprite(dev.virtbase) + 3);
	dev.choice = *choice;
}

static void write_location(tetromino_location_t *tetro_loco)
{
	iowrite32(tetro_loco->hor, fallh(dev.virtbase) );
	iowrite32(tetro_loco->vert, fallv(dev.virtbase) );
	dev.tetro_loco = *tetro_loco;
	//trying to write 16 now
}

static void write_row(tetromino_row_t *row){
	if (row->num == 19){
		iowrite32(row->contents, backN(dev.virtbase));
		dev.row = *row;
	}
	else{
		int offset = row->num * 4;
		iowrite32(row->contents, fallingSprite(dev.virtbase) + 4 + offset); //what gets sent first 
		dev.row = *row;
	}
}
static void write_next(tetromino_choice_t *next){
	iowrite8(next->tetro, nextS(dev.virtbase)); //what gets sent first 
	iowrite8(0x00, nextS(dev.virtbase)+ 1); //is in the register at bits [7:0]
	iowrite8(0x00, nextS(dev.virtbase) + 2);
	iowrite8(0x00, nextS(dev.virtbase)+ 3);
	//iowrite8(choice->tetro, fallingSprite(dev.virtbase) + 3);
	dev.next = *next;
}


/*static void write_orientation(tetromino_ori_t *ori)
{
	iowrite32(ori->ori, fallingOri(dev.virtbase) );
        dev.ori = *ori;	
}
*/
/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long vga_ball_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	vga_ball_arg_t vla;

	switch (cmd) {
	case VGA_BALL_WRITE_BACKGROUND:
		if (copy_from_user(&vla, (vga_ball_arg_t *) arg,
				   sizeof(vga_ball_arg_t)))
			return -EACCES;
		//printk(KERN_INFO "background write location color pointer: %lu\n", &vla.background);
		write_background(&vla.background);
		break;

	case VGA_BALL_READ_BACKGROUND:
	  	vla.background = dev.background;
		if (copy_to_user((vga_ball_arg_t *) arg, &vla,
				 sizeof(vga_ball_arg_t)))
			return -EACCES;
		break;

	case TETROMINO_WRITE_LOCATION:
		if (copy_from_user(&vla, (vga_ball_arg_t *) arg,
				   sizeof(vga_ball_arg_t)))
			return -EACCES;
		//printk(KERN_INFO "Tetrimono Write Location pointer: %lu\n", &vla.tetro_loco);
		write_location(&vla.tetro_loco);
		break;

	case TETROMINO_WRITE_NEXT:
		if (copy_from_user(&vla, (vga_ball_arg_t *) arg,
				   sizeof(vga_ball_arg_t)))
			return -EACCES;
		write_next(&vla.next);
		break;
	

	/*case TETROMINO_WRITE_ORIENTATION:
		if (copy_from_user(&vla, (vga_ball_arg_t *) arg,
				   sizeof(vga_ball_arg_t)))
			return -EACCES;
		write_orientation(&vla.ori); 
		break;
		*/

	case TETROMINO_WRITE_CHOICE:
		if (copy_from_user(&vla, (vga_ball_arg_t *) arg,
				   sizeof(vga_ball_arg_t)))
			return -EACCES;
		write_choice(&vla.choice); 
		break;


	case TETROMINO_WRITE_ROW:
		if (copy_from_user(&vla, (vga_ball_arg_t *) arg,
				   sizeof(vga_ball_arg_t)))
			return -EACCES;
		write_row(&vla.row); 
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations vga_ball_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = vga_ball_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice vga_ball_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &vga_ball_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init vga_ball_probe(struct platform_device *pdev)
{
        vga_ball_color_t beige = { 0xf9, 0xe4, 0xb7 };
	int ret;

	/* Register ourselves as a misc device: creates /dev/vga_ball */
	ret = misc_register(&vga_ball_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
        
	/* Set an initial color */
        write_background(&beige);

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&vga_ball_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int vga_ball_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&vga_ball_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id vga_ball_of_match[] = {
	{ .compatible = "csee4840,vga_ball-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, vga_ball_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver vga_ball_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(vga_ball_of_match),
	},
	.remove	= __exit_p(vga_ball_remove),
};

/* Called when the module is loaded: set things up */
static int __init vga_ball_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&vga_ball_driver, vga_ball_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit vga_ball_exit(void)
{
	platform_driver_unregister(&vga_ball_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(vga_ball_init);
module_exit(vga_ball_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("VGA ball driver");
