// compile w/: gcc -o main tetris.c -Itetris.h
#include <stdio.h>
#include <unistd.h>
#include <pthread.h>
#include <stdlib.h>
#include <stdint.h>
#include <time.h>
#include <string.h>
#include <stdbool.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "usbgamepad.h"
#include "tetris.h"
#include "vga_ball.h"

#define L_BORDER 				(0xdd99)		// 56729
#define T_BORDER 				(0x4721)		// 18209
#define PX_PER_BLOCK			(5125)          // 4739
#define TETROMINO_START_POS		{L_BORDER, T_BORDER}

static const char filename[] = "/dev/vga_ball";
static int vga_ball_fd;
static pthread_t screen_thread;
static void *screen_thread_f(void *);
static int score = 0;
volatile enum gamepad_state gp_state = NONE;
bool key_released = true;
static tetromino_location_t hw_current_tetro = TETROMINO_START_POS;
static pthread_mutex_t current_tetrimino_lock;

static void _debug_printgameboard();
static _debug_currenttetromino _debug_createtetro();
static int random_number(int min, int max);
static void _debug_addcurrenttetro(_debug_currenttetromino current_tetro);
static void _debug_removecurrenttetro(_debug_currenttetromino current_tetro);
static void _debug_movedowntetro(_debug_currenttetromino *current_tetro);
static void _debug_movelefttetro(_debug_currenttetromino *current_tetro);
static void _debug_moverighttetro(_debug_currenttetromino *current_tetro);
static bool _debug_canmovetetro(_debug_currenttetromino current_tetro, _debug_tetromino_moveopt moveopt);
static void _debug_rotatetetro(_debug_currenttetromino *current_tetro, _debug_tetromino_rotopt rotopt);
static bool _debug_canrotatetetro(_debug_currenttetromino current_tetro, _debug_tetromino_orientation ori);
static int _debug_firstrowcomplete();
static void _debug_shiftrowsdown(int row);

static void print_background_color();
static void set_tetromino_position(const tetromino_location_t *a, _debug_currenttetromino current_tetro);
static void set_tetromino_choice(const tetromino_choice_t *c);
static void set_tetromino_type_and_ori(_debug_currenttetromino current_tetro);
static void set_tetromino_row(const tetromino_row_t *r);
static void set_tetromino_to_background(_debug_currenttetromino current_tetro);
static void set_tetromino_clear_row(int row);
static void set_tetromino_clear_board();
static void set_tetromino_redraw_board();

int main() {
    struct libusb_device_handle *gamepad = NULL;
    uint8_t endpoint_address;

    // gamepad setup
    if ((gamepad = opengamepad(&endpoint_address)) == NULL) {
        fprintf(stderr, "Did not find a gamepad...\n");
        exit(1);
    }

    // random number generator setup
    srand(time(NULL));

    // start the screen thread
	pthread_create(&screen_thread, NULL, screen_thread_f, NULL);
    pthread_mutex_init(&current_tetrimino_lock, NULL);

    // game runtime, update gamepad state (remove sticky keys)
    for (;;) {
		gp_state = gamepad_state(gamepad, endpoint_address);
        switch (gp_state) {
			case LEFT:
			case RIGHT:
			case DOWN:
			case A: 
			case B: 
			case SELECT:
                if (key_released) {
					key_released = false;
				}
                break;
			case NONE:
				key_released = true;
				break;
			default:
				break;
		}
    }

    // thread cleanup
	pthread_cancel(screen_thread);
  	pthread_join(screen_thread, NULL);
    pthread_mutex_destroy(&current_tetrimino_lock);

    return 0;
}

void *screen_thread_f(void *ignored) {
    // setup
    _debug_currenttetromino current_tetro = _debug_createtetro();

    // open the vga ball fd to interact with
	if ((vga_ball_fd = open(filename, O_RDWR)) == -1) {
		fprintf(stderr, "Could not open %s\n", filename);
		return NULL;
	}

    // setup initial background state
	print_background_color();
    set_tetromino_clear_board();

    // set initial position of tetrinimo on screen
	set_tetromino_position(&hw_current_tetro, current_tetro);
    set_tetromino_type_and_ori(current_tetro);

    // loop
    for(;;) {
        // clear the board
        printf("\e[1;1H\e[2J");
        
        // save current action in temp state
        enum gamepad_state current_action = gp_state;
        key_released = true;

        // update current tetrimino and position on board based on given action
        _debug_removecurrenttetro(current_tetro);
        printf("score: %i | (%c)\n", score, current_action);
        switch (current_action) {
			case LEFT:
                _debug_movelefttetro(&current_tetro);
                set_tetromino_position(&hw_current_tetro, current_tetro);
                break;
			case RIGHT:
                _debug_moverighttetro(&current_tetro);
                set_tetromino_position(&hw_current_tetro, current_tetro);
                break;
			case DOWN:
                _debug_movedowntetro(&current_tetro);
                set_tetromino_position(&hw_current_tetro, current_tetro);
                break;
			case A:
                _debug_rotatetetro(&current_tetro, tetromino_rotopt_CLOCKWISE);
                set_tetromino_type_and_ori(current_tetro);
                set_tetromino_position(&hw_current_tetro, current_tetro);
                break;
			case B:
                _debug_rotatetetro(&current_tetro, tetromino_rotopt_COUNTERCLOCKWISE);
                set_tetromino_type_and_ori(current_tetro);
                set_tetromino_position(&hw_current_tetro, current_tetro);
                break;
            case START:
			case SELECT:
			case NONE:
			default:
				break;
		}

        // add additional downshift if needed
        _debug_movedowntetro(&current_tetro);
        set_tetromino_position(&hw_current_tetro, current_tetro);

        // check if you're at the top and cant move down anymore (game over)
        if (current_tetro.pos_y == 0 && !_debug_canmovetetro(current_tetro, tetromino_moveopt_DOWN)) {
            printf("\e[1;1H\e[2J");
            printf("Game Over\n");
            exit(EXIT_SUCCESS);
        }

        // check if any rows have to be completed, if so clear them and shift everything above down one row
        for (int fullrow = _debug_firstrowcomplete(); fullrow >= 0; fullrow = _debug_firstrowcomplete()) {
            _debug_shiftrowsdown(fullrow);
            set_tetromino_redraw_board();
            score++;
        }

        // check if bottom hit or another piece below, if so add block to background and 
        //  initialize a new current tetrimino
        if (_debug_canmovetetro(current_tetro, tetromino_moveopt_DOWN) == false) {
            _debug_addcurrenttetro(current_tetro);
            set_tetromino_to_background(current_tetro);
            current_tetro = _debug_createtetro();
        }

        // update screen
        _debug_addcurrenttetro(current_tetro);
        set_tetromino_type_and_ori(current_tetro);
        set_tetromino_position(&hw_current_tetro, current_tetro);

        // display gameboard
        _debug_printgameboard();
        usleep(500000);
    }
    
    // break
    return NULL;
}

// print the current board to the screen
static void _debug_printgameboard() {
    for (int i = 0; i < ROWS; i++) {
        for (int j = 0; j < COLS; j++) {
            printf("[%c]", _debug_currentboard[i][j]);
        }
        printf("\n");
    }
}

// create a randomly generated new tetromino
static _debug_currenttetromino _debug_createtetro() {
    _debug_tetromino_type ttype = random_number(0, 6);
    _debug_currenttetromino new_tetro = {
        ttype, 
        tetromino_orientation_0, 
        4, 0, 
        _debug_tetromino_block_offset[ttype][tetromino_orientation_0][0], 
        _debug_tetromino_block_offset[ttype][tetromino_orientation_0][1]};

    return new_tetro;
}

// random number within range generation abstraction
static int random_number(int min, int max) {
    return (rand() % (max - min + 1)) + min;
}

// add given tetromino to the gameboard
static void _debug_addcurrenttetro(_debug_currenttetromino current_tetro) {
    // loop through current tetro and add any "X" vals to its corresponding location on the board
    for (int i = 0; i < 4; i++) {
        for (int j = 0; j < 4; j++) {
            if (_debug_tetromino_matrix[current_tetro.tetromino_type][current_tetro.tetromino_orientation][i][j] != ' ') {
                // consider postition and offset and add to board
                _debug_currentboard[current_tetro.pos_y + i - current_tetro.offset_y][current_tetro.pos_x + j - current_tetro.offset_x] = _debug_type_to_color[current_tetro.tetromino_type];
            }
        }
    }
}

// remove a given tetromino from the gameboard
static void _debug_removecurrenttetro(_debug_currenttetromino current_tetro) {
    // loop through current tetro and add any "X" vals to its corresponding location on the board
    for (int i = 0; i < 4; i++) {
        for (int j = 0; j < 4; j++) {
            if (_debug_tetromino_matrix[current_tetro.tetromino_type][current_tetro.tetromino_orientation][i][j] != ' ') {
                // consider postition and offset and add to board
                _debug_currentboard[current_tetro.pos_y + i - current_tetro.offset_y][current_tetro.pos_x + j - current_tetro.offset_x] = ' ';
            }
        }
    }
}

// move the tetromino in the available directions
static void _debug_movedowntetro(_debug_currenttetromino *current_tetro) {
    if (_debug_canmovetetro(*current_tetro, tetromino_moveopt_DOWN))
        current_tetro->pos_y += 1;
}

static void _debug_movelefttetro(_debug_currenttetromino *current_tetro) {
    if (_debug_canmovetetro(*current_tetro, tetromino_moveopt_LEFT))
        current_tetro->pos_x -= 1;
}

static void _debug_moverighttetro(_debug_currenttetromino *current_tetro) {
    if (_debug_canmovetetro(*current_tetro, tetromino_moveopt_RIGHT))
        current_tetro->pos_x += 1;
}

// check if the tetromino CAN move in the available directions
static bool _debug_canmovetetro(_debug_currenttetromino current_tetro, _debug_tetromino_moveopt moveopt) {
    int coord[4][2];
    int coord_idx = 0;
    int move_x = 0;
    int move_y = 0;

    // set which direction you want to try and move
    switch (moveopt) {
        case tetromino_moveopt_LEFT:
            move_x = -1;
            break;
        case tetromino_moveopt_RIGHT:
            move_x = 1;
            break;
        case tetromino_moveopt_DOWN:
            move_y = 1;
            break;
        default:
            break;
    }

    // get the xy coordinates of the tetromino on the board, plus the direction wanted to move
    for (int i = 0; i < 4; i++) {
        for (int j = 0; j < 4; j++) {
            if (_debug_tetromino_matrix[current_tetro.tetromino_type][current_tetro.tetromino_orientation][i][j] != ' ') {
                coord[coord_idx][0] = current_tetro.pos_x + j - current_tetro.offset_x + move_x;
                coord[coord_idx][1] = current_tetro.pos_y + i - current_tetro.offset_y + move_y;
                coord_idx++;
            }
        }
    }

    // check if gameboard boundary reached
    for (int i = 0; i < 4; i++) {
        // check x range
        if (!(coord[i][0] >= 0 && coord[i][0] < COLS)) {
            return false;
        }

        // check y range
        if (!(coord[i][1] >= 0 && coord[i][1] < ROWS)) {
            return false;
        }
    }

    // check if hitting another 'X'
    for (int i = 0; i < 4; i++) {
        if (_debug_currentboard[coord[i][1]][coord[i][0]] != ' ') {
            return false;
        }
    }

    return true;
}

// check if the tetrimino can rotate in the available directions
static bool _debug_canrotatetetro(_debug_currenttetromino current_tetro, _debug_tetromino_orientation ori) {
    int coord[4][2];
    int coord_idx = 0;
    _debug_currenttetromino rotated_tetro = current_tetro;

    // track what happens to the current tetromino with the new rotation data
    rotated_tetro.tetromino_orientation = ori;
    rotated_tetro.offset_x = _debug_tetromino_block_offset[rotated_tetro.tetromino_type][rotated_tetro.tetromino_orientation][0];
    rotated_tetro.offset_y = _debug_tetromino_block_offset[rotated_tetro.tetromino_type][rotated_tetro.tetromino_orientation][1];

    // get the xy coordinates of the tetromino on the board, plus the direction wanted to move
    for (int i = 0; i < 4; i++) {
        for (int j = 0; j < 4; j++) {
            if (_debug_tetromino_matrix[rotated_tetro.tetromino_type][rotated_tetro.tetromino_orientation][i][j] != ' ') {
                coord[coord_idx][0] = rotated_tetro.pos_x + j - rotated_tetro.offset_x;
                coord[coord_idx][1] = rotated_tetro.pos_y + i - rotated_tetro.offset_y;
                coord_idx++;
            }
        }
    }

    // check if gameboard boundary reached
    for (int i = 0; i < 4; i++) {
        // check x range
        if (!(coord[i][0] >= 0 && coord[i][0] < COLS)) {
            return false;
        }

        // check y range
        if (!(coord[i][1] >= 0 && coord[i][1] < ROWS)) {
            return false;
        }
    }

    // check if hitting another 'X'
    for (int i = 0; i < 4; i++) {
        if (_debug_currentboard[coord[i][1]][coord[i][0]] != ' ') {
            return false;
        }
    }

    return true;
}

// rotate the tetromino in the specified direction
static void _debug_rotatetetro(_debug_currenttetromino *current_tetro, _debug_tetromino_rotopt rotopt) {
    // loop back options
    _debug_tetromino_orientation new_orientation;
    if ((current_tetro->tetromino_orientation == tetromino_orientation_0) && (rotopt == tetromino_rotopt_COUNTERCLOCKWISE)) {
        new_orientation = tetromino_orientation_3;
    } else if ((current_tetro->tetromino_orientation == tetromino_orientation_3) && (rotopt == tetromino_rotopt_CLOCKWISE)) {
        new_orientation = tetromino_orientation_0;
    } else {
        if (rotopt == tetromino_rotopt_CLOCKWISE) {
            new_orientation = current_tetro->tetromino_orientation + 1;
        } else {
            new_orientation = current_tetro->tetromino_orientation - 1;
        }
    }

    if (_debug_canrotatetetro(*current_tetro, new_orientation)) {
        current_tetro->tetromino_orientation = new_orientation;
        current_tetro->offset_x = _debug_tetromino_block_offset[current_tetro->tetromino_type][current_tetro->tetromino_orientation][0];
        current_tetro->offset_y = _debug_tetromino_block_offset[current_tetro->tetromino_type][current_tetro->tetromino_orientation][1];
    }    
}

// check if a row has been completed
static int _debug_firstrowcomplete() {
    int x_count = 1;

    for (int i = 0; i < ROWS; i++) {
        for (int j = 0; j < COLS; j++) {
            if (_debug_currentboard[i][j] != ' ') {
                x_count++;
            }
        }

        if (x_count == COLS) {
            return i;
        } else {
            x_count = 0;
        }
    }

    return -1;
}

// delete a given row and shift everything down
static void _debug_shiftrowsdown(int row) {    
    // shift all rows above row down by one row
    for (int i = row - 1; i >= 0; i--) {
        for (int j = 0; j < COLS; j++) {
            _debug_currentboard[i+1][j] = _debug_currentboard[i][j];
        }
    }
    
    // clear the top row that was shifted down
    for (int j = 0; j < COLS; j++) {
        _debug_currentboard[0][j] = ' ';
    }
}

// ----------------------------------------------------------------------------

// read and print the background color
static void print_background_color() {
	vga_ball_arg_t vla;

	if (ioctl(vga_ball_fd, VGA_BALL_READ_BACKGROUND, &vla)) {
		perror("ioctl(VGA_BALL_READ_BACKGROUND) failed");
		return;
	}

	printf("%02x %02x %02x\n", vla.background.red, vla.background.green, vla.background.blue);
}

// set the location of the current tetrimino on the game screen
static void set_tetromino_position(const tetromino_location_t *a, _debug_currenttetromino current_tetro) {
	pthread_mutex_lock(&current_tetrimino_lock);

    vga_ball_arg_t vla;
    //a->hor  = L_BORDER + (PX_PER_BLOCK *(current_tetro.pos_x - current_tetro.offset_x));
    //a->vert = T_BORDER + (PX_PER_BLOCK *(current_tetro.pos_y - current_tetro.offset_y));
	vla.tetro_loco = *a;
    vla.tetro_loco.hor  = L_BORDER + (PX_PER_BLOCK *(current_tetro.pos_x - current_tetro.offset_x));
    vla.tetro_loco.vert = T_BORDER + (PX_PER_BLOCK *(current_tetro.pos_y - current_tetro.offset_y));

	if (ioctl(vga_ball_fd, TETROMINO_WRITE_LOCATION, &vla)) {
		perror("ioctl(TETROMINO_WRITE_LOCATION failed");
		return;
	}
	
    pthread_mutex_unlock(&current_tetrimino_lock);
}

// select the choice of your tetromino
static void set_tetromino_choice(const tetromino_choice_t *c) {
	pthread_mutex_lock(&current_tetrimino_lock);
	vga_ball_arg_t vla;
	vla.choice = *c;
	if (ioctl(vga_ball_fd, TETROMINO_WRITE_CHOICE, &vla)) {
		perror("ioctl(TETROMINO_WRITE_CHOICE failed");
		return;
	}
	pthread_mutex_unlock(&current_tetrimino_lock);
}

// helper function for the above set tetromino choice function
static void set_tetromino_type_and_ori(_debug_currenttetromino current_tetro) {
    unsigned int i = (unsigned int)(current_tetro.tetromino_type * 4) + (unsigned int)current_tetro.tetromino_orientation;
	set_tetromino_choice((tetromino_choice_t *)&i);
}

// set the values of a row on the background of the screen
static void set_tetromino_row(const tetromino_row_t *r) {
	pthread_mutex_lock(&current_tetrimino_lock);
	vga_ball_arg_t vla;
	vla.row = *r;
	if (ioctl(vga_ball_fd, TETROMINO_WRITE_ROW, &vla)) {
		perror("ioctl(TETROMINO_WRITE_ROW failed");
		return;
	}
	pthread_mutex_unlock(&current_tetrimino_lock);
}

// helper function for the above set row function
static void set_tetromino_to_background(_debug_currenttetromino current_tetro) {
    unsigned int row_update_start = (uint32_t)current_tetro.pos_y;
    for (unsigned int i = row_update_start; i < row_update_start + 4; i++) {
        // get row color data
        unsigned int row_data = 0;
        for (char col = 0; col < COLS; col++) {
            // get color of each column block in row
            unsigned int color = 0;
            switch(_debug_currentboard[i][col]) {
                case 'B':
                    color = 0b1;
                    break;
                case 'R':
                    color = 0b10;
                    break;
                case 'W':
                    color = 0b11;
                    break;
                default:
                    break;
            }

            // write that to the row data and shift it accordingly
            row_data |= color;
            row_data <<= 2;
        }

        // convert your row color data to correct format
        row_data <<= 10;
        tetromino_row_t tetro_row;
        tetro_row.contents = row_data;
        tetro_row.num = ROWS - i - 1;
        set_tetromino_row(&tetro_row);
    }
}

// helper function for the above set row function
static void set_tetromino_clear_row(int row) {
    tetromino_row_t clear_data;
    clear_data.contents = 0;
    clear_data.num = row;
    set_tetromino_row(&clear_data);
}

// helper function to clear the board
static void set_tetromino_clear_board() {
    for (int i = 0; i < ROWS; i++) {
        set_tetromino_clear_row(i);
    }
}

// helper function for the above set row function
static void set_tetromino_redraw_board() {
    for (int i = 0; i < ROWS; i++) {
        // get row color data
        unsigned int row_data = 0;
        for (char col = 0; col < COLS; col++) {
            // get color of each column block in row
            unsigned int color = 0;
            switch(_debug_currentboard[i][col]) {
                case 'B':
                    color = 0b1;
                    break;
                case 'R':
                    color = 0b10;
                    break;
                case 'W':
                    color = 0b11;
                    break;
                default:
                    break;
            }

            // write that to the row data and shift it accordingly
            row_data |= color;
            row_data <<= 2;
        }

        // convert your row color data to correct format
        row_data <<= 10;
        tetromino_row_t tetro_row;
        tetro_row.contents = row_data;
        tetro_row.num = ROWS - i - 1;
        set_tetromino_row(&tetro_row);
    }

    unsigned int row_data = 0;
    tetromino_row_t tetro_row;
    tetro_row.contents = row_data;
    tetro_row.num = 19;
    set_tetromino_row(&tetro_row);
}

// set the next tetromino on the display
static void set_tetromino_next(const tetromino_choice_t *c) {
	pthread_mutex_lock(&current_tetrimino_lock);
	vga_ball_arg_t vla;
	vla.next = *c;
	if (ioctl(vga_ball_fd, TETROMINO_WRITE_NEXT, &vla)) {
		perror("ioctl(TETROMINO_WRITE_NEXT failed");
		return;
	}
	pthread_mutex_unlock(&current_tetrimino_lock);
}