/* * Device driver for the VGA video generator
 *
 * A Platform device implemented using the misc subsystem
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod cam_vga.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree cam_vga.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "cam_vga.h"

#define DRIVER_NAME "cam_vga"

/* Device registers */
#define DEFAULT(x) (x)
#define OPTIONAL_SETTER(x) ((x)+1) //maybe used later? dunno.

/*
 * Information about our device
 */
struct cam_vga {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
	cam_vga_t mode;
} dev;
/*
ioread takes in an address and RETURNS what value is writen there. 
Read whatever data needed set by our mode, pixel,VS, etc.
*/
static unsigned int read_info(void)
{
	return ioread32(DEFAULT(dev.virtbase));
}

static void write_mode(cam_vga_t *mode)
{
	iowrite8(mode->mode, OPTIONAL_SETTER(dev.virtbase));
}
/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
 */
/*
static void write_background(cam_vga_t *background)
{
	iowrite8(background->red, BG_RED(dev.virtbase) );
	iowrite8(background->green, BG_GREEN(dev.virtbase) );
	iowrite8(background->blue, BG_BLUE(dev.virtbase) );
	dev.background = *background;
}

static void write_cords(cam_vga_cords_t *cords) {
	iowrite8(cords->x, CORDS_X(dev.virtbase) );
	iowrite8(cords->y, CORDS_Y(dev.virtbase) );
	dev.cords = *cords;
}
*/

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long cam_vga_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	cam_vga_arg_t vla;

	switch (cmd) {
		case cam_vga_WRITE_MODE:
			if (copy_from_user(&vla, (cam_vga_arg_t *) arg,
						sizeof(cam_vga_arg_t)))
				return -EACCES;
			write_mode(&vla.mode);
			break;

		case cam_vga_READ_INFO:
			;
			unsigned int info = read_info(); //read pixel? read VS?
			if (copy_to_user((int *) arg, &info,
						sizeof(int)))
				return -EACCES;
			break;
		default:
			return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations cam_vga_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = cam_vga_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice cam_vga_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &cam_vga_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init cam_vga_probe(struct platform_device *pdev)
{
	int ret;

	/* Register ourselves as a misc device: creates /dev/cam_vga */
	ret = misc_register(&cam_vga_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
				DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}

	// /* Set an initial color */
	// write_background(&teal);
	// /* Set an initial position */
	// write_cords(&center);

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&cam_vga_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int cam_vga_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&cam_vga_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id cam_vga_of_match[] = {
	{ .compatible = "csee4840,cam_vga-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, cam_vga_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver cam_vga_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(cam_vga_of_match),
	},
	.remove	= __exit_p(cam_vga_remove),
};

/* Called when the module is loaded: set things up */
static int __init cam_vga_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&cam_vga_driver, cam_vga_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit cam_vga_exit(void)
{
	platform_driver_unregister(&cam_vga_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(cam_vga_init);
module_exit(cam_vga_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("VGA ball driver");
