/*
 * Avalon memory-mapped peripheral that generates VGA
 * Modified for use in Live-Feed Design Project
 * Original Code: Stephen A. Edwards
 * Columbia University
 */

module vga_ball(input logic        clk,
	        input logic 	   reset,
		input logic [31:0]  writedata,
		input logic 	   write,
		input 		   chipselect,
		input logic [2:0]  address,
		output logic [7:0] VGA_R, VGA_G, VGA_B,
		output logic 	   VGA_CLK, VGA_HS, VGA_VS,
		                   VGA_BLANK_n,
		output logic 	   VGA_SYNC_n,
		output logic [9:0] DEBUG_LED
		);
	
	/////////////////// Variables //////////////// 

	//vga logic
	logic [10:0]	hcount;
	logic [9:0]     vcount;

	logic [7:0] 	background_r, background_g, background_b;
	logic 			endOfField;
	logic	[2:0]	read_state = 0;
	logic	[2:0]	write_state = 0;

	// ram logic
	logic	[7:0]	network_pixel;
	logic	[18:0]  read_addr = 19'h0;	//inital address to read
	logic	[18:0]  write_addr = 0;
	logic			write_enable = 1;	// If set to 1: testing
	logic			read_enable = 0;	//set to 1 for testing reading from memory
	logic	[7:0]  	vga_pixel;

	
	//vga logic
	logic [7:0] pixel_color;
	logic [18:0] buffer;
	
	///////////// Instantiate Block Ram and Counter ///////////// 

	   
	vga_counters counters(.clk50(clk), .*);

	ram memory(
		//inputs
		.clock(clk),
		.rdaddress(read_addr),
		.wraddress(write_addr),
		.wren(write_enable),
		.data(network_pixel),
		.q(vga_pixel)
	);

	////////////////// Logic ///////////////////////
	// Test to see if LED is lighting up (active low?)

	assign DEBUG_LED[0] = (write_addr[0] == 1) ? 1'b1 : 1'b0;
	assign DEBUG_LED[1] = (write_addr[1] == 1) ? 1'b1 : 1'b0;
	assign DEBUG_LED[2] = (write_addr[2] == 1) ? 1'b1 : 1'b0;
	assign DEBUG_LED[3] = (write_addr[3] == 1) ? 1'b1 : 1'b0;
	assign DEBUG_LED[4] = (read_state == 0) ? 1'b1 : 1'b0;
	assign DEBUG_LED[5] = (read_state == 1) ? 1'b1 : 1'b0;
	assign DEBUG_LED[6] = (read_state == 2) ? 1'b1 : 1'b0;
	assign DEBUG_LED[7] = (write_state == 0) ? 1'b1 : 1'b0;
	assign DEBUG_LED[8] = (write_state == 1) ? 1'b1 : 1'b0;		// Test if on
	assign DEBUG_LED[9] = (write_state == 2) ? 1'b1 : 1'b0;

	// Read State Machine
	always_ff @(posedge clk)
		if (reset)
			{background_r, background_g, background_b} <= {8'h0, 8'h0, 8'h0};
			

	always_ff @(posedge clk)
		case (write_state)
			0: begin 
				// Write data from software into BRAM
				// Once HPS tells me we wrote to the last one, switch states				
				if (chipselect && write && address == 1'b0) begin
					network_pixel 	<= writedata[7:0];
					write_addr 		<= writedata[26:8];	
					
					if (write_addr == 307199) begin
						write_state 	<= 1;		// Change state to 1
						write_enable 	<= 0;		// Turn off writing
					end
				end
							
			end
			1: begin
				// reset write_addr and wait 
				write_addr <= 0;
				if (read_state == 2)
					write_state 	<= 0;
					write_enable	<= 1;
			end
			2: begin
				if (chipselect && write && address == 1'b0) begin
					if (writedata[31:16] == 16'b0011_0000_0000_0000) begin 
						write_state 	<= 0;
						write_enable 	<= 1;
					end
				end

			end
		endcase
	
	always_ff @(posedge clk)
		case (read_state)
			0:		// Idle
				if (write_state == 1 && endOfField == 1) begin
					read_state 	<= 1;
					read_addr 	<= 0;
				end
			1: begin	// Basic: increment through BRAM and display contents
				if (read_addr == 307199)
					read_state 	<= 2;   //2		
				else if (hcount[0] && VGA_BLANK_n)
					read_addr 	<= read_addr + 1;

			end
			2: 	// Momentarily hold state for writing to restart
				read_state <= 0;



		endcase		


	always_comb begin
		{VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0};
		if (VGA_BLANK_n) begin
			if (read_state == 1) begin
				//decide correct pixel to read from address
				{VGA_R, VGA_G, VGA_B} = {vga_pixel,vga_pixel,vga_pixel};
			end

			//else if(hcount [10:1] > 20) begin
			//	{VGA_R, VGA_G, VGA_B} = {8'hFF, 8'hFF, 8'hFF};
			//end		
		end
	end	       
endmodule

module vga_counters(
 input logic 	     clk50, reset,

 output logic [10:0] hcount,  // hcount[10:1] is pixel column
 output logic [9:0]  vcount,  // vcount[9:0] is pixel row
 output logic 	     VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n, endOfField);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC +
                            HBACK_PORCH; // 1600

   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC +
                            VBACK_PORCH; // 525

   logic endOfLine;

   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;	
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;

   //logic endOfField;

   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) &
		      !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1'b0; // For putting sync on the green signal; unused
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: rising edge sensitive
   
endmodule



// Inferred Bram Taken From Class Lecture
module ram(
	input logic clock,
	input logic [18:0] rdaddress, wraddress,
	input logic wren,
	input logic [7:0] data,
	output logic [7:0] q);
	
	logic [7:0] mem [307199:0];
	always_ff @(posedge clock) begin
		if (wren) mem[wraddress] <= data;
		q <= mem[rdaddress];
	end
endmodule
