`timescale 1ns / 1ps
// Fully Connected Layer
// The module declaration, with a bunch of parameters. M, N, Ma, N_f, C, and S_P_o are presumably controlling the size and behavior of the FC layer in some way.
module FC#(parameter M=8,N=0,Ma=16,N_f=12,C=10,S_P_o=4)(

// Input and output declarations. The inputs include the clock and reset signals, weights W0 to W9, and an input Ii. The outputs include FCi, Fi, LeNet_Out, and a 'finish' signal to indicate the computations are done.
input clk,rst,
[M-1:0]W0,W1,W2,W3,W4,W5,W6,W7,W8,W9,
Ii,
output reg [Ma-1:0]FCi,Fi, 
[M-1:0]O0,O1,O2,O3,O4,O5,O6,O7,O8,O9,LeNet_Out, 
reg finish);

// Here are some internal wires and registers used for computations and for keeping track of intermediate results.
wire [M-1:0]Po0,Po1,Po2,Po3,Po4,Po5,Po6,Po7,Po8,Po9;
wire Ps0,Ps1,Ps2,Ps3,Ps4,Ps5,Ps6,Ps7,Ps8,Ps9;
reg [M-1:0]Pi0,Pi1,Pi2,Pi3,Pi4,Pi5,Pi6,Pi7,Pi8,Pi9;

// This block controls the index for the columns. It resets to 0 when the reset signal is high, or when the FC layer has finished processing and the 'finish' signal is low. Otherwise, it increments the index Fi.
always @(posedge clk)
if(rst || FCi<S_P_o*S_P_o && Fi==N_f-1 && ~finish)
   Fi<=0;
else if(FCi<S_P_o*S_P_o && Fi<N_f-1 && ~finish)
   Fi<=Fi+1;

// This block controls the index for the rows. It resets to 0 when the reset signal is high, and increments when the FC layer has finished processing a column and the 'finish' signal is low.
always @(posedge clk)
if(rst)
   FCi<=0;
else if(FCi<S_P_o*S_P_o-1 && Fi==N_f-1)
   FCi<=FCi+1;

// This block updates the 'finish' signal. It resets to 0 when the reset signal is high, and sets to 1 when the FC layer has finished processing all rows and columns.
always @(posedge clk)
if(rst)
   finish<=0;
else if(FCi==S_P_o*S_P_o-1 && Fi==N_f-1)
   finish <= 1;

// This appears to be a softmax operation, which is commonly used in the output layer of a neural network to generate probabilities for each class. However, this isn't valid Verilog syntax; it seems to be more of a pseudocode representation of the softmax function.
softmax max(Po0,Po1,Po2,Po3,Po4,Po5,Po6,Po7,Po8, Po9,Ps0,Ps1,Ps2,Ps3,Ps4,Ps5,Ps6,Ps7,Ps8,Ps9);

// These lines are assigning the outputs after the FC layer finishes its computation. The ternary operator is used to check the MSB of Po0-Po9. If it's 1, it fills the upper half with '1's, else with '0's.
assign O0=(Po0[M-1])?{16'hffff,Po0[M-1:16]}:{16'h0000,Po0[M-1:16]};
assign O1=(Po1[M-1])?{16'hffff,Po1[M-1:16]}:{16'h0000,Po1[M-1:16]};
assign O2=(Po2[M-1])?{16'hffff,Po2[M-1:16]}:{16'h0000,Po2[M-1:16]};
assign O3=(Po3[M-1])?{16'hffff,Po3[M-1:16]}:{16'h0000,Po3[M-1:16]};
assign O4=(Po4[M-1])?{16'hffff,Po4[M-1:16]}:{16'h0000,Po4[M-1:16]};
assign O5=(Po5[M-1])?{16'hffff,Po5[M-1:16]}:{16'h0000,Po5[M-1:16]};
assign O6=(Po6[M-1])?{16'hffff,Po6[M-1:16]}:{16'h0000,Po6[M-1:16]};
assign O7=(Po7[M-1])?{16'hffff,Po7[M-1:16]}:{16'h0000,Po7[M-1:16]};
assign O8=(Po8[M-1])?{16'hffff,Po8[M-1:16]}:{16'h0000,Po8[M-1:16]};
assign O9=(Po9[M-1])?{16'hffff,Po9[M-1:16]}:{16'h0000,Po9[M-1:16]};

// This line assigns the output digit value based on the outputs of the softmax operation. If 'finish' is 1, it calculates the sum of the probabilities, else it assigns -1.
assign LeNet_Out =(finish)?
                  (0*Ps0+1*Ps1+2*Ps2+3*Ps3+4*Ps4+5*Ps5+6*Ps6+7*Ps7+8*Ps8+9*Ps9):
                  -1;

// This block sets all Pi registers to 0 when the reset signal is high. Otherwise, it updates the Pi registers with the Po outputs when the 'finish' signal is low.
always@(posedge clk)
if(rst)
begin 
   Pi0<=0;
   Pi1<=0;
   Pi2<=0;
   Pi3<=0;
   Pi4<=0;
   Pi5<=0;
   Pi6<=0;
   Pi7<=0;
   Pi8<=0;
   Pi9<=0;
end
else if(!finish) 
begin 
    Pi0<=Po0;
    Pi1<=Po1;
    Pi2<=Po2;
    Pi3<=Po3;
    Pi4<=Po4;
    Pi5<=Po5;
    Pi6<=Po6;
    Pi7<=Po7;
    Pi8<=Po8;
    Pi9<=Po9;
end

// These lines instantiate multiple instances of a multiply-accumulate (MACC) module to perform the necessary computations. The input to each MACC module includes the 'finish' signal, a weight (W0-W9), the input Ii, and the corresponding Pi and Po.
MACC #(M,N)M0(finish,W0,Ii,Pi0,Po0);
MACC #(M,N)M1(finish,W1,Ii,Pi1,Po1);
MACC #(M,N)M2(finish,W2,Ii,Pi2,Po2);
MACC #(M,N)M3(finish,W3,Ii,Pi3,Po3);
MACC #(M,N)M4(finish,W4,Ii,Pi4,Po4);
MACC #(M,N)M5(finish,W5,Ii,Pi5,Po5);
MACC #(M,N)M6(finish,W6,Ii,Pi6,Po6);
MACC #(M,N)M7(finish,W7,Ii,Pi7,Po7);
MACC #(M,N)M8(finish,W8,Ii,Pi8,Po8);
MACC #(M,N)M9(finish,W9,Ii,Pi9,Po9);

// End of the module
endmodule

