/*
 * Avalon memory-mapped peripheral that generates VGA
 *
 * Stephen A. Edwards
 * Columbia University
 */

module vga_ball(
   input logic          clk,
   input logic 	      reset,
   input logic [31:0]   writedata,
   input logic 	      write,
   input 		         chipselect,
   input logic [2:0]    address,
   input logic          left_chan_ready,
	input logic          right_chan_ready,

   output logic [15:0]  sample_data_l,
   output logic [15:0]  sample_data_r,
   output logic         sample_valid_l,
   output logic         sample_valid_r,
   output logic [7:0]   VGA_R, VGA_G, VGA_B,
   output logic 	      VGA_CLK, VGA_HS, VGA_VS,VGA_BLANK_n,
   output logic 	      VGA_SYNC_n
);

   logic [31:0]         blocks, next_blocks;
   
   logic [10:0]	      hcount;
   logic [9:0]          vcount;

   logic [9:0]          paddle_pos, next_paddle_pos;
   logic [9:0]	         ball_hpos, next_ball_hpos;
   logic [8:0]	         ball_vpos, next_ball_vpos;

   logic [7:0] 	      background_r, background_g, background_b;

   logic [7:0]          score, next_score;
   logic [3:0]          audio, next_audio;
   logic [2:0]          lives, next_lives;

   logic                start, next_start;
	
   vga_counters counters(.clk50(clk), .*);

   always_comb begin
      next_ball_hpos = ball_hpos;
      next_ball_vpos = ball_vpos;
      next_paddle_pos = paddle_pos;
      next_lives = lives;
      next_blocks = blocks;
      next_audio = audio;
      next_score = score;
      next_start = start;
      case (address)
	      3'h1: begin
            next_ball_hpos = writedata[9:0];
            next_ball_vpos = writedata[18:10];
            next_paddle_pos = writedata[28:19];
		   end
         3'h2: begin
            next_blocks = writedata;
         end
         3'h3: begin
            next_lives = writedata[2:0];
            next_audio = writedata[6:3];
            next_score = writedata[14:7];
            next_start = writedata[15];
         end
      endcase
   end


   always_ff @(posedge clk) begin
      if (reset) begin
	      background_r <= 8'h0;
	      background_g <= 8'h0;
	      background_b <= 8'h80;
         blocks <= 32'b0;
		   ball_hpos <= 10'd300;
		   ball_vpos <= 9'd220;
         paddle_pos <= 10'd300;
         score <= 8'b0;
         audio <= 5'b0;
         lives <= 3'b111;
      end else if (chipselect && write)
         case (address)
	         3'h0 : begin
               background_r <= writedata[7:0];
               background_g <= writedata[15:8];
               background_b <= writedata[23:16];
            end
         endcase
      ball_hpos <= next_ball_hpos;   
	   ball_vpos <= next_ball_vpos;
      paddle_pos <= next_paddle_pos;
      blocks <= next_blocks;
      lives <= next_lives;
      audio <= next_audio;
      score <= next_score;
      start <= next_start;
   end



   // ============ AUDIO ============ // 

   logic [13:0] counter;

   // Brick Hit
   logic	[8:0]  bhs_addr;
	logic [15:0] bhs_output;
   logic bhs_en; 
	brick_sound_ROM bhs(.address(bhs_addr), .clock(clk), .q(bhs_output));

   // Paddle Hit
   logic	[13:0] phs_addr;
	logic [15:0] phs_output;
   logic phs_en; 
	coin_sound_ROM phs(.address(phs_addr), .clock(clk), .q(phs_output));

   // Game Win
   logic	[15:0] gws_addr;
	logic [15:0] gws_output;
   logic gws_en; 
	game_win_ROM gws(.address(gws_addr), .clock(clk), .q(gws_output));

   // Life Lost
   logic	[15:0] lls_addr;
	logic [15:0] lls_output;
   logic lls_en; 
	life_lost_ROM lls(.address(lls_addr), .clock(clk), .q(lls_output));


   // Sound Select Logic
   always_ff @(posedge clk) begin
		if (reset) begin
			counter <= 0;
			sample_valid_l <= 0; 
         sample_valid_r <= 0;
		end
		else if(left_chan_ready == 1 && right_chan_ready == 1 && counter < 6250) begin
			counter <= counter + 1;
			sample_valid_l <= 0; 
         sample_valid_r <= 0;
		end
		else if(left_chan_ready == 1 && right_chan_ready == 1 && counter == 6250) begin
			counter <= 0;
			sample_valid_l <= 1; 
         sample_valid_r <= 1;
         if (audio[0] || bhs_en == 1'b0) begin
				if (bhs_addr < 9'd290) begin
					bhs_addr <= bhs_addr+1;
					bhs_en <= 1'b0;
				end
				else begin
					bhs_addr <= 0;
					bhs_en <= 1'b1;
				end
				sample_data_l <= bhs_output;
				sample_data_r <= bhs_output;
         end else if (audio[1] || phs_en == 1'b0) begin
				if (phs_addr < 14'd9241) begin
					phs_addr <= phs_addr+1;
					phs_en <= 1'b0;
				end
				else begin
					phs_addr <= 0;
					phs_en <= 1'b1;
				end
				sample_data_l <= phs_output;
				sample_data_r <= phs_output;
         end else if (audio[2] || gws_en == 1'b0) begin
            if (gws_addr < 16'd33539) begin
               gws_addr <= gws_addr+1;
               gws_en <= 1'b0;
            end
            else begin
               gws_addr <= 0;
               gws_en <= 1'b1;
            end
            sample_data_l <= gws_output;
            sample_data_r <= gws_output;
         end else if (audio[3] || lls_en == 1'b0) begin
            if (lls_addr < 16'd36052) begin
               lls_addr <= lls_addr+1;
               lls_en <= 1'b0;
            end
            else begin
               lls_addr <= 0;
               lls_en <= 1'b1;
            end
            sample_data_l <= lls_output;
            sample_data_r <= lls_output;
         end
         else begin
            sample_data_l <= 0;
            sample_data_r <= 0;
         end
      end else begin
         sample_valid_l <= 0;
         sample_valid_r <= 0;
	   end
   end



   // ============ VIDEO ============ // 

   // Ball Logic 
   logic [7:0] ball_addr;
	logic [23:0] ball_output;
   logic ball_en;
	ball_ROM ball(.address(ball_addr),.clock(clk),.q(ball_output));
   always_ff @(posedge clk) begin
      if (hcount[10:1] >= ball_hpos-8 && hcount[10:1] < ball_hpos+8 && vcount[9:0] >= ball_vpos-8 && vcount[9:0]<ball_vpos+8) begin
         ball_en <= 1;
         ball_addr <= (hcount[10:1] - (ball_hpos-8)) + (vcount[9:0] - (ball_vpos-8)) * 16;
      end else begin
         ball_en <= 0;
         ball_addr <= 0;
      end
   end

   // Lives Logic 
   logic [7:0] lives_addr;
	logic [23:0] lives_output;
   logic lives_en;
	lives_ROM life(.address(lives_addr),.clock(clk),.q(lives_output));
   always_ff @(posedge clk) begin
      if (hcount[10:1] >= 608 && hcount[10:1] <= 624 && vcount[9:0] >= 8 && vcount[9:0] < 24 && lives[0]) begin
         lives_en <= 1;
         lives_addr <= (hcount[10:1] - 608) + (vcount[9:0] - 8) * 16;
      end else if (hcount[10:1] >= 584 && hcount[10:1] <= 600 && vcount[9:0] >= 8 && vcount[9:0] < 24 && lives[1]) begin
         lives_en <= 1;
         lives_addr <= (hcount[10:1] - 584) + (vcount[9:0] - 8) * 16;
      end else if (hcount[10:1] >= 560 && hcount[10:1] <= 576 && vcount[9:0] >= 8 && vcount[9:0] < 24 && lives[2]) begin
         lives_en <= 1;
         lives_addr <= (hcount[10:1] - 560) + (vcount[9:0] - 8) * 16;
      end else begin
         lives_en <= 0;
         lives_addr <= 0;
      end
   end

   // Game Logic
   logic [11:0] game_addr;
	logic [23:0] game_output;
   logic game_en;
	game_ROM game(.address(game_addr),.clock(clk),.q(game_output));
   always_ff @(posedge clk) begin
      if (hcount[10:1] >= 280 && hcount[10:1] <= 360 && vcount[9:0] >= 220 && vcount[9:0] < 240 && score[7:4] == 8) begin
         game_en <= 1;
         game_addr <= ((hcount[10:1] - 280) + (vcount[9:0] - 220) * 80);
      end else if (hcount[10:1] >= 280 && hcount[10:1] <= 360 && vcount[9:0] >= 220 && vcount[9:0] < 240 && lives == 0) begin
         game_en <= 1;
         game_addr <= (hcount[10:1] - 280) + (vcount[9:0] - 220) * 80 + 1600;
      end else begin
         game_en <= 0;
         game_addr <= 0;
      end
   end

   // Paddle Logic
   logic [10:0] paddle_addr;
	logic [23:0] paddle_output;
   logic paddle_en;
	paddle_ROM paddle(.address(paddle_addr),.clock(clk),.q(paddle_output));
   always_ff @(posedge clk) begin
      if (hcount[10:1] >= paddle_pos-40 && hcount[10:1] < paddle_pos+40 && vcount[9:0] >= 450 && vcount[9:0] <= 470) begin
         paddle_en <= 1;
         paddle_addr <= (hcount[10:1] - (paddle_pos-40)) + (vcount[9:0] - 450) * 80;
      end else begin
         paddle_en <= 0;
         paddle_addr <= 0;
      end
   end

   // Score Logic 
   logic [11:0] score_addr;
	logic [23:0] score_output;
   logic score_en;
	score_ROM score_mem(.address(score_addr),.clock(clk),.q(score_output));
   always_ff @(posedge clk) begin
      if (hcount[10:1] >= 16 && hcount[10:1] <= 32 && vcount[9:0] >= 8 && vcount[9:0] < 24) begin
         score_en <= 1;
         score_addr <= ((hcount[10:1] - 16) + (vcount[9:0] - 8) * 16) + (256 * score[7:4]);
      end else if (hcount[10:1] >= 40 && hcount[10:1] <= 56 && vcount[9:0] >= 8 && vcount[9:0] < 24) begin
         score_en <= 1;
         score_addr <= ((hcount[10:1] - 40) + (vcount[9:0] - 8) * 16) + (256 * score[3:0]);
      end else begin
         score_en <= 0;
         score_addr <= 0;
      end
   end


   // First Brick Row 
   logic [10:0] pink_addr;
   logic [23:0] pink_output;
   logic pink_en;
   pink_ROM pink(.address(pink_addr),.clock(clk),.q(pink_output));
   always_ff @(posedge clk) begin
      if (blocks[0] == 1 && hcount[10:1] >= 15 && hcount[10:1] <= 79 && vcount[9:0] >= 32 && vcount[9:0] < 64) begin
         pink_en <= 1;
         pink_addr <= (hcount[10:1]-15) + (vcount[9:0] - 32) * 64;
      end else if (blocks[1] == 1 && hcount[10:1] >= 93 && hcount[10:1] <= 157 && vcount[9:0] >= 32 && vcount[9:0] < 64) begin
         pink_en <= 1;
         pink_addr <= (hcount[10:1]-93) + (vcount[9:0] - 32) * 64;
      end else if (blocks[2] == 1 && hcount[10:1] >= 171 && hcount[10:1] <= 235 && vcount[9:0] >= 32 && vcount[9:0] < 64) begin
         pink_en <= 1;
         pink_addr <= (hcount[10:1]-171) + (vcount[9:0] - 32) * 64;
      end else if (blocks[3] == 1 && hcount[10:1] >= 249 && hcount[10:1] <= 313 && vcount[9:0] >= 32 && vcount[9:0] < 64) begin
         pink_en <= 1;
         pink_addr <= (hcount[10:1]-249) + (vcount[9:0] - 32) * 64;
      end else if (blocks[4] == 1 && hcount[10:1] >= 327 && hcount[10:1] <= 391 && vcount[9:0] >= 32 && vcount[9:0] < 64) begin
         pink_en <= 1;
         pink_addr <= (hcount[10:1]-327) + (vcount[9:0] - 32) * 64;
      end else if (blocks[5] == 1 && hcount[10:1] >= 405 && hcount[10:1] <= 469 && vcount[9:0] >= 32 && vcount[9:0] < 64) begin
         pink_en <= 1;
         pink_addr <= (hcount[10:1]-405) + (vcount[9:0] - 32) * 64;
      end else if (blocks[6] == 1 && hcount[10:1] >= 483 && hcount[10:1] <= 547 && vcount[9:0] >= 32 && vcount[9:0] < 64) begin
         pink_en <= 1;
         pink_addr <= (hcount[10:1]-483) + (vcount[9:0] - 32) * 64;
      end else if (blocks[7] == 1 && hcount[10:1] >= 561 && hcount[10:1] <= 625 && vcount[9:0] >= 32 && vcount[9:0] < 64) begin
         pink_en <= 1;
         pink_addr <= (hcount[10:1]-561) + (vcount[9:0] - 32) * 64;
      end else begin
         pink_en <= 0;
         pink_addr <= 0;
      end
   end

   // Second Brick Row
   logic green_en;
   logic [10:0] green_addr;
   logic [23:0] green_output;
   green_ROM green(.address(green_addr),.clock(clk),.q(green_output)); 
   always_ff @(posedge clk) begin
      if (blocks[8] == 1 && hcount[10:1] >= 15 && hcount[10:1] <= 79 && vcount[9:0] >= 74 && vcount[9:0] < 106) begin
         green_en <= 1;
         green_addr <= (hcount[10:1]-15) + (vcount[9:0] - 74) * 64;
      end else if (blocks[9] == 1 && hcount[10:1] >= 93 && hcount[10:1] <= 157 && vcount[9:0] >= 74 && vcount[9:0] < 106) begin
         green_en <= 1;
         green_addr <= (hcount[10:1]-93) + (vcount[9:0] - 74) * 64;
      end else if (blocks[10] == 1 && hcount[10:1] >= 171 && hcount[10:1] <= 235 && vcount[9:0] >= 74 && vcount[9:0] < 106) begin
         green_en <= 1;
         green_addr <= (hcount[10:1]-171) + (vcount[9:0] - 74) * 64;
      end else if (blocks[11] == 1 && hcount[10:1] >= 249 && hcount[10:1] <= 313 && vcount[9:0] >= 74 && vcount[9:0] < 106) begin
         green_en <= 1;
         green_addr <= (hcount[10:1]-249) + (vcount[9:0] - 74) * 64;
      end else if (blocks[12] == 1 && hcount[10:1] >= 327 && hcount[10:1] <= 391 && vcount[9:0] >= 74 && vcount[9:0] < 106) begin
         green_en <= 1;
         green_addr <= (hcount[10:1]-327) + (vcount[9:0] - 74) * 64;
      end else if (blocks[13] == 1 && hcount[10:1] >= 405 && hcount[10:1] <= 469 && vcount[9:0] >= 74 && vcount[9:0] < 106) begin
         green_en <= 1;
         green_addr <= (hcount[10:1]-405) + (vcount[9:0] - 74) * 64;
      end else if (blocks[14] == 1 && hcount[10:1] >= 483 && hcount[10:1] <= 547 && vcount[9:0] >= 74 && vcount[9:0] < 106) begin
         green_en <= 1;
         green_addr <= (hcount[10:1]-483) + (vcount[9:0] - 74) * 64;
      end else if (blocks[15] == 1 && hcount[10:1] >= 561 && hcount[10:1] <= 625 && vcount[9:0] >= 74 && vcount[9:0] < 106) begin
         green_en <= 1;
         green_addr <= (hcount[10:1]-561) + (vcount[9:0] - 74) * 64;
      end else begin
         green_en <= 0;
         green_addr <= 0;
      end
   end

   
   // Third Brick Row
   logic purple_en;
   logic [10:0] purple_addr;
   logic [23:0] purple_output;
   purple_ROM purple(.address(purple_addr),.clock(clk),.q(purple_output));
   always_ff @(posedge clk) begin
       if (blocks[16] == 1 && hcount[10:1] >= 15 && hcount[10:1] <= 79 && vcount[9:0] >= 116 && vcount[9:0] < 148) begin
         purple_en <= 1;
         purple_addr <= (hcount[10:1]-15) + (vcount[9:0] - 116) * 64;
      end else if (blocks[17] == 1 && hcount[10:1] >= 93 && hcount[10:1] <= 157 && vcount[9:0] >= 116 && vcount[9:0] < 148) begin
         purple_en <= 1;
         purple_addr <= (hcount[10:1]-93) + (vcount[9:0] - 116) * 64;
      end else if (blocks[18] == 1 && hcount[10:1] >= 171 && hcount[10:1] <= 235 && vcount[9:0] >= 116 && vcount[9:0] < 148) begin
         purple_en <= 1;
         purple_addr <= (hcount[10:1]-171) + (vcount[9:0] - 116) * 64;
      end else if (blocks[19] == 1 && hcount[10:1] >= 249 && hcount[10:1] <= 313 && vcount[9:0] >= 116 && vcount[9:0] < 148) begin
         purple_en <= 1;
         purple_addr <= (hcount[10:1]-249) + (vcount[9:0] - 116) * 64;
      end else if (blocks[20] == 1 && hcount[10:1] >= 327 && hcount[10:1] <= 391 && vcount[9:0] >= 116 && vcount[9:0] < 148) begin
         purple_en <= 1;
         purple_addr <= (hcount[10:1]-327) + (vcount[9:0] - 116) * 64;
      end else if (blocks[21] == 1 && hcount[10:1] >= 405 && hcount[10:1] <= 469 && vcount[9:0] >= 116 && vcount[9:0] < 148) begin
         purple_en <= 1;
         purple_addr <= (hcount[10:1]-405) + (vcount[9:0] - 116) * 64;
      end else if (blocks[22] == 1 && hcount[10:1] >= 483 && hcount[10:1] <= 547 && vcount[9:0] >= 116 && vcount[9:0] < 148) begin
         purple_en <= 1;
         purple_addr <= (hcount[10:1]-483) + (vcount[9:0] - 116) * 64;
      end else if (blocks[23] == 1 && hcount[10:1] >= 561 && hcount[10:1] <= 625 && vcount[9:0] >= 116 && vcount[9:0] < 148) begin
         purple_en <= 1;
         purple_addr <= (hcount[10:1]-561) + (vcount[9:0] - 116) * 64;
      end else begin
         purple_en <= 0;
         purple_addr <= 0;
      end
   end

   // Fourth Brick Row
   logic orange_en;
   logic [10:0] orange_addr;
   logic [23:0] orange_output;
   orange_ROM orange(.address(orange_addr),.clock(clk),.q(orange_output));
   always_ff @(posedge clk) begin
       if (blocks[24] == 1 && hcount[10:1] >= 15 && hcount[10:1] <= 79 && vcount[9:0] >= 158 && vcount[9:0] < 190) begin
         orange_en <= 1;
         orange_addr <= (hcount[10:1]-15) + (vcount[9:0] - 158) * 64;
      end else if (blocks[25] == 1 && hcount[10:1] >= 93 && hcount[10:1] <= 157 && vcount[9:0] >= 158 && vcount[9:0] < 190) begin
         orange_en <= 1;
         orange_addr <= (hcount[10:1]-93) + (vcount[9:0] - 158) * 64;
      end else if (blocks[26] == 1 && hcount[10:1] >= 171 && hcount[10:1] <= 235 && vcount[9:0] >= 158 && vcount[9:0] < 190) begin
         orange_en <= 1;
         orange_addr <= (hcount[10:1]-171) + (vcount[9:0] - 158) * 64;
      end else if (blocks[27] == 1 && hcount[10:1] >= 249 && hcount[10:1] <= 313 && vcount[9:0] >= 158 && vcount[9:0] < 190) begin
         orange_en <= 1;
         orange_addr <= (hcount[10:1]-249) + (vcount[9:0] - 158) * 64;
      end else if (blocks[28] == 1 && hcount[10:1] >= 327 && hcount[10:1] <= 391 && vcount[9:0] >= 158 && vcount[9:0] < 190) begin
         orange_en <= 1;
         orange_addr <= (hcount[10:1]-327) + (vcount[9:0] - 158) * 64;
      end else if (blocks[29] == 1 && hcount[10:1] >= 405 && hcount[10:1] <= 469 && vcount[9:0] >= 158 && vcount[9:0] < 190) begin
         orange_en <= 1;
         orange_addr <= (hcount[10:1]-405) + (vcount[9:0] - 158) * 64;
      end else if (blocks[30] == 1 && hcount[10:1] >= 483 && hcount[10:1] <= 547 && vcount[9:0] >= 158 && vcount[9:0] < 190) begin
         orange_en <= 1;
         orange_addr <= (hcount[10:1]-483) + (vcount[9:0] - 158) * 64;
      end else if (blocks[31] == 1 && hcount[10:1] >= 561 && hcount[10:1] <= 625 && vcount[9:0] >= 158 && vcount[9:0] < 190) begin
         orange_en <= 1;
         orange_addr <= (hcount[10:1]-561) + (vcount[9:0] - 158) * 64;
      end else begin
         orange_en <= 0;
         orange_addr <= 0;
      end
   end

   always_comb begin
      {VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0};
      if (VGA_BLANK_n)
         // Display Ball
	  	   if (ball_en)
	         {VGA_R, VGA_G, VGA_B} = ball_output;
         // Display Paddle
         else if (paddle_en)
            {VGA_R, VGA_G, VGA_B} = paddle_output;
         // Display First Row of Blocks
         else if (pink_en)
            {VGA_R, VGA_G, VGA_B} = pink_output;
         // Display Second Row of Blocks
         else if (green_en)
            {VGA_R, VGA_G, VGA_B} = green_output;
         // Display Third Row of Blocks
         else if (purple_en)
            {VGA_R, VGA_G, VGA_B} = purple_output;
         // Display Fourth Row of Blocks
         else if (orange_en)
            {VGA_R, VGA_G, VGA_B} = orange_output;
         // Display Lives
         else if (lives_en)
            {VGA_R, VGA_G, VGA_B} = lives_output;
         // Display Score
         else if (score_en)
            {VGA_R, VGA_G, VGA_B} = score_output;
         // Display Game Win/Loss
         else if (game_en)
            {VGA_R, VGA_G, VGA_B} = game_output;
	      else
	         {VGA_R, VGA_G, VGA_B} = {background_r, background_g, background_b};
         
         if (~start)
            {VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0};
   end
	       
endmodule

module vga_counters(
 input logic 	      clk50, reset,
 output logic [10:0] hcount,  // hcount[10:1] is pixel column
 output logic [9:0]  vcount,  // vcount[9:0] is pixel row
 output logic 	      VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC +
                            HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC +
                            VBACK_PORCH; // 525

   logic endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;
       
   logic endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) &
		      !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1'b0; // For putting sync on the green signal; unused
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: rising edge sensitive
   
endmodule
