/*
 * Userspace program that communicates with the vga_ball device driver
 * through ioctls
 *
 * Stephen A. Edwards
 * Columbia University
 */
#include "usbcontroller.h"
#include "vga_ball.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>

#define BALL_RADIUS         8
#define BLOCK_HEIGHT        16
#define BLOCK_WIDTH         32
#define PADDLE_WIDTH        40
#define PADDLE_TOP          450
#define PADDLE_BOTTOM       470
#define PADDLE_ACCELERATION 10
#define PADDLE_DECELERATION 1

#define BRICK_HIT_SOUND     1
#define PADDLE_HIT_SOUND    2
#define GAME_WIN_SOUND      4
#define LIFE_LOST_SOUND     8

struct libusb_device_handle *controller;
struct usb_controller_packet packet;
uint8_t endpoint_address;
int transferred, vga_ball_fd;

/* Read and print the background color */
void print_background_color()
{
  vga_ball_arg_t vla;

  if (ioctl(vga_ball_fd, VGA_BALL_READ_BACKGROUND, &vla))
  {
    perror("ioctl(VGA_BALL_READ_BACKGROUND) failed");
    return;
  }
  printf("%02x %02x %02x\n",
         vla.background.red, vla.background.green, vla.background.blue);
}

/* Set the background color */
void set_background_color(const vga_ball_color_t *c)
{
  vga_ball_arg_t vla;
  vla.background = *c;
  if (ioctl(vga_ball_fd, VGA_BALL_WRITE_BACKGROUND, &vla))
  {
    perror("ioctl(VGA_BALL_SET_BACKGROUND) failed");
    return;
  }
}

/* Read and print the position */
void print_position()
{
  vga_ball_arg_t vla;

  if (ioctl(vga_ball_fd, VGA_BALL_READ_POSITION, &vla))
  {
    perror("ioctl(VGA_BALL_READ_POSITION) failed");
    return;
  }
  int ball_x = vla.position.ball_x;
  int ball_y = vla.position.ball_y;
  int paddle = vla.position.paddle;
  // printf("Ball: %d %d Paddle: %d\n", ball_x, ball_y, paddle);
}

/* Set the ball position */
void set_position(const vga_positions_t *c)
{
  vga_ball_arg_t vla;
  vla.position = *c;
  if (ioctl(vga_ball_fd, VGA_BALL_WRITE_POSITION, &vla))
  {
    perror("ioctl(VGA_BALL_SET_POSITION) failed");
    return;
  }
}

/* Set the ball position */
void set_blocks(const vga_blocks_t *c)
{
  vga_ball_arg_t vla;
  vla.blocks = *c;
  if (ioctl(vga_ball_fd, VGA_BLOCKS_WRITE_POSITION, &vla))
  {
    perror("ioctl(VGA_BLOCKS_SET_POSITION) failed");
    return;
  }
}

void set_info(const vga_gameinfo_t *c)
{
  vga_ball_arg_t vla;
  vla.gameinfo = *c;
  if (ioctl(vga_ball_fd, VGA_BALL_WRITE_INFO, &vla))
  {
    perror("ioctl(VGA_BALL_WRITE_INFO) failed");
    return;
  }
}

vga_positions_t encode_position(int ball_x, int ball_y, int paddle)
{
  vga_positions_t position;
  position.ball_x = ball_x & 0x3FF;
  position.ball_y = ball_y & 0x1FF;
  position.paddle = paddle & 0x3FF;
  // position.lives = lives & 3;
  // position.audio = audio & 1;
  return position;
}

unsigned short get_score(int score)
{
  char score1 = score / 10;
  char score2 = score % 10;
  return (score1 << 4) | score2;
}

unsigned short get_lives(int lives)
{
  if (lives == 1)
    return 1;
  else if (lives == 2)
    return 3;
  else if (lives == 3)
    return 7;
  return 0;
}

vga_gameinfo_t encode_info(int lives, int audio, int score)
{
  vga_gameinfo_t info;
  info.lives = get_lives(lives) & 7;
  info.audio = audio & 0xF;
  info.score = get_score(score) & 0xFF;
  return info;
}

int brick_hit(game_info_t *game_info, int block_x, int block_y)
{
  // Hits bottom of block
  if (game_info->ball_y > block_y + BLOCK_HEIGHT - BALL_RADIUS/2 &&
      game_info->ball_y < block_y + BLOCK_HEIGHT + BALL_RADIUS &&
      game_info->ball_x > block_x - BLOCK_WIDTH - BALL_RADIUS &&
      game_info->ball_x < block_x + BLOCK_WIDTH + BALL_RADIUS &&
      game_info->ball_U == 0)
  {
    printf("bottom\n");
    game_info->ball_U = 1;
    return 1;
  }
  // Hits top of block
  else if (game_info->ball_y < block_y - BLOCK_HEIGHT + BALL_RADIUS/2 &&
           game_info->ball_y > block_y - BLOCK_HEIGHT - BALL_RADIUS &&
           game_info->ball_x > block_x - BLOCK_WIDTH - BALL_RADIUS &&
           game_info->ball_x < block_x + BLOCK_WIDTH + BALL_RADIUS &&
           game_info->ball_U == 1)
  {
    printf("top\n");
    game_info->ball_U = 0;
    return 1;
  }
  // Hits left side of block
  else if (game_info->ball_y > block_y - BLOCK_HEIGHT &&
           game_info->ball_y < block_y + BLOCK_HEIGHT &&
           game_info->ball_x > block_x - BLOCK_WIDTH - 2*BALL_RADIUS &&
           game_info->ball_x < block_x - BLOCK_WIDTH + 2*BALL_RADIUS &&
           game_info->ball_v > 0)
  {
    printf("left side\n");
    game_info->ball_v = -game_info->ball_v;
    return 1;
  }
  // Hits right side of block
  else if (game_info->ball_y > block_y - BLOCK_HEIGHT &&
           game_info->ball_y < block_y + BLOCK_HEIGHT &&
           game_info->ball_x > block_x + BLOCK_WIDTH - 2*BALL_RADIUS &&
           game_info->ball_x < block_x + BLOCK_WIDTH + 2*BALL_RADIUS &&
           game_info->ball_v < 0)
  {
    printf("right side\n");
    game_info->ball_v = -game_info->ball_v;
    return 1;
  }
  return 0;
}

void update_position(game_info_t *game_info)
{
  int audio = 0;
  for (uint32_t i = 0; i < 8; i++)
  {
    if (game_info->blocks & (1 << i) && brick_hit(game_info, 47 + i * 78, 48))
    {
      game_info->blocks &= ~(1 << i);
      game_info->score += 4;
      audio = BRICK_HIT_SOUND;
    }
    if (game_info->blocks & (1 << (i+8)) && brick_hit(game_info, 47 + i * 78, 90))
    {
      game_info->blocks &= ~(1 << (i + 8));
      game_info->score += 3;
      audio = BRICK_HIT_SOUND;
    }
    if (game_info->blocks & (1 << (i+16)) && brick_hit(game_info, 47 + i * 78, 132))
    {
      game_info->blocks &= ~(1 << (i + 16));
      game_info->score += 2;
      audio = BRICK_HIT_SOUND;
    }
    if (game_info->blocks & (1 << (i+24)) && brick_hit(game_info, 47 + i * 78, 174))
    {
      game_info->blocks &= ~(1 << (i + 24));
      game_info->score += 1;
      audio = BRICK_HIT_SOUND;
    }
  }
  // Ball hits side of paddle
  if ((game_info->ball_y >= PADDLE_TOP &&
       game_info->ball_x > game_info->paddle_x - 75 &&
       game_info->ball_x < game_info->paddle_x - 45) ||
      (game_info->ball_y >= PADDLE_TOP &&
       game_info->ball_x > game_info->paddle_x + 45 &&
       game_info->ball_x < game_info->paddle_x + 75))
  {
    // if paddle moving same direction as ball then speed up
    if ((game_info->ball_v > 0 && game_info->paddle_v > 0) ||
        (game_info->ball_v < 0 && game_info->paddle_v < 0))
    {
      game_info->ball_v += game_info->paddle_v / 2;
    }
    else
    {
      game_info->ball_v = -game_info->ball_v;
    }
    audio = PADDLE_HIT_SOUND;
  }
  // Ball hits side of screen
  if ((game_info->ball_x < -game_info->ball_v + BALL_RADIUS) ||
      (game_info->ball_x > 620 - game_info->ball_v + BALL_RADIUS))
  {
    game_info->ball_v = -game_info->ball_v;
  }
  if (game_info->ball_U)
  {
    game_info->ball_y += 6;
    // Ball hits top of paddle
    if ((game_info->ball_y > PADDLE_TOP - BALL_RADIUS*2 &&
         game_info->ball_y < PADDLE_TOP &&
         game_info->ball_x > game_info->paddle_x - PADDLE_WIDTH - BALL_RADIUS &&
         game_info->ball_x < game_info->paddle_x + PADDLE_WIDTH + BALL_RADIUS))
    {
      game_info->ball_U = 0;
      game_info->ball_v = game_info->last_ball_v + game_info->paddle_v / 2;
      audio = PADDLE_HIT_SOUND;
    }
    // Ball is out of bounds at bottom
    else if (game_info->ball_y > 460)
    {
      game_info->ball_y = 220;
      game_info->ball_U = 1;
      game_info->ball_v = 4;
      game_info->lives--;
      audio = LIFE_LOST_SOUND;
    }
  }
  // Ball hits top of screen
  else
  {
    game_info->ball_y -= 6;
    if (game_info->ball_y < 20)
      game_info->ball_U = 1;
  }

  game_info->ball_x += game_info->ball_v;
  game_info->last_ball_v = game_info->ball_v;

  vga_positions_t position = encode_position(game_info->ball_x, game_info->ball_y, game_info->paddle_x);
  vga_gameinfo_t info = encode_info(game_info->lives, audio, game_info->score);
  set_position(&position);
  set_info(&info);
  set_blocks(&game_info->blocks);
}

int main()
{
  vga_ball_arg_t vla;
  int i;
  static const char filename[] = "/dev/vga_ball";

  static const vga_ball_color_t colors[] = {
      {0xff, 0x00, 0x00}, /* Red */
      {0x00, 0xff, 0x00}, /* Green */
      {0x00, 0x00, 0xff}, /* Blue */
      {0xff, 0xff, 0x00}, /* Yellow */
      {0x00, 0xff, 0xff}, /* Cyan */
      {0xff, 0x00, 0xff}, /* Magenta */
      {0x80, 0x80, 0x80}, /* Gray */
      {0x00, 0x00, 0x00}, /* Black */
                          // { 0xff, 0xff, 0xff }  /* White */
  };

#define COLORS 8

  printf("VGA ball Userspace program started\n");

  if ((vga_ball_fd = open(filename, O_RDWR)) == -1)
  {
    fprintf(stderr, "could not open %s\n", filename);
    return -1;
  }

  if ((controller = open_controller(&endpoint_address)) == NULL)
  {
    fprintf(stderr, "Did not find a controller\n");
    exit(1);
  }

  printf("initial state: ");
  print_background_color();

  game_info_t game_info;
  game_info.ball_x = 300;
  game_info.ball_y = 220;
  game_info.ball_v = 4;
  game_info.last_ball_v = 4;
  game_info.ball_U = 1;
  game_info.paddle_x = 300;
  game_info.paddle_v = 0;
  game_info.blocks = 0xFFFFFFFF;
  game_info.lives = 3;
  game_info.score = 0;

  while (game_info.lives > 0 && game_info.blocks != 0)
  {
    int result = libusb_interrupt_transfer(controller, endpoint_address, (unsigned char *)&packet, sizeof(packet), &transferred, 20);
    if (result == LIBUSB_SUCCESS && transferred == sizeof(packet))
    {
      if (packet.direction == USB_RIGHT)
      {
        game_info.paddle_v = PADDLE_ACCELERATION;
      }
      else if (packet.direction == USB_LEFT)
      {
        game_info.paddle_v = -PADDLE_ACCELERATION;
      }
    }
    else if (result == LIBUSB_ERROR_TIMEOUT) // Timeout occurred
    {
      if (game_info.paddle_v > 0)
      {
        game_info.paddle_v -= PADDLE_DECELERATION;
      }
      else if (game_info.paddle_v < 0)
      {
        game_info.paddle_v += PADDLE_DECELERATION;
      }
    }
    else
    {
      fprintf(stderr, "libusb_interrupt_transfer error: %s\n", libusb_error_name(result));
    }

    game_info.paddle_x += game_info.paddle_v;
    if (game_info.paddle_x > 640 - PADDLE_WIDTH)
      game_info.paddle_x = 640 - PADDLE_WIDTH;
    if (game_info.paddle_x < PADDLE_WIDTH)
      game_info.paddle_x = PADDLE_WIDTH;

    set_background_color(&colors[7]);
    update_position(&game_info);
    usleep(50000);
  }

  int audio = game_info.blocks == 0 ? GAME_WIN_SOUND : 0;
  vga_gameinfo_t info = encode_info(game_info.lives, audio, game_info.score);
  set_info(&info);
  usleep(50000);
  info = encode_info(game_info.lives, 0, game_info.score);
  set_info(&info);

  printf("VGA BALL Userspace program terminating\n");
  return 0;
}
