import System.Exit(die)
import System.Environment(getArgs, getProgName)
import qualified Data.Set as Set
import Control.Monad(unless)
import Data.List(inits, tails)
import qualified Control.Parallel.Strategies as P hiding(parMap)
import Data.List.Split (chunksOf)

type StringSet = Set.Set String

usage :: IO ()
usage = do
    pn <- getProgName
    die $ "Usage: "++pn++" <dictionary-filename> <from-word> <to-word>"

readDict :: String -> Int -> IO StringSet
readDict filename wordLength =
    (Set.fromList . filter validWord . words) `fmap` readFile filename
    where validWord w = length w == wordLength

parMap :: (a -> b) -> [a] -> [b]
parMap f = P.withStrategy (P.parList P.rseq) . map f

search :: StringSet -> String -> String -> Int -> Maybe [String]
search dictionary fromWord toWord maxDepth =
    search1 [[fromWord]] (Set.singleton fromWord) maxDepth
    where 
        search1 :: [[String]] -> StringSet -> Int -> Maybe [String]
        search1 _ _ 0 = Nothing
        search1 paths visited depth =
            case filter ((==toWord) . head) paths of
                (solution:_) -> Just solution
                [] -> search1 newPaths newVisited (depth-1)
                where
                    paths' = concat $ parMap parTakeAStep paths -- parallelize for each immediate next path from node

                    parTakeAStep [] = error "takeAStep: empty list?"
                    parTakeAStep p@(x:_) = concat $ parMap parAllLetters $ zip (inits x) (tails x) -- parallelize for each index of the word
                        where 
                            -- replace the letter at the given index with a letter from a-z in parallel
                            parAllLetters :: (String, String) -> [[String]]
                            parAllLetters (_, []) = []
                            parAllLetters (pre, suf) = concat $ parMap replaceLetter [(lc, (pre, suf)) | lc <- chunksOf 13 ['a'..'z']]

                            replaceLetter :: ([Char], (String, String)) -> [[String]]
                            replaceLetter (_, (_, []))  = error "replaceLetter: empty suffix?"
                            replaceLetter (lc, (pre, (_:ws))) =
                                [ b : p | c <- lc, let b = (pre ++ [c] ++ ws), Set.member b dictionary, not $ Set.member b visited]

                    (newPaths, newVisited) = foldr validStep ([], visited) paths'
                    validStep np@(w:_) (exisitng, v)
                        |Set.member w dictionary && not (Set.member w v) 
                            = (np:exisitng, Set.insert w v)
                        |otherwise = (exisitng, v)
                    validStep [] _ = error "validStep: empty list?"

maxSteps :: Int
maxSteps = 20

main :: IO ()
main = do
    args <- getArgs
    case args of
        [dictFile, fromWord, toWord] -> do
            unless (length fromWord == length toWord) $ do
                die $ "Words must be the same length"
            dictionary <- readDict dictFile (length fromWord)
            unless (Set.member fromWord dictionary) $ die $
                "Word \""++fromWord++"\" not in dictionary"
            unless (Set.member toWord dictionary) $ die $
                "Word \""++toWord++"\" not in dictionary"
            let solution = search dictionary fromWord toWord maxSteps

            case solution of
                Nothing -> die "No solution found"
                Just sol -> do
                    mapM_ putStrLn $ reverse sol
        _ -> usage
