module Lib
    (
        parseLine,
        readLines,
        buildGraph,
        findAugmentingPath,
        bottleneckCapacity,
        addFlow,
        fordFulkerson,
    ) where

import Data.List.Split (splitOn)
import Data.Maybe (fromJust)
import Data.Set (Set)
import Data.Map (Map)
import qualified Data.Set as Set
import qualified Data.Map as Map

import qualified Data.Sequence as Seq
import Data.Sequence ((><))

parseLine :: String -> (Int, Int, Int)
parseLine line = (read src, read dest, read weight)
    where 
        [src, dest, weight] = case splitOn " " line of 
            [x, y, z] -> [x, y, z]
            _ -> error "Invalid input"

readLines :: FilePath -> IO [(Int, Int, Int)]
readLines filePath = do
    fileContent <- readFile filePath
    let fileLines = lines fileContent
    return (map parseLine fileLines)

type Graph = ([Int], Map Int (Set Int), Map (Int, Int) Int)

buildGraph :: [(Int, Int, Int)] -> Graph
buildGraph edges = (vs, es, cs)
    where
        vs = Set.toList $ Set.fromList $ concat [[s, t] | (s, t, _) <- edges]
        es = Map.fromList 
                [
                    (v, Set.union 
                            (Set.fromList [ s | (s, t, _) <- edges, t == v]) 
                            (Set.fromList [ t | (s, t, _) <- edges, s == v])) 
                    | v <- vs 
                ]
        cs' = foldr (\(src, dst, c) m -> Map.insert (src, dst) c m) Map.empty edges
        cs = foldr (\(src, dst, _) m -> Map.insert (dst, src) 0 m) cs' edges

findAugmentingPath :: Graph -> Int -> Int -> Maybe [(Int, Int, Int)]
findAugmentingPath (_, es, cs) s t = go (Seq.singleton s) Set.empty Map.empty
    where
        getPath v prevLookup 
            | v == s = []
            | otherwise = (dst, src, wei) : getPath src prevLookup
                where (dst, src, wei) = fromJust $ Map.lookup v prevLookup
        go pathSeq visited prevLookup
            | Seq.length pathSeq == 0 = Nothing
            | v == t = Just $ getPath t prevLookup
            | Set.member v visited = go path visited prevLookup
            | otherwise =
                let 
                    neighbors = Seq.fromList 
                        $ filter (\v2 -> (Map.findWithDefault 0 (v, v2) cs) > 0 && 
                                    (not $ Set.member v2 visited)) 
                            $ Set.toList $ Map.findWithDefault Set.empty v es
                    prevLookup' = foldr 
                        (\v2 m -> Map.insert v2 (v2, v, (Map.findWithDefault 0 (v, v2) cs)) m) 
                            prevLookup neighbors
                in go (path >< neighbors) (Set.insert v visited) prevLookup'
                where 
                    v = Seq.index pathSeq 0
                    path = Seq.deleteAt 0 pathSeq

bottleneckCapacity :: [(Int, Int, Int)] -> Int
bottleneckCapacity path = minimum (map (\(_, _, c) -> c) path)

addFlow :: Int -> [(Int, Int, Int)] -> Map (Int, Int) Int -> Map (Int, Int) Int
addFlow f path cs = go path cs
    where
        go [] cs' = cs'
        go ((dst, src, _):path') csTemp =
            let capacity = Map.findWithDefault 0 (src, dst) csTemp
                capacity' = Map.findWithDefault 0 (dst, src) csTemp
                cs' = Map.insert (src, dst) (capacity - f) cs
                cs'' = Map.insert (dst, src) (capacity' + f) cs'
            in go path' cs''

fordFulkerson :: Graph -> Int -> Int -> Int
fordFulkerson (vs, es, cs) s t = go 0 cs
    where
        go f cs' =
            let p = findAugmentingPath (vs, es, cs') s t
            in case p of
                Nothing -> f
                Just path ->
                    let c = bottleneckCapacity path
                        f' = f + c
                        cs'' = addFlow c path cs'
                    in go f' cs''

