module NQueens ( nQueensParallel, nQueensSequential ) where

import Data.Bits
import qualified Data.Map as Map
import Control.Parallel.Strategies(parList, rseq, r0, Strategy, runEval)

-- | The strategy for parallelizing the n-queens problem.
strategy :: Int -> Int -> [Word] -> Strategy [Int]
strategy depth number rows 
    | depth < number - length rows = r0
    | otherwise = parList rseq

-- | A list of the positions that a queen can occupy on a given row.
occupiable :: Word -> Word -> Word -> [Word]
occupiable l c r = (l' .|. c .|. r') : occupiable l' c r'
    where l' = shift l (-1)
          r' = shift r 1

-- | A map of the positions that a queen can occupy on a given row.
unsafeBoard :: Int -> Map.Map Int [Word]
unsafeBoard n = Map.fromAscList [ (x, occupiable bx bx bx) | x <- [0..n-1], let bx = bit x ]

-- | Prune the search tree if the number of queens on the board is greater than
--   the number of remaining rows.
prune :: Int -> [Word] -> Bool
prune n b = colsPruned && rowsPruned
    where 
        colsPruned = n >= length b + (popCount $ foldr (.&.) ((bit n) - 1 :: Word) b)
        rowsPruned = notElem ((bit n) - 1 :: Word) $ map (.&. ((bit n) - 1 :: Word)) b

-- | Add a queen to the board.
addQueen :: Int -> [Word] -> Int -> [Word]
addQueen n rows x = zipWith (.|.) rows $ (unsafeBoard n) Map.! x

-- | Count the number of solutions to the n-queens problem.
countMap :: Int -> [Word] -> ([Word] -> Int) -> [Int]
countMap _ [] _ = []
countMap n (row:rows) countDFS = map countDFS $ filter (prune n) (map (addQueen n rows) $ filter (not . testBit row) [0..n-1])

nQueensSequential :: Int -> Int
nQueensSequential n = countDFS $ replicate n 0
  where
    countDFS [] = 1
    countDFS (row:rows) = sum $ (countMap n (row:rows) countDFS)

nQueensParallel :: Int -> Int -> Int
nQueensParallel d n = countDFS $ replicate n 0
  where
    countDFS [] = 1
    countDFS (row:rows) = sum (runEval ((strategy d n rows) (countMap n (row:rows) countDFS)))
