module ParLib
    ( convexHull
    ) where

import qualified Data.List as List
import Control.Parallel.Strategies
import Control.DeepSeq
import Types (Point(..), Cap(..), Hull(..))

{- get the convexity of an ordered triplet of points
 - by computing a quantity proportional to the signed
 - area of the triangle formed by these points
 - this is > 0 when convex, < 0 when concave and
 - = 0 for collinear points 
 -}
convexity :: Point -> Point -> Point -> Double
convexity (Point ax ay) 
          (Point bx by) 
          (Point cx cy) = (ax * by + bx * cy + cx * ay) - 
                          (ax * cy + bx * ay + cx * by)

{- find the upper hull of an xy-sorted set of points 
 - maintain a stack and enforce convexity at insertion
 -}
upperHull :: [Point] -> [Point]
upperHull xs = List.foldl insertConvex [] xs -- for every point
  where insertConvex :: [Point] -> Point -> [Point]
        insertConvex stack@(a : pop@(b : _)) p =
          if convexity b a p < 0
          then p : stack          -- add on top of stack
          else insertConvex pop p -- pop and retry
        insertConvex stack p = 
          p : stack

{- find the lower hull of an xy-sorted set of points 
 - maintain a stack and enforce cocavity at insertion
 -}
lowerHull :: [Point] -> [Point]
lowerHull xs = List.foldl insertConcav [] xs -- for every point
  where insertConcav :: [Point] -> Point -> [Point]
        insertConcav stack@(a : pop@(b : _)) p =
          if convexity b a p > 0
          then p : stack          -- add on top of stack
          else insertConcav pop p -- pop and retry
        insertConcav stack p = 
          p : stack

{- compute the lower and upper hull using the naive
 - Graham's scan algorithm and then wrap them into
 - a "Hull" instance
 -}
convexHullNaive :: [Point] -> Hull
convexHullNaive xs =
  let lower = lowerHull xs in
  let upper = upperHull xs in
  Hull (Cap (reverse lower) lower) (Cap (reverse upper) upper)

{- find the common upper tangent of two hulls
 - the input consists of the list starting at the
 - rightmost point of the left hull, followed by
 - the list starting at the leftmost point of the
 - right hull
 - algorithm: shift the tangent incrementally 
 - until the optimum is reached
 -}
upperTangent :: [Point] -> [Point] -> ([Point], [Point])
upperTangent xl xr
  -- can shift tangent to the right
  | (l : _) <- xl
  , (r : xrs@(rnext : _)) <- xr
  , convexity l r rnext > 0 = upperTangent xl xrs
  -- can shift tangent to the left
  | (l : xls@(lnext : _)) <- xl
  , (r : _) <- xr
  , convexity lnext l r > 0 = upperTangent xls xr
  -- optimum reached
  | otherwise = (xl, xr)

{- similar to upperTangent, but checks for concavity
 - instead of convexity
 -}
lowerTangent :: [Point] -> [Point] -> ([Point], [Point])
lowerTangent xl xr
  -- can shift tangent to the right
  | (l : _) <- xl
  , (r : xrs@(rnext : _)) <- xr
  , convexity l r rnext < 0 = lowerTangent xl xrs
  -- can shift tangent to the left
  | (l : xls@(lnext : _)) <- xl
  , (r : _) <- xr
  , convexity lnext l r < 0 = lowerTangent xls xr
  -- optimum reached
  | otherwise = (xl, xr)

{- to merge two hulls, compute their common upper tangent
 - and lower tangent, then reconstruct the resulting hull
 -}
mergeHulls :: Hull -> Hull -> Hull
mergeHulls (Hull (Cap _ lowerRL) (Cap _ upperRL))
           (Hull (Cap lowerLR _) (Cap upperLR _)) =
  -- compute upper/lower tangents
  let (lowerL, lowerR) = lowerTangent lowerRL lowerLR in
  let (upperL, upperR) = upperTangent upperRL upperLR in
  -- combine into a `Hull` instance
  let lower = (reverse lowerL) ++ lowerR in
  let upper = (reverse upperL) ++ upperR in
  Hull (Cap lower $ reverse lower) (Cap upper $ reverse upper)

{- recursively partition the input array; call naive
 - convex hull algorithm on leaves; merge left and right
 - hulls at internal nodes
 -}
convexHullHelper :: [Point] -> Hull
convexHullHelper xs =
  if length xs <= 16000
  then convexHullNaive $ quicksort xs
  else -- partition list of points into left/right halves
       let pivot = head xs in
       let (ls, _, rs) = partition xs pivot [] [] [] in
        
       -- compute the convex hulls of the two halves in parallel
       let lhull = convexHullHelper ls in
       let rhull = convexHullHelper rs in
       let parRes = do lhull' <- rpar (force lhull)
                       rhull' <- rpar (force rhull)
                       _ <- rseq lhull'
                       _ <- rseq rhull'
                       return (lhull', rhull') in
       let (lhull', rhull') = runEval $ parRes in

       -- merge the left hull, the pivot and the right hull
       lhull' `mergeHulls` (Hull (Cap [pivot] [pivot]) (Cap [pivot] [pivot]))
              `mergeHulls` rhull'

{- very simple quicksort method used in the leaves of
 - the helper above; for some reason, this is faster
 - than the library List.sort
 -}
quicksort :: (NFData a, Ord a) => [a] -> [a]
quicksort [] = []
quicksort (x : xs) = 
  let l = quicksort [y | y <- xs, y <= x] in
  let r = quicksort [y | y <- xs, y > x] in
  l ++ [x] ++ r

{- partition function used in the convexHullHelper;
 - splits a list into elements less than, equal to, 
 - and greater than a given pivot
 -}
partition :: Ord a => [a] -> a -> [a] -> [a] -> [a] -> ([a], [a], [a])
partition [] _ lt eq gt = (lt, eq, gt)
partition (x : xs) pivot lt eq gt =
  case x `compare` pivot of
      LT -> partition xs pivot (x : lt) eq gt
      EQ -> partition xs pivot lt (x : eq) gt
      GT -> partition xs pivot lt eq (x : gt)

{- convert a Hull to list-of-point representation
 - since the upper/lower hulls have two common endpoints,
 - we have to remove these duplicates before concatenating
 - their associated lists; because of Haskell laziness,
 - the use of `init` and `++` should not decrease the
 - performance by a lot
 -}
toList :: Hull -> [Point]
toList (Hull (Cap lowerLR _) (Cap _ upperRL)) = 
  (init lowerLR) ++ (init upperRL)

{- entry point into the convex-hull library
 - calls the underlying helper method and converts the output to 
 - a list-of-point format
 -}
convexHull :: [Point] -> [Point]
convexHull xs = 
  toList $ convexHullHelper xs

