module Lib
    (convexHull
    ) where

import qualified Data.List as List
import Types (Point(..))

{- get the convexity of an ordered triplet of points
 - by computing a quantity proportional to the signed
 - area of the triangle formed by these points
 - this is > 0 when convex, < 0 when concave and
 - = 0 for collinear points 
 -}
convexity :: Point -> Point -> Point -> Double
convexity (Point ax ay) 
          (Point bx by) 
          (Point cx cy) = (ax * by + bx * cy + cx * ay) - 
                          (ax * cy + bx * ay + cx * by)

{- find the upper hull of an xy-sorted set of points 
 - maintain a stack and enforce convexity at insertion
 -}
upperHull :: [Point] -> [Point]
upperHull xs = List.foldl insertConvex [] xs
  where insertConvex :: [Point] -> Point -> [Point]
        insertConvex stack@(a : pop@(b : _)) p =
          if convexity b a p < 0
          then p : stack
          else insertConvex pop p
        insertConvex stack p = 
          p : stack

{- find the lower hull of an xy-sorted set of points 
 - maintain a stack and enforce cocavity at insertion
 -}
lowerHull :: [Point] -> [Point]
lowerHull xs = List.foldl insertConcav [] xs
  where insertConcav :: [Point] -> Point -> [Point]
        insertConcav stack@(a : pop@(b : _)) p =
          if convexity b a p > 0
          then p : stack
          else insertConcav pop p
        insertConcav stack p = 
          p : stack

{- compute the lower and upper hull and concatenate
 - them; the two hulls have two common endpoints, so
 - remove these from the upper hull before appending
 -}
convexHull :: [Point] -> [Point]
convexHull xs =
  let xsSorted = quicksort xs in
  let lower = lowerHull xsSorted in
  let upper = upperHull xsSorted in
  combine lower upper
  where combine :: [Point] -> [Point] -> [Point]
        combine as bs = 
          let ta = List.reverse .tail $ as in
          let tb = init bs in
          ta ++ tb

{- very simple quicksort method used in the leaves of
 - the helper above; for some reason, this is faster
 - than the library List.sort
 -}
quicksort :: Ord a => [a] -> [a]
quicksort [] = []
quicksort (x : xs) = 
  let l = quicksort [y | y <- xs, y <= x] in
  let r = quicksort [y | y <- xs, y > x] in
  l ++ [x] ++ r
