module Main (main) where

import qualified Data.Text as B (words, lines)
import qualified Data.Text.IO as B (readFile)
import qualified Data.Text.Read as B (double)
import Types (Point(..))
import qualified Lib (convexHull)
import qualified ParLib (convexHull)
import System.Environment (getArgs, getProgName)
import System.Exit (exitFailure)
import System.IO (hPutStrLn, stderr, withFile, IOMode(WriteMode))

main :: IO ()
main = do
  args <- getArgs
  case args of
    [mode, inputFilename, outputFilename] -> do
      contents <- B.readFile inputFilename
      let pts = map (\p -> case p of
                           [xstr, ystr] -> (case (B.double xstr, B.double ystr) of
                                              (Right (x, _), Right (y, _)) -> 
                                                 Point x y
                                              _ -> error "error while parsing input"
                                           )
                           _            -> error "malformed input"
                    ) 
                    (map B.words $ B.lines contents)
      let hull = case mode of
                   "p" -> ParLib.convexHull pts
                   "s" -> Lib.convexHull pts
                   _   -> error "invalid mode"
      withFile outputFilename WriteMode 
               (\h -> do mapM_ (\p -> hPutStrLn h $ show p) $ hull)
      
    _ -> do
      pn <- getProgName
      hPutStrLn stderr $ "Usage: " ++ 
                         pn ++
                         "<mode(p/s)> <input-filename> <output-filename>"
      exitFailure
