{-
This is a sequential implementation of the Mini-Max algorithm,
essentially a BFS-Style algorithm.

The purpose of this file is to show that even with ALPHA-BETA pruning,
the omitting of parallelization will undoubtedly affect the efficiency of the algorithm.

-}
import System.Exit
import System.Environment
import Control.Parallel.Strategies
import Data.List
import Control.Monad





data Piece =  WKing | WQueen | WBishop | WKnight | WRook | WPawn 
            | BKing | BQueen | BBishop | BKnight | BRook | BPawn 
            | Empty deriving (Eq, Show)

-- Define the type for a chess board, represented as a list of lists of pieces
type Board = [[Piece]]

-- Define the type for a move, represented as the starting and ending positions on the board
type Move = ((Int, Int), (Int, Int))

outPutBoard :: Board -> IO ()
outPutBoard board = do
    mapM_ print (reverse board)

getPoints :: Piece -> Int
getPoints piece
 | piece == WKing   || piece == BKing   = 9999
 | piece == WQueen  || piece == BQueen  = 9
 | piece == WBishop || piece == BBishop = 3
 | piece == WKnight || piece == BKnight = 3
 | piece == WRook   || piece == BRook   = 5
 | piece == WPawn   || piece == BPawn   = 1
 | otherwise = 0

sumPieceList :: [Piece] -> Int
sumPieceList [] = 0
sumPieceList (x:xs) = (getPoints x) + (sumPieceList xs)


-- Function to evaluate the value of a given board
-- Higher values indicate a better position for the player
-- Negative (better for -1 black) 
-- positive (better for +1 white)
--Possible algorithm, how much total material is there on the board for each player? King is negligible
-- pawn is 1
-- knight and bishop are 3
-- Rook is 5
-- Queen is 9
--Basically just counts the piece count.
evalBoard :: Board -> Int
evalBoard board = whiteCount - blackCount
    where 
        whiteList = [board !! x !! y | x <- [0..7], y <- [0..7], snd (isOccupied board (x, y)) == 1]
        blackList = [board !! x !! y | x <- [0..7], y <- [0..7], snd (isOccupied board (x, y)) == (-1)]
        whiteCount = sumPieceList whiteList
        blackCount = sumPieceList blackList




--Checks if an index on the board is occupied and if its within bounds.
-- IF its not in bounds, a tuple will be returned with the int as code 999, which signifies an out-of bounds
-- call without stopping the board evaluation.

isOccupied :: Board -> (Int, Int) -> (Bool, Int)
isOccupied board (x, y)
 | x < 0 || x > 7 || y < 0 || y > 7 = (True, 999)
 | (board !! x) !! y == Empty = (False, 0)
 | head (show piece) == 'W' = (True, 1)
 | otherwise = (True, -1)
    where
        piece = board !! x !! y


--Custom function that traverses a list, and returns a tuple of the list and the index at which it had to stop.
takeWhileInclusive :: (a -> Bool) -> [a] -> [a]
takeWhileInclusive _ []  = []
takeWhileInclusive p (x:xs) = x : if p x then takeWhileInclusive p xs
                                         else []

checkLastElemCap :: Board -> Int -> [Move] -> [Move]
checkLastElemCap _ _ [] = []
checkLastElemCap board player list
 | snd (isOccupied board (snd (last list))) == (-player) = list
 | otherwise = fst (splitAt (length list - 1) list)


--Wrote this while working on rook moves, is applicable to any list of moves for any piece (including knight). Takes board, int for player designation, 
getListOfMoves :: Board -> Int -> [Move] -> [Move]
getListOfMoves board player moves = checkLastElemCap board player ((takeWhileInclusive (\((r, c), (r', c')) -> (fst (isOccupied board (r', c')) == False)) moves))


--Generates legal moves for Pawns, regardless of whether or not the pawn is white or black. 
--This is essentially a filter function that takes a position and returns the legal moves for the piece at a set of coordinates.
-- Either a WPawn or a BPawn will be passed in.


pawnLegalMoves :: Board -> Int -> Piece -> (Int, Int) -> [Move]
pawnLegalMoves board player piece (row, col)
 | piece == WPawn = [x | x <- wMoveList, let check = snd (isOccupied board (snd x)), check == 0 && check /= 999] ++ 
                    [y | y <- wCapList, let check = snd (isOccupied board (snd y)), check == (-player) && check /= 999]
 | piece == BPawn = [x | x <- bMoveList, let check = snd (isOccupied board (snd x)), check == 0 && check /= 999] ++ 
                    [y | y <- bCapList, let check = snd (isOccupied board (snd y)), check == (-player) && check /= 999]
 | otherwise = []
    where 
        wMoveList = [((row, col),(row + 1, col))] 
        wCapList =  [((row, col),(row + 1, col + 1)), ((row, col),(row + 1, col - 1))]
        bMoveList = [((row, col),(row - 1, col))] 
        bCapList =  [((row, col),(row - 1, col - 1)), ((row, col),(row - 1, col + 1))]


rookLegalMoves :: Board -> Int -> Piece -> (Int, Int) -> [Move]
rookLegalMoves board player piece (row, col)
 | piece == WRook || piece == WQueen = (getListOfMoves board player vertNegMoveList) ++ (getListOfMoves board player vertPosMoveList) ++ (getListOfMoves board player horiNegMoveList) ++ (getListOfMoves board player horiPosMoveList)
 | piece == BRook || piece == BQueen = (getListOfMoves board player vertNegMoveList) ++ (getListOfMoves board player vertPosMoveList) ++ (getListOfMoves board player horiNegMoveList) ++ (getListOfMoves board player horiPosMoveList)
 | otherwise = []
    where
        vertNegMoveList = [((row, col),(row - 1, col)), ((row, col),(row - 2, col)), ((row, col),(row - 3, col)), ((row, col),(row - 4, col)), ((row, col),(row - 5, col)), ((row, col),(row - 6, col)), ((row, col),(row - 7, col))]           
        vertPosMoveList = [((row, col),(row + 1, col)), ((row, col),(row + 2, col)), ((row, col),(row + 3, col)), ((row, col),(row + 4, col)), ((row, col),(row + 5, col)), ((row, col),(row + 6, col)), ((row, col),(row + 7, col))]       
        horiNegMoveList = [((row, col),(row, col - 1)), ((row, col),(row, col - 2)), ((row, col),(row, col - 3)), ((row, col),(row, col - 4)), ((row, col),(row, col - 5)), ((row, col),(row, col - 6)), ((row, col),(row, col - 7))]             
        horiPosMoveList = [((row, col),(row, col + 1)), ((row, col),(row, col + 2)), ((row, col),(row, col + 3)), ((row, col),(row, col + 4)), ((row, col),(row, col + 5)), ((row, col),(row, col + 6)), ((row, col),(row, col + 7))] 

--This move list can just use a simple list comprehension: If the move is on a square that is empty or the opposite color or inbounds, it counts.
knightLegalMoves :: Board -> Int -> Piece -> (Int, Int) -> [Move]
knightLegalMoves board player piece (row, col) 
 | piece == WKnight || piece == BKnight = [x | x <- moveList, let check = snd (isOccupied board (snd x)), check == 0 || check == (-player)]
 | otherwise = []
    where
        moveList = [((row, col),(row + 2, col + 1)), ((row, col),(row + 2, col - 1)), ((row, col),(row - 2, col + 1)), ((row, col),(row - 2, col + 1)), 
                    ((row, col),(row + 1, col + 2)), ((row, col),(row - 1, col + 2)), ((row, col),(row + 1, col - 2)), ((row, col),(row - 1, col - 2))]

bishopLegalMoves :: Board -> Int -> Piece -> (Int, Int) -> [Move]
bishopLegalMoves board player piece (row, col)
 | piece == WBishop || piece == WQueen = (getListOfMoves board player ppMoveList) ++ (getListOfMoves board player pnMoveList) ++ (getListOfMoves board player npMpveList) ++(getListOfMoves board player nnMoveList)
 | piece == BBishop || piece == BQueen = (getListOfMoves board player ppMoveList) ++ (getListOfMoves board player pnMoveList) ++ (getListOfMoves board player npMpveList) ++(getListOfMoves board player nnMoveList)
 | otherwise = []
    where
        ppMoveList = [((row, col),(row + 1, col + 1)), ((row, col),(row + 2, col + 2)), ((row, col),(row + 3, col + 3)), ((row, col),(row + 4, col + 4)), ((row, col),(row + 5, col + 5)), ((row, col),(row + 6, col + 6)), ((row, col),(row + 7, col + 7))]       
        pnMoveList = [((row, col),(row + 1, col - 1)), ((row, col),(row + 2, col - 2)), ((row, col),(row + 3, col - 3)), ((row, col),(row + 4, col - 4)), ((row, col),(row + 5, col - 5)), ((row, col),(row + 6, col - 6)), ((row, col),(row + 7, col - 7))]       
        npMpveList = [((row, col),(row - 1, col + 1)), ((row, col),(row - 2, col + 2)), ((row, col),(row - 3, col + 3)), ((row, col),(row - 4, col + 4)), ((row, col),(row - 5, col + 5)), ((row, col),(row - 6, col + 6)), ((row, col),(row - 7, col + 7))]
        nnMoveList = [((row, col),(row - 1, col - 1)), ((row, col),(row - 2, col - 2)), ((row, col),(row - 3, col - 3)), ((row, col),(row - 4, col - 4)), ((row, col),(row - 5, col - 5)), ((row, col),(row - 6, col - 6)), ((row, col),(row - 7, col - 7))]

queenLegalMoves :: Board -> Int -> Piece -> (Int, Int) -> [Move]
queenLegalMoves board player piece (row, col) 
 | piece == WQueen = (rookLegalMoves board player piece (row, col)) ++ (bishopLegalMoves board player piece (row, col))
 | piece == BQueen = (rookLegalMoves board player piece (row, col)) ++ (bishopLegalMoves board player piece (row, col))
 | otherwise = []

kingLegalMoves :: Board -> Int -> Piece -> (Int, Int) -> [Move]
kingLegalMoves board player piece (row, col)
 | piece == WKing || piece == BKing = [x | x <- moveList, let check = snd (isOccupied board (snd x)), check == 0 || check == (-player)]
 | otherwise = []
    where
        moveList = [((row, col), (row + 1, col)), ((row, col), (row - 1, col)), ((row, col), (row, col + 1)), 
                    ((row, col), (row, col - 1)), ((row, col), (row + 1, col + 1)), ((row, col), (row - 1, col - 1))]

--This one is tricky, because the king cannot move into a square that is being attacked. 
--you may have to generate the list of all acceptable legal moves makeable by the enemy,
-- and then check if the list of king moves is in that list. --Or, just don't move the king at all.
-- that's the easiest way hahaha
-- maybe make the king the last piece you'd ever want to move. like, the engine would never think about `
--Just don't move the king. It's so much easier if you just don't touch it. 


-- Modifies the board to make a new board, with a move being made. 
makeMove :: Board -> Move -> Board
makeMove board ((r, c), (dr, dc)) = 
    let piece = board !! r !! c
        (r', c') = splitAt c (board !! r) 
        (moda, modb) = splitAt r board
        modPreRow =  r' ++ [Empty] ++ (snd (splitAt 1 c'))
        modPreBoard = moda ++ [modPreRow] ++ (snd (splitAt 1 modb))

        (dr', dc') = splitAt dc (modPreBoard !! dr)
        (modc, modd) = splitAt dr modPreBoard
        modPostRow = dr' ++ [piece] ++ (snd (splitAt 1 dc'))
        modPostBoard = modc ++ [modPostRow] ++ (snd (splitAt 1 modd))
    in 
        modPostBoard

generateMovesHelper :: Board -> Int -> [(Piece , (Int, Int))] -> [Move]
generateMovesHelper _ _ [] = []
generateMovesHelper board player (x:xs) = 
                             (pawnLegalMoves board player (fst x) (snd x)) ++ 
                             (knightLegalMoves board player (fst x) (snd x)) ++ 
                             (bishopLegalMoves board player (fst x) (snd x)) ++
                             (rookLegalMoves board player (fst x) (snd x)) ++ 
                             (queenLegalMoves board player (fst x) (snd x)) ++ 
                             (kingLegalMoves board player (fst x) (snd x)) ++
                             (generateMovesHelper board player xs)


--Generates all possible moves for a player (white or black) on a given board.

--Try to make it all one continuous list.
generateMoves :: Board -> Int -> [Move]
generateMoves board player = generateMovesHelper board player pieceList
    where
        pieceList = [((board !! x !! y), (x, y)) | x <- [0..7], y <- [0..7], snd (isOccupied board (x, y)) == player]



-- Minimax function, which takes the current board, the depth of the search,
-- and the player who is currently making a move (1 for player, -1 for opponent)
-- Structure is that for each move, the minimax algorithm is called. you don't actually have to change the minimax algo
-- For each move that is created by generate Moves, run minimax. This is where parallelization occurs. 

--Struggles heavily past a depth of 3.
--for correct usage, depth must be an odd number. This way, the function evaluates the correct player, and returns
--the evaluation for the result of the corresponding player's move as opposed to the opposite player.

minimaxSeq :: Board -> Int -> Int -> Int
minimaxSeq board depth player = 
  if depth == 0
    then evalBoard board
    else
      let moves = generateMoves board player
          -- Recursively evaluate each possible move, using the opposite player.
          -- this applies minimax to every move.
          values = map (\move -> minimaxSeq (makeMove board move) (depth-1) (-player)) moves
      in
        -- If the current player is maximizing, return the maximum value
        if player == 1
          then maximum values
          else minimum values

--Minimax SEQ with ALPHA-BETA pruning
minimaxSeqPrune :: Board -> Int -> Int -> Int -> Int -> Int
minimaxSeqPrune board depth player alpha beta =
  if depth == 0
    then evalBoard board
    else
      let moves = generateMoves board player
          maxBestVal = minBound :: Int
          minBestVal = maxBound :: Int
          values = map (\move -> minimaxSeqPrune (makeMove board move) (depth-1) (-player) alpha beta) moves
          value = maximum values
          --minBest = min minBestVal value
      in
        if player == 1
          then
            let alphBest = (max alpha (max maxBestVal value))
            in 
                if alphBest >= beta
                    then beta
                else
                    max maxBestVal value
          else 
            let minBest = min minBestVal value
                betaBest = min beta minBest
            in 
                if betaBest >= alpha
                    then alpha
                else
                    minBest



--Parallel Function
--SPARKS for every single node (move) generated.
-- Granularity is incredibly high, too costly to be so much less efficient.

minimaxParMap :: Board -> Int -> Int -> Int
minimaxParMap board depth player = 
  if depth == 0
    then evalBoard board
    else
      let moves = generateMoves board player
          -- Recursively evaluate each possible move, using the opposite player.
          -- this applies minimax to every move.
          values = parMap rpar (\move -> minimaxParMap (makeMove board move) (depth-1) (-player)) moves
      in
        -- If the current player is maximizing, return the maximum value
        if player == 1
          then maximum values
          else minimum values

minimaxParGran :: Board -> Int -> Int -> Int
minimaxParGran board depth player= 
    let
        moves = generateMoves board player
        values = parMap rpar (\move -> minimaxSeq (makeMove board move) (depth-1) (-player)) moves
    in 
        if player == 1
         then maximum values
         else minimum values


minimaxParPrune :: Board -> Int -> Int -> Int -> Int -> Int
minimaxParPrune board depth player alpha beta = 
    let
        moves = generateMoves board player
        values = parMap rpar (\move -> minimaxSeqPrune (makeMove board move) (depth-1) (-player) alpha beta) moves
    in 
        if player == 1
         then maximum values
         else minimum values



main :: IO ()
main = do
    --print("Usage: minimax type (1 - minimaxSeq, 2-minimaxSeqPrune, 3-minimaxParMap, 4-minimaxParGran) boardname depth player alpha beta")
    argNames <- getArgs

    let minimax = read (argNames !! 0) :: Integer
        board = [[WRook, WKnight, WBishop, WQueen, WKing, WBishop, WKnight, WRook], 
                 [WPawn, WPawn, WPawn, WPawn, WPawn, WPawn, Empty, WPawn], 
                 [Empty, Empty, Empty, Empty, Empty, Empty, WPawn, Empty], 
                 [Empty, Empty, Empty, Empty, BKnight, Empty, Empty, Empty], 
                 [Empty, Empty, WQueen, WBishop, Empty, Empty, Empty, Empty], 
                 [BKnight, Empty, Empty, BQueen, BBishop, Empty, Empty, Empty], 
                 [BPawn, BPawn, BPawn, Empty, BPawn, BPawn, BPawn, BPawn], 
                 [BRook, Empty, Empty, Empty, BKing, BBishop, Empty, BRook]]
        depth = read (argNames !! 1) :: Int
        player = read (argNames !! 2) :: Int
        alpha = read (argNames !! 3) :: Int
        beta = read (argNames !! 4) :: Int

    when(minimax == 1) $ print (minimaxSeq board depth player)
    when(minimax == 2) $ print (minimaxSeqPrune board depth player alpha beta)
    when(minimax == 3) $ print (minimaxParMap board depth player)
    when(minimax == 4) $ print (minimaxParGran board depth player)
    when(minimax == 5) $ print (minimaxParPrune board depth player alpha beta)

    outPutBoard board

