/*
 * Userspace program that communicates with the delay device driver
 * through ioctls
 *
 * Stephen A. Edwards
 * Columbia University
 */

#include <stdio.h>
#include "delay.h"
#include "hardclip.h"
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include "biquad.h"
#include "fir.h"
#include "vga_ball.h"
#include "eq_coeff.h"

#include <stdlib.h>
#include "usbkeyboard.h"
#include <pthread.h>
#include "keycode.h"
#define KEY_ENTER 0x28
#define KEY_SPACE 0x2c
#define KEY_BACKSPACE 0x2a
#define KEY_RIGHT 0x4f
#define KEY_LEFT 0x50 
#define KEY_1 0x1e
#define KEY_2 0x1f
#define KEY_3 0x20
#define KEY_4 0x21 
#define KEY_Q 0x14
#define KEY_A 0x04
#define KEY_W 0x1a
#define KEY_S 0x16 
#define KEY_E 0x08
#define KEY_D 0x07
#define KEY_R 0x15
#define KEY_F 0X09
#define KEY_T 0x17
#define KEY_G 0x0a
#define KEY_Y 0x1c
#define KEY_H 0x0b
#define KEY_X 0x1b
#define KEY_Z 0x1d
#define KEY_B 0x05 
#define KEY_N 0x11

int delay_fd;
int hardclip_fd;
int biquad_fd;
int fir_fd;
int vga_ball_fd;


struct libusb_device_handle *keyboard;
uint8_t endpoint_address;


uint8_t prevkey_1;
uint8_t prevkey_2;

//preset locations
short unsigned int  preset1=132;
short unsigned int  preset2=188;
short unsigned int  preset3=245;
short unsigned int  preset4=304;
char clippreset=5;

//slider locations
short unsigned int  slider1=215;
short unsigned int  slider2=215;
short unsigned int  slider3=215;
short unsigned int  slider4=215;
short unsigned int  slider5=215;
short unsigned int  slider6=215;



//LUTS 
static const char filename_delay[] = "/dev/delay";
delay_param_t delayparam =     { 0x0000, 4000, 8192  };//bypass, samples, mix
static const char filename_hardclip[] = "/dev/hardclip";
static const hardclip_param_t hardclipparam[] = {//lookup table for hardclip threshold
    {32767,-32767 },   {2750,-2750 },    {2500,-2500 },     {2250,-2250 },     {2000,-2000 },     {1750,-1750 },   {1500,-1500 },     {1250,-1250 },     {1000,-1000 },     {750,-750 },   {500,-500 },  };
static const char filename_biquad[] = "/dev/biquad";
biquad_param_t biquadparam[] = {
    {{ 1024, 0, 0, 0, 0,1024,0,0,0,0, 1024,0,0,0,0, 1024,0,0,0,0,1 }},
    {{ 32767,0,0,0,0,32767,0,0,0,0,32767,0,0,0,0,32767,0,0,0,0,0 }}

  };
static const char filename_fir[] = "/dev/fir";
static const fir_param_t firparam[] = {
    {{  15369, 32767, 26902, 13680, 8056, 3137, -4996, -15891, -22587, -15494, -4256, 586, 349, -1217, 2072, 4979, 1068, -922, -386, 892, 7679, 10773, 3222, -2272, -1960, -2079, -2618, -4738, -5895, -2462, 1068, 3137, 2055, -1784, 539, 6166, 4087, -335, -3707, -7377, -6319, -3771, -3812, -2984, 1224, 3459, 362, -3826, -5345, -5193, -2845, 1512, 3096, 3385, 4847, 3235, -2737, -6434, -4799, -1370, 464, 681, 244, -1173, -3039, -3626, -4104, -4986, -4178, -2408, -1563, -410, 817, 1492, 1380, -23, -1048, -1926, -3127, -3012, -2978, -4067, -4148, -3982, -4033, -2781, -1153, 88, 1414, 2177, 2174, 1624, 325, -546, -613, -1685, -3408, -4182, -3497, -1811, -240, 1122, 2120, 1316, -766, -2120, -3073, -3602, -2442, -831, 142, 546, 593, 447, 10, -461, -78, 437, -179, -1180, -1882, -2309, -2242, -1465, -362, 651, 1373, 1302, 264, -576, -491, 27, 603, 437, -457, -756, -661, -254, 851, 1794, 2255, 2557, 2235, 1455, 759, 27, -471, -739, -956, -973, -854, -885, -858, -861, -1075, -671, 284, 1082, 1845, 2428, 2672, 2666, 2062, 841, -267, -807, -864, -773, -736, -583, -223, 247, 691, 1092, 1536, 2045, 2306, 2014, 1288, 753, 708, 688, 427, -88, -644, -800, -586, -142, 539, 1092, 1187, 953, 552, 596, 1197, 1567, 1543, 1282, 949, 695, 335, -217, -508, -447, -240, 61, 176, 186, 237, 274, 437, 841, 1326, 1784, 1930, 1655, 1451, 1322, 1085, 824, 451, 128, 47, 67, 227, 454, 651, 970, 1177, 1156, 1122, 888, 518, 251, 40, -57, 30, 189, 457, 644, 586, 675, 797, 725, 593, 315, 13, 40, 291, 542, 637, 596, 705, 915, 1027, 942, 824, 915, 1071, 1058, 868, 603, 329, 135, 61, 23, -30, -6, 169, 400, 546, 627, 790, 956, 987, 820, 430, 98, 20, 47, 37, -20, -98, -145, -179, -156, 108, 535, 888, 1136, 1265, 1126, 786, 485, 237, 40, -40, -115, -122, 47, 88, -37, 37, 139, 91, 142, 281, 376, 451, 424, 220, 33, 54, 125, 13, -189, -257, -203, -206, -172, -16, 125, 156, -3, -278, -345, -142, 30, 105, 162, 271, 447, 379, 10, -257, -281, -301, -349, -383, -376, -322, -312, -220, 67, 301, 356, 410, 362, 98, -179, -342, -410, -535, -746, -820, -746, -603, -400, -267, -210, -64, 30, -64, -196, -206, -101, -16, 10, 57, 115, 78, 0, -108, -230, -274, -261, -267, -308, -312, -234, -105, -71, -159, -234, -200, -101, -132, -322, -508, -610, -641, -691, -773, -790, -708, -525, -247, -61, 10, 81, 145, 172, 105, -30, -101, -122, -234, -413, -593, -719, -753, -695, -542, -383, -244, -61, 122, 230, 230, 91, -84, -203, -339, -468, -518, -549, -596, -675, -776, -837, -783, -593, -349, -179, -78, 23, 108, 115, 44, -47, -54, -6, -13, -98, -237, -376, -481, -556, -580, -539, -417, -234, -67, 6, -6, -57, -101, -81, -71, -132, -244, -366, -447, -474, -505, -451, -284, -172, -169, -220, -298, -325, -271, -176, -94, -47, -50, -78, -111, -111, -50, -13, -20, -40, -108, -240, -335, -356, -359, -373, -383, -390, -373, -318, -251, -183, -71, 54, 81, 13}},
    {{  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 586, 349, -1217, 2072, 4979, 1068, -922, -386, 892, 7679, 10773, 3222, -2272, -1960, -2079, -2618, -4738, -5895, -2462, 1068, 3137, 2055, -1784, 539, 6166, 4087, -335, -3707, -7377, -6319, -3771, -3812, -2984, 1224, 3459, 362, -3826, -5345, -5193, -2845, 1512, 3096, 3385, 4847, 3235, -2737, -6434, -4799, -1370, 464, 681, 244, -1173, -3039, -3626, -4104, -4986, -4178, -2408, -1563, -410, 817, 1492, 1380, -23, -1048, -1926, -3127, -3012, -2978, -4067, -4148, -3982, -4033, -2781, -1153, 88, 1414, 2177, 2174, 1624, 325, -546, -613, -1685, -3408, -4182, -3497, -1811, -240, 1122, 2120, 1316, -766, -2120, -3073, -3602, -2442, -831, 142, 546, 593, 447, 10, -461, -78, 437, -179, -1180, -1882, -2309, -2242, -1465, -362, 651, 1373, 1302, 264, -576, -491, 27, 603, 437, -457, -756, -661, -254, 851, 1794, 2255, 2557, 2235, 1455, 759, 27, -471, -739, -956, -973, -854, -885, -858, -861, -1075, -671, 284, 1082, 1845, 2428, 2672, 2666, 2062, 841, -267, -807, -864, -773, -736, -583, -223, 247, 691, 1092, 1536, 2045, 2306, 2014, 1288, 753, 708, 688, 427, -88, -644, -800, -586, -142, 539, 1092, 1187, 953, 552, 596, 1197, 1567, 1543, 1282, 949, 695, 335, -217, -508, -447, -240, 61, 176, 186, 237, 274, 437, 841, 1326, 1784, 1930, 1655, 1451, 1322, 1085, 824, 451, 128, 47, 67, 227, 454, 651, 970, 1177, 1156, 1122, 888, 518, 251, 40, -57, 30, 189, 457, 644, 586, 675, 797, 725, 593, 315, 13, 40, 291, 542, 637, 596, 705, 915, 1027, 942, 824, 915, 1071, 1058, 868, 603, 329, 135, 61, 23, -30, -6, 169, 400, 546, 627, 790, 956, 987, 820, 430, 98, 20, 47, 37, -20, -98, -145, -179, -156, 108, 535, 888, 1136, 1265, 1126, 786, 485, 237, 40, -40, -115, -122, 47, 88, -37, 37, 139, 91, 142, 281, 376, 451, 424, 220, 33, 54, 125, 13, -189, -257, -203, -206, -172, -16, 125, 156, -3, -278, -345, -142, 30, 105, 162, 271, 447, 379, 10, -257, -281, -301, -349, -383, -376, -322, -312, -220, 67, 301, 356, 410, 362, 98, -179, -342, -410, -535, -746, -820, -746, -603, -400, -267, -210, -64, 30, -64, -196, -206, -101, -16, 10, 57, 115, 78, 0, -108, -230, -274, -261, -267, -308, -312, -234, -105, -71, -159, -234, -200, -101, -132, -322, -508, -610, -641, -691, -773, -790, -708, -525, -247, -61, 10, 81, 145, 172, 105, -30, -101, -122, -234, -413, -593, -719, -753, -695, -542, -383, -244, -61, 122, 230, 230, 91, -84, -203, -339, -468, -518, -549, -596, -675, -776, -837, -783, -593, -349, -179, -78, 23, 108, 115, 44, -47, -54, -6, -13, -98, -237, -376, -481, -556, -580, -539, -417, -234, -67, 6, -6, -57, -101, -81, -71, -132, -244, -366, -447, -474, -505, -451, -284, -172, -169, -220, -298, -325, -271, -176, -94, -47, -50, -78, -111, -111, -50, -13, -20, -40, -108, -240, -335, -356, -359, -373, -383, -390, -373, -318, -251, -183, -71, 54, 81, 13}},
    {{  15369, 32767, 26902, 13680, 8056, 3137, -4996, -15891, -22587, -15494, -4256, 586, 349, -1217, 2072, 4979, 1068, -922, -386, 892, 7679, 10773, 3222, -2272, -1960, -2079, -2618, -4738, -5895, -2462, 1068, 3137, 2055, -1784, 539, 6166, 4087, -335, -3707, -7377, -6319, -3771, -3812, -2984, 1224, 3459, 362, -3826, -5345, -5193, -2845, 1512, 3096, 3385, 4847, 3235, -2737, -6434, -4799, -1370, 464, 681, 244, -1173, -3039, -3626, -4104, -4986, -4178, -2408, -1563, -410, 817, 1492, 1380, -23, -1048, -1926, -3127, -3012, -2978, -4067, -4148, -3982, -4033, -2781, -1153, 88, 1414, 2177, 2174, 1624, 325, -546, -613, -1685, -3408, -4182, -3497, -1811, -240, 1122, 2120, 1316, -766, -2120, -3073, -3602, -2442, -831, 142, 546, 593, 447, 10, -461, -78, 437, -179, -1180, -1882, -2309, -2242, -1465, -362, 651, 1373, 1302, 264, -576, -491, 27, 603, 437, -457, -756, -661, -254, 851, 1794, 2255, 2557, 2235, 1455, 759, 27, -471, -739, -956, -973, -854, -885, -858, -861, -1075, -671, 284, 1082, 1845, 2428, 2672, 2666, 2062, 841, -267, -807, -864, -773, -736, -583, -223, 247, 691, 1092, 1536, 2045, 2306, 2014, 1288, 753, 708, 688, 427, -88, -644, -800, -586, -142, 539, 1092, 1187, 953, 552, 596, 1197, 1567, 1543, 1282, 949, 695, 335, -217, -508, -447, -240, 61, 176, 186, 237, 274, 437, 841, 1326, 1784, 1930, 1655, 1451, 1322, 1085, 824, 451, 128, 47, 67, 227, 454, 651, 970, 1177, 1156, 1122, 888, 518, 251, 40, -57, 30, 189, 457, 644, 586, 675, 797, 725, 593, 315, 13, 40, 291, 542, 637, 596, 705, 915, 1027, 942, 824, 915, 1071, 1058, 868, 603, 329, 135, 61, 23, -30, -6, 169, 400, 546, 627, 790, 956, 987, 820, 430, 98, 20, 47, 37, -20, -98, -145, -179, -156, 108, 535, 888, 1136, 1265, 1126, 786, 485, 237, 40, -40, -115, -122, 47, 88, -37, 37, 139, 91, 142, 281, 376, 451, 424, 220, 33, 54, 125, 13, -189, -257, -203, -206, -172, -16, 125, 156, -3, -278, -345, -142, 30, 105, 162, 271, 447, 379, 10, -257, -281, -301, -349, -383, -376, -322, -312, -220, 67, 301, 356, 410, 362, 98, -179, -342, -410, -535, -746, -820, -746, -603, -400, -267, -210, -64, 30, -64, -196, -206, -101, -16, 10, 57, 115, 78, 0, -108, -230, -274, -261, -267, -308, -312, -234, -105, -71, -159, -234, -200, -101, -132, -322, -508, -610, -641, -691, -773, -790, -708, -525, -247, -61, 10, 81, 145, 172, 105, -30, -101, -122, -234, -413, -593, -719, -753, -695, -542, -383, -244, -61, 122, 230, 230, 91, -84, -203, -339, -468, -518, -549, -596, -675, -776, -837, -783, -593, -349, -179, -78, 23, 108, 115, 44, -47, -54, -6, -13, -98, -237, -376, -481, -556, -580, -539, -417, -234, -67, 6, -6, -57, -101, -81, -71, -132, -244, -366, -447, -474, -505, -451, -284, -172, -169, -220, -298, -325, -271, -176, -94, -47, -50, -78, -111, -111, -50, -13, -20, -40, -108, -240, -335, -356, -359, -373, -383, -390, -373, -318, -251, -183, -71, 54, 81, 13}},
    {{  0, 32767, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  };
static const char filename[] = "/dev/vga_ball";
vga_ball_color_t colors= { 0,215, 215,215,215,215,215,215};

void setup(void){
	  
    if ( (delay_fd = open(filename_delay, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", filename_delay);
    return -1;
  }
  if ( (hardclip_fd = open(filename_hardclip, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", filename_hardclip);
    return -1;
  }
  
  if ( (biquad_fd = open(filename_biquad, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", filename_biquad);
    return -1;
  }
  if ( (fir_fd = open(filename_fir, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", filename_fir);
    return -1;
  }
  if ( (vga_ball_fd = open(filename, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", filename);
    return -1;
  }
}

void set_background_color(const vga_ball_color_t *c)
{
  vga_ball_arg_t vla;
  vla.background = *c;
  if (ioctl(vga_ball_fd, VGA_BALL_WRITE_BACKGROUND, &vla)) {
      perror("ioctl(VGA_BALL_SET_BACKGROUND) failed");
      return;
  }
  printf("Sliders: %d %d %d %d %d %d %d %d\n", c->bypass, c->sely, c->sly1,c->sly2,c->sly3,c->sly4,c->sly5,c->sly6);
}

void set_hardclip_param(const hardclip_param_t *c)
{
  hardclip_arg_t hardclipvla;
  hardclipvla.hardclipparam = *c;
  if (ioctl(hardclip_fd, HARDCLIP_WRITE_PARAM, &hardclipvla)) {
      perror("ioctl(HARDCLIP_WRITE_PARAM) failed");
      return;
  }
}

void set_delay_param(const delay_param_t *c)
{
  delay_arg_t delayvla;
  delayvla.delayparam = *c;
  if (ioctl(delay_fd, DELAY_WRITE_PARAM, &delayvla)) {
      perror("ioctl(DELAY_WRITE_PARAM) failed");
      return;
  }
}
void set_biquad_param(const biquad_param_t *c)
{
  biquad_arg_t biquadvla;
  biquadvla.biquadparam = *c;
  if (ioctl(biquad_fd, BIQUAD_WRITE_PARAM, &biquadvla)) {
      perror("ioctl(BIQUAD_WRITE_PARAM) failed");
      return;
  }
}
void set_fir_param(const fir_param_t *c)
{
  fir_arg_t firvla;
  firvla.firparam = *c;
  if (ioctl(fir_fd, FIR_WRITE_PARAM, &firvla)) {
      perror("ioctl(FIR_WRITE_PARAM) failed");
      return;
  }
}


int main()
{
	struct usb_keyboard_packet packet;
	int transferred;
	setup();
	
	/* Open the keyboard */
	if ( (keyboard = openkeyboard(&endpoint_address)) == NULL ) {
		fprintf(stderr, "Did not find a keyboard\n");
		exit(1);
	}
	set_biquad_param(biquadparam);
	
	/* Look for and handle keypresses */
	for (;;) {
		libusb_interrupt_transfer(keyboard, endpoint_address,
			(unsigned char *) &packet, sizeof(packet),
		&transferred, 0);
		if (transferred == sizeof(packet)) {
			
			if (packet.keycode[0] == 0 )
			{
			}
			else if (packet.keycode[0] == prevkey_1 && packet.keycode[1] == 0) 
			{
			}
			else if ( packet.keycode[0] == prevkey_2 )
			{
				
			}
			else if (packet.keycode[0] == 0x29) { // if press esc
			
			}
			else  if (packet.keycode[0] == KEY_1) { //if enter pressed
				colors.sely=preset1;
				set_background_color(&colors);
				set_fir_param(&firparam[0]);

			}			
			else  if (packet.keycode[0] == KEY_2) { //if enter pressed
				colors.sely=preset2;
				set_background_color(&colors);
				set_fir_param(&firparam[1]);

			}
			else  if (packet.keycode[0] == KEY_3) { //if enter pressed
				colors.sely=preset3;
				set_background_color(&colors);
				set_fir_param(&firparam[2]);

			}			
			else  if (packet.keycode[0] == KEY_4) { //if enter pressed
				colors.sely=preset4;
				set_background_color(&colors);
				set_fir_param(&firparam[3]);

			}
			else  if (packet.keycode[0] == KEY_BACKSPACE) { //if backspace
				
			}
			
            /////////////////////////////----------------------------/////////////////////////////////////////
			else  if (packet.keycode[0] == KEY_Q) { //HARDCLIP++
				if(slider1>146){
					slider1-=14;
				clippreset-=1;}
				colors.sly1=slider1;
				set_background_color(&colors);
				set_hardclip_param(&hardclipparam[clippreset]);

			}  
			else  if (packet.keycode[0] == KEY_A) {//HARDCLIP--
				if(slider1<285){
					slider1+=14;
								clippreset+=1;}

				colors.sly1=slider1;
				set_background_color(&colors);
				set_hardclip_param(&hardclipparam[clippreset]);

			}  
            /////////////////////////////----------------------------/////////////////////////////////////////
			
			else  if (packet.keycode[0] == KEY_W) { //DELAY++
				if(slider2>146){
					slider2-=14;
					delayparam.length+=700;
				}
				colors.sly2=slider2;
				set_background_color(&colors);
				set_delay_param(&delayparam);
			}  
			else  if (packet.keycode[0] == KEY_S) { //DELAY--
				if(slider2<285){
					slider2+=14;
					delayparam.length-=700;
				}
				colors.sly2=slider2;
				set_background_color(&colors);
				set_delay_param(&delayparam);
			}  
			else  if (packet.keycode[0] == KEY_B) { //BYPASS DELAY
				delayparam.bypass=1;
				colors.bypass=0;
				set_background_color(&colors);
				set_delay_param(&delayparam);
			}  
			else  if (packet.keycode[0] == KEY_N) { //ENABLE DELAY
				delayparam.bypass=0;
				colors.bypass=1;
				set_background_color(&colors);
				set_delay_param(&delayparam);
			}  
			
			else  if (packet.keycode[0] == KEY_E) { //MIX++
				if(slider3>146){
					slider3-=14;
					delayparam.mix+=1500;
				}
				colors.sly3=slider3;
				set_background_color(&colors);
				set_delay_param(&delayparam);
			}
			else  if (packet.keycode[0] == KEY_D) { //MIX--
				if(slider3<285){
					slider3+=14;
					delayparam.mix-=1500;
				}
				colors.sly3=slider3;
				set_delay_param(&delayparam);
				set_background_color(&colors);
			}

			else  if (packet.keycode[0] == KEY_R) { //BASS++
				if(slider4>146){
					slider4-=14;
					eq_coeff((double)slider4, (double)slider5, (double)slider6, biquadparam);
				}
				colors.sly4=slider4;
				set_background_color(&colors);
				set_biquad_param(biquadparam);
			}
			else  if (packet.keycode[0] == KEY_F) { //BASS--
				if(slider4<285){
					slider4+=14;
					eq_coeff((double)slider4, (double)slider5, (double)slider6, biquadparam);
				}
				colors.sly4=slider4;
				set_background_color(&colors);
				set_biquad_param(biquadparam);
			}

			else  if (packet.keycode[0] == KEY_T) { //MID++
				if(slider5>146){
					slider5-=14;
					eq_coeff((double)slider4, (double)slider5, (double)slider6, biquadparam);
				}
				colors.sly5=slider5;
				set_background_color(&colors);
				set_biquad_param(biquadparam);
			}
			else  if (packet.keycode[0] == KEY_G) { //MID--
				if(slider5<285){
					slider5+=14;
					eq_coeff((double)slider4, (double)slider5, (double)slider6, biquadparam);
				}
				colors.sly5=slider5;
				set_background_color(&colors);
				set_biquad_param(biquadparam);
			}
				
			else  if (packet.keycode[0] == KEY_Y) { //TREBLE++
				if(slider6>146){
					slider6-=14;
					eq_coeff((double)slider4, (double)slider5, (double)slider6, biquadparam);
				}
				colors.sly6=slider6;
				set_background_color(&colors);
				set_biquad_param(biquadparam);
			}
			else  if (packet.keycode[0] == KEY_H) { //TREBLE--
				if(slider6<285){
					slider6+=14;
					eq_coeff((double)slider4, (double)slider5, (double)slider6, biquadparam);
				}
				colors.sly6=slider6;
				set_background_color(&colors);
				set_biquad_param(biquadparam);
			}

			else  if (packet.keycode[0] == KEY_Z) { //BYPASS EQ
				biquadparam[0].biquadcoeff[20]=1;
				set_background_color(&colors);
				set_biquad_param(biquadparam);
			}  
			else  if (packet.keycode[0] == KEY_X) { //ENABLE EQ
				biquadparam[0].biquadcoeff[20]=0;
				set_background_color(&colors);
				set_biquad_param(biquadparam);
			}  
			

			else  if (packet.keycode[0] == KEYU) { //BYPASS EQ
				if(biquadparam[0].biquadcoeff[15] < 16000)
					biquadparam[0].biquadcoeff[15] = (short)(((float)biquadparam[0].biquadcoeff[15]) * 1.2);
				set_background_color(&colors);
				set_biquad_param(biquadparam);
			}  
			else  if (packet.keycode[0] == KEYJ) { //ENABLE EQ
				if(biquadparam[0].biquadcoeff[15] > 100)
					biquadparam[0].biquadcoeff[15] = (short)(((float)biquadparam[0].biquadcoeff[15]) / 1.2);
				set_background_color(&colors);
				set_biquad_param(biquadparam);
			} 

            /////////////////////////////----------------------------/////////////////////////////////////////

			
			

			prevkey_1 = packet.keycode[0];
			prevkey_2 = packet.keycode[1];
		}
	}
	

	return 0;
}
