/* * Device driver for the variable hardclip module
 *
 * A Platform device implemented using the misc subsystem
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod hardclip.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree hardclip.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "hardclip.h"

#define DRIVER_NAME "hardclip"

/* Device registers */
#define HARDCLIP_POSCLIP(x) ((x))
#define HARDCLIP_NEGCLIP(x) ((x)+2)


/*
 * Information about our device
 */
struct hardclip_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
        hardclip_param_t x;
} dev;

/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
 */
static void write_hardclip_param(hardclip_param_t *x)
{
	iowrite16(x->posclip, HARDCLIP_POSCLIP(dev.virtbase) );
	iowrite16(x->negclip, HARDCLIP_NEGCLIP(dev.virtbase) );
	dev.x = *x;
}

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long hardclip_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	hardclip_arg_t vla;

	switch (cmd) {
	
	
	case HARDCLIP_WRITE_PARAM:
		if (copy_from_user(&vla, (hardclip_arg_t *) arg,
				   sizeof(hardclip_arg_t)))
			return -EACCES;
		write_hardclip_param(&vla.hardclipparam);
		break;
		
		
	case HARDCLIP_READ_PARAM:
	  	vla.hardclipparam = dev.x;
		if (copy_to_user((hardclip_arg_t *) arg, &vla,
				 sizeof(hardclip_arg_t)))
			return -EACCES;
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations hardclip_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = hardclip_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice hardclip_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &hardclip_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init hardclip_probe(struct platform_device *pdev)
{
        hardclip_param_t beige = { 32767, -32767 };
	int ret;

	/* Register ourselves as a misc device: creates /dev/hardclip */
	ret = misc_register(&hardclip_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
        
	/* Set an initial color */
        write_hardclip_param(&beige);

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&hardclip_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int hardclip_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&hardclip_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id hardclip_of_match[] = {
	{ .compatible = "csee4840,hardclip-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, hardclip_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver hardclip_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(hardclip_of_match),
	},
	.remove	= __exit_p(hardclip_remove),
};

/* Called when the module is loaded: set things up */
static int __init hardclip_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&hardclip_driver, hardclip_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit hardclip_exit(void)
{
	platform_driver_unregister(&hardclip_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(hardclip_init);
module_exit(hardclip_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("LILONGYINIUBI, Columbia University");
MODULE_DESCRIPTION("hardclip driver");
