/* * Device driver for the variable delay module
 *
 * A Platform device implemented using the misc subsystem
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod delay.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree delay.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "delay.h"

#define DRIVER_NAME "delay"

/* Device registers */
#define DELAY_BYPASS(x) (x) //bypass, active high
#define DELAY_LENGTH(x) ((x)+2) //number of samples to delay
#define DELAY_MIX(x) ((x)+4) //mixing, 1=-6dB, 2=-12dB, etc.


/*
 * Information about our device
 */
struct delay_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
        delay_param_t x;
} dev;

/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
 */
static void write_delay_param(delay_param_t *x)
{
	iowrite16(x->bypass, DELAY_BYPASS(dev.virtbase) );
	iowrite16(x->length, DELAY_LENGTH(dev.virtbase) );
	iowrite16(x->mix, DELAY_MIX(dev.virtbase) );
	dev.x = *x;
}

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long delay_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	delay_arg_t vla;

	switch (cmd) {
	
	
	case DELAY_WRITE_PARAM:
		if (copy_from_user(&vla, (delay_arg_t *) arg,
				   sizeof(delay_arg_t)))
			return -EACCES;
		write_delay_param(&vla.delayparam);
		break;
		
		
	case DELAY_READ_PARAM:
	  	vla.delayparam = dev.x;
		if (copy_to_user((delay_arg_t *) arg, &vla,
				 sizeof(delay_arg_t)))
			return -EACCES;
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations delay_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = delay_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice delay_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &delay_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init delay_probe(struct platform_device *pdev)
{
        delay_param_t beige = { 1, 4096, 1 };
	int ret;

	/* Register ourselves as a misc device: creates /dev/delay */
	ret = misc_register(&delay_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
        
	/* Set an initial color */
        write_delay_param(&beige);

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&delay_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int delay_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&delay_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id delay_of_match[] = {
	{ .compatible = "csee4840,delay-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, delay_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver delay_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(delay_of_match),
	},
	.remove	= __exit_p(delay_remove),
};

/* Called when the module is loaded: set things up */
static int __init delay_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&delay_driver, delay_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit delay_exit(void)
{
	platform_driver_unregister(&delay_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(delay_init);
module_exit(delay_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("LILONGYINIUBI, Columbia University");
MODULE_DESCRIPTION("delay driver");
