/* * Device driver for the PWM motor control
 *
 * A Platform device implemented using the misc subsystem
 *
 Alexandre Msellati
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod pwm.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree pwm.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "pwm.h"

#define DRIVER_NAME "pwm"

/* Device registers */

#define ROT_DIR(x) (x)
#define DUTY_CMD(x) (x + 2)
#define ENC_CNT(x) (x + 4)
// #define ENC_CNT(x) (x + 3)
#define ADC_VAL(x) (x + 6)





/*
 * Information about our device
 */
struct pwm_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
		controller_cmd_t command;
		unsigned short encoder_count, adc_value;
} dev;



/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
 */
static void write_command(controller_cmd_t *command)
{
	iowrite16(command->duty_cycle, DUTY_CMD(dev.virtbase) );
	iowrite16(command->rotation_direction, ROT_DIR(dev.virtbase) );
	dev.command = *command;
}

static void read_encoder_count(unsigned short *encoder_count)
{
	*encoder_count = ioread16(ENC_CNT(dev.virtbase));
	dev.encoder_count = *encoder_count;
}

static void read_adc_value(unsigned short *adc_value)
{
	*adc_value = ioread16(ADC_VAL(dev.virtbase));
	dev.adc_value = *adc_value;
}


/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long pwm_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	controller_cmd_t command;
	unsigned short encoder_count, adc_value;

	switch (cmd) {

	case PWM_WRITE_CMD:
		if (copy_from_user(&command, (controller_cmd_t *) arg,
				   sizeof(controller_cmd_t)))
			return -EACCES;
		write_command(&command);
		break;
	case PWM_READ_CNT:
	  	encoder_count = dev.encoder_count;
		if (copy_to_user((unsigned short *) arg, &encoder_count,
				 sizeof(unsigned short)))
			return -EFAULT;
		read_encoder_count(&encoder_count);
		break;

	case PWM_READ_ADC:
	  	adc_value = dev.adc_value;
		if (copy_to_user((unsigned short *) arg, &adc_value,
				 sizeof(unsigned short)))
			return -EFAULT;
		read_adc_value(&adc_value);
		break;

	default:
		return -EINVAL;
	}

	return 0;
}


/* The operations our device knows how to do */
static const struct file_operations pwm_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = pwm_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice pwm_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &pwm_fops,
};



/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init pwm_probe(struct platform_device *pdev)
{	
	controller_cmd_t command_ini = {2000, 1};  // No duty cycle and rotation for acceleration
	int ret;

	/* Register ourselves as a misc device: creates /dev/vga_ball */
	ret = misc_register(&pwm_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
        
	/* Set an initial command */
		write_command(&command_ini);
        pr_info(DRIVER_NAME ": Writing initial command \n");
	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&pwm_misc_device);
	return ret;
}


/* Clean-up code: release resources */
static int pwm_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&pwm_misc_device);
	return 0;
}


/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id pwm_of_match[] = {
	{ .compatible = "csee4840,pwm-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, pwm_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver pwm_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(pwm_of_match),
	},
	.remove	= __exit_p(pwm_remove),
};


/* Called when the module is loaded: set things up */
static int __init pwm_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&pwm_driver, pwm_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit pwm_exit(void)
{
	platform_driver_unregister(&pwm_driver);
	pr_info(DRIVER_NAME ": exit\n");
}


module_init(pwm_init);
module_exit(pwm_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Alexandre MSELLATI, Columbia University");
MODULE_DESCRIPTION("PWM driver");
