/*
 * Userspace program that communicates with the vga_ball device driver
 * through ioctls
 *
 * Stephen A. Edwards
 * Columbia University
 */

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdint.h>
#include <stdbool.h>
#include <pthread.h>
#include <stdlib.h>
#include <stdio.h>
#include "pwm.h"

#include <string.h>
#include <stdbool.h>
#include <time.h>
#include <unistd.h>
#include <math.h>
#include "pi_controller.h"


// ======= Declarations of functions
float adc_to_speed_command_computation(unsigned short adc_value);
void *adc_command_reading_thread_f(void *ignored);  // Update the actual motor speed
void *speed_reading_thread_f(void *ignored);  // Update the actual motor speed
void *duty_cycle_thread_f(void *ignored);
float count_to_speed_computation(unsigned short* encoder_count);
void print_encoder_count(unsigned short * encoder_count);
void send_motor_command(const controller_cmd_t *new_command);
// void *col_speed_thread_f(void *ignored);

// === Constants ====

// Thread
pthread_t speed_reading_thread, duty_cycle_thread, adc_command_reading_thread;

// Driver
int pwm_fd;

// Controller
const float T_sample = 1e-2;  // [s], 100 Hz
const float Kp_i = 5;
const float Ki_i = 1e3;
const float Ka_i = 0.2;
const float k = 0.02;  // [Vs]
const float Kp_s = 5;
const float Ki_s = 1e3;
const float Ka_s = 0.2;

// Motor
const int Vmax = 12;  // [V]
const int Imax = 1;  // [A]
const float Tmax = 0.247;  // [Nm]
const float speed_min =  105; // [rad/s]
const float R = 0.2;  // Ohm

// Reading and writing
// unsigned short * encoder_count;
// float omega_motor_speed;  // rad/s 
// controller_cmd_t *new_command;

unsigned short * adc_value, *encoder_count;
float* omega_motor_speed;  // rad/s 
controller_cmd_t *new_command;

float duty_cycle;
int duty_cycle_command;
bool rotation_direction_command;
float* speed_command, *current_command, *voltage_command, *current_measure;
float* speed_error, *current_error;


int main(void){
    

    // Reading and writing
    // unsigned short * encoder_count;
    // float* omega_motor_speed;  // rad/s 
    // controller_cmd_t *new_command;

    printf(" Initialization of the driver \n");

    static const char filename[] = "/dev/pwm";

    printf("PWM Userspace program started\n");

    if ( (pwm_fd = open(filename, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", filename);
    return -1;
    }

    printf("Initialization of the controllers \n");

    // ================= Controller initialization ================= //
    // Allocating memory for the controller
    pi_t p_controller_i = (pi_t) malloc(sizeof(pi_t));
    pi_t pi_i = (pi_t) malloc(sizeof(pi_t));
    pi_t pi_wup_i = (pi_t) malloc(sizeof(pi_t));
    pi_t pi_s = (pi_t) malloc(sizeof(pi_t));
    pi_t pi_wup_s = (pi_t) malloc(sizeof(pi_t));

    // Creating the controllers
    // Current
    p_controller_i = pi_create(p_controller_i, k, 0, T_sample); // Feedforward
    pi_i = pi_create(pi_i, Kp_i, Ki_i, T_sample); // PI controller
    pi_wup_i = pi_create(pi_wup_i, 0, Ki_i * Ka_i, T_sample);  // Anti-windup
    // Speed
    pi_s = pi_create(pi_s, Kp_s, Ki_s, T_sample);  // PI controller
    pi_wup_s = pi_create(pi_wup_s, 0, Ki_s * Ka_s, T_sample);  // Anti_windup

    // Pointers for retrieving data
    encoder_count = (unsigned short *) malloc(sizeof(unsigned short *));
    new_command = (controller_cmd_t*) malloc(sizeof(controller_cmd_t));
    new_command->duty_cycle = duty_cycle_command;
    new_command->rotation_direction = rotation_direction_command;
    speed_command = (float*) malloc(sizeof(float*));  // User command through potentiometer
    omega_motor_speed = (float*) malloc(sizeof(float*));

    printf("Launching the threads \n");
    // Launching the reading encoder value thread
    pthread_create(&adc_command_reading_thread, NULL, adc_command_reading_thread_f, NULL);
    pthread_create(&speed_reading_thread, NULL, speed_reading_thread_f, NULL);
    pthread_create(&duty_cycle_thread, NULL, duty_cycle_thread_f, NULL);
    printf("Threads launched \n");
    // Current controller
    p_controller_i->input = omega_motor_speed;  // Feedforward
    *pi_wup_i->input = 0;
    *pi_i->input = 0;
    pi_compute(p_controller_i);
    pi_compute(pi_i);
    pi_compute(pi_wup_i);
    printf("Current controller computed \n");
    // Speed controller
    *pi_wup_s->input = 0;
    *pi_s->input = 0;
    pi_compute(pi_s);
    pi_compute(pi_wup_s);
    printf("Speed controller computed \n");

    // Initialization
    
    current_measure = (float*) malloc(sizeof(float*));
    current_command = (float*) malloc(sizeof(float*));
    voltage_command = (float*) malloc(sizeof(float*));
    speed_error = (float*) malloc(sizeof(float*));
    current_error = (float*) malloc(sizeof(float*));

    
    *speed_error = 100;
    printf("End of initialization \n");

    printf("Entering the while loop of control \n");

    while (true){

    // Computing the errors
    *speed_error = *speed_command - *omega_motor_speed;  // Updated through the threads
    // printf("Speed command in the main: %f \n", *speed_command);
    printf("Omega motor speed: %f \n", *omega_motor_speed);
    // printf("Speed error: %f \n", *speed_error);
    // printf("Duty cycle: %d, %d \n", new_command->duty_cycle, new_command->rotation_direction);
    // *current_measure = ((*voltage_command) - (k * (*omega_motor_speed)) / R); // Actually an estimation based on motor param
    *current_measure = 0; // Actually an estimation based on motor param
    *current_error = *current_command - *current_measure;
    printf("current measure %f \n", *current_measure);
    // Computing the command outputs of the controllers
    // current_command = speed_controller(0, k, pi_s, pi_wup_s);
    // voltage_command = current_controller(1, p_controller_i, pi_i, pi_wup_i);
    current_command = speed_controller(*speed_error, k, pi_s, pi_wup_s);
    voltage_command = current_controller(*current_error, p_controller_i, pi_i, pi_wup_i);
    // printf("voltage command: %f \n", *voltage_command);
    // Adapting and storing the data
    rotation_direction_command = (*voltage_command >= 0);
    duty_cycle = fabs(*voltage_command) / Vmax * 2000;  // btw 0 and 2000
    duty_cycle_command = (int) duty_cycle;

    // Updating the commands
    new_command->duty_cycle = duty_cycle_command;
    new_command->rotation_direction = rotation_direction_command;
    // new_command->duty_cycle = 1000;
    // new_command->rotation_direction = 1;
    
    // Freeing up memory
    free(current_command);
    free(voltage_command);

    // if (i == 1000){
    //     begin = clock();
    //     current_controller(u[i], p_controller_i, pi_i, pi_wup_i);
    //     // sleep(1);
    //     end = clock();
    // }
    // printf("Voltage command: %f \n", *voltage_command);
    // printf("Conversion in duty cycle %f \n", duty_cycle);
    // printf("Duty cycle command %d \n", duty_cycle_command);
    // printf("Value of rotation_direction: %d \n", rotation_direction);



    
    // send_motor_command(new_command);
    // sleep(0.01); // Refresh at 100 Hz
    }


    // printf("After new command sent: ");
    // print_running_command();


    return 0;
}


float adc_to_speed_command_computation(unsigned short adc_value){
    float speed_command = 0;
    float conversion = 0.977;  // 1 = 0.977 mV
    float adc_voltage = adc_value * conversion;  // mV
    float mV_to_radps_conversion = 0.124;  // 1 mV = 0.149 rad/s
    speed_command = 100 + adc_voltage * mV_to_radps_conversion;  // min speed = 1000RPM
    return speed_command;
}

float count_to_speed_computation(unsigned short* encoder_count){
    // printf("count to speed ini");
    float rpm_speed = *encoder_count * 60 / 7 * (1 / T_sample);  // [RPM]
    return 2 * 3.14 * rpm_speed / 60;
}


void print_adc_command(unsigned short * adc_value){
  

  if (ioctl(pwm_fd, PWM_READ_ADC, adc_value)) {
        perror("ioctl(PWM_READ_ADC) failed");
        return;
    }
}


void print_encoder_count(unsigned short * encoder_count){
  

  if (ioctl(pwm_fd, PWM_READ_CNT, encoder_count)) {
        perror("ioctl(PWM_READ_CNT) failed");
        return;
    }
    // perror("ioctl(PWM_READ_CLK) Success");
    // printf("\n Duty cyle: %d \n Rotation direction: %d \n",
     // command.duty_cycle, command.rotation_direction);
}

/* Send the new command */
void send_motor_command(const controller_cmd_t *new_command)
{
  
  if (ioctl(pwm_fd, PWM_WRITE_CMD, new_command)) {
      perror("ioctl(PWM_WRITE_CMD) failed");
      return;
  }
  else{
    // perror("ioctl(PWM_WRITE_CMD): Success");  
  }
}


void *adc_command_reading_thread_f(void *ignored)
{
    // printf("ADC thread ini \n");
    while (true) {
    // Reading encoder value
        // Add a value in case of an error
        print_adc_command(adc_value);
        // *speed_command = adc_to_speed_command_computation(*adc_value);
        *speed_command = 500;  // [rad/s]  
        usleep(10); // Update every 10 us (100 kHz)
  } 
  return NULL;
}

void *speed_reading_thread_f(void *ignored)
{
    // printf("Speed thread ini \n");
    while (true) {
    // Reading encoder value
        // printf("Speed thread running \n");
        print_encoder_count(encoder_count);
        // printf("Value of encoder_count: %d \n", *encoder_count);
        // printf("Speed thread reading encoder \n");
        *omega_motor_speed = count_to_speed_computation(encoder_count);  // Computing the speed
        // printf("Speed thread end \n");
        sleep(0.001); // Update every ms, 1 kHz
  } 
  return NULL;
}

void *duty_cycle_thread_f(void *ignored)
{
    // printf("Duty cycle thread ini \n");
    while (true) {
    // Reading encoder value
        // printf("Duty cycle thread running \n");
        new_command->duty_cycle = 1000;
        send_motor_command(new_command);
        printf("Duty cycle thread: %d, %d \n", new_command->duty_cycle, new_command->rotation_direction);
        printf("Duty cycle thread end \n");
        usleep(10); // Update every 10 us (100 kHz)
  } 
  return NULL;
}