module readFMPipeline(
	input	logic						clk, reset,

	// Input from HPS
	input	logic						start,
	input	logic	[15: 0]				read_length,
	input	logic	[5: 0]				read_data_dim,

	// Output to pipeline
	output	logic						in_data_ready,
	output	logic	[5: 0]				in_data_dim,
	output	logic	[32*32*16-1: 0]		feat_map_in,
	output	logic	[(3*3+1)*16-1: 0]	weight_bias,


	// On-Chip RAM 0 s2 (read)
	input	logic	[ 7: 0]				ocm0_readdata,
	output	logic	[16: 0]				ocm0_addr,
	output	logic						ocm0_chip,
	output	logic						ocm0_clk_enab,

	// Debug
	output	logic	[ 2: 0]				debug_state
);

	// FSM
	logic [ 2: 0]	state;
	assign debug_state = state;
	parameter S0 = 0, S1 = 1, S2 = 2, S3 = 3, S4 = 4;

	// For OCM Read
	logic [16: 0]	addr = 0;
	logic [15: 0]	read_counter = 0;
	logic			is_wb;
	assign			is_wb = (read_counter < 20)? 1 : 0;	// addr is addr for next cycle, so -1 for this cycle

	assign in_data_dim = read_data_dim;

	always @ (posedge clk) begin
		case (state)
			S0:	// reset
				begin
					addr = 0;
					ocm0_clk_enab = 0;
					in_data_ready = 0;
				end
			S1:	// prepare to read
				begin
					read_counter = 0;
					
					ocm0_addr = addr;
					ocm0_chip = 1;
					ocm0_clk_enab = 1;
					
					addr = addr + 1;
				end
			S2:	// Read
				begin
					ocm0_addr = addr;
					ocm0_chip = 1;
					ocm0_clk_enab = 1;

					if (is_wb) begin
						weight_bias[(read_counter)*8 +: 8] = ocm0_readdata;
					end else begin
						feat_map_in[(read_counter-20)*8 +: 8] = ocm0_readdata;
					end

					addr = addr + 1;
					read_counter = read_counter + 1;
				end
			S3: // Read last cycle
				begin
					ocm0_addr = 0;
					ocm0_chip = 1;
					ocm0_clk_enab = 1;

					feat_map_in[(read_counter-20)*8 +: 8] = ocm0_readdata;

					read_counter = read_counter + 1;
				end
			S4:	// Finished, wait for start->0
				begin
					in_data_ready = 1;
					ocm0_clk_enab = 0;
				end
		endcase
	end

	// Determine the next state
	always @ (posedge clk or posedge reset) begin
		if (reset)
			begin
				state <= S0;
			end
		else
			case (state)
				S0:
					if(start)
						state <= S1;
				S1:
					state <= S2;
				S2:
					begin
						if(addr >= read_length - 1)
							state <= S3;
					end
				S3:
					state <= S4;
				S4:
					if(start == 0)
						state <= S0;

			endcase
	end

endmodule
