/*
 * Avalon memory-mapped peripheral that generates VGA
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * Hardware Part of Breakout Game Remastered Project 
 * CSEE 4840, Spring 2022, Columbia University
 */

module vga_ball(input logic        clk,
	  input logic 	   reset,
		input logic [15:0]  writedata,
		input logic 	   write,
		input 		   chipselect,
		input logic [3:0]  address,

		input left_chan_ready,
		input right_chan_ready,

		output logic [15:0] sample_data_l,
		output logic sample_valid_l,
		output logic [15:0] sample_data_r,
		output logic sample_valid_r,

		output logic [7:0] VGA_R, VGA_G, VGA_B,
		output logic 	   VGA_CLK, VGA_HS, VGA_VS,
		                   VGA_BLANK_n,
		output logic 	   VGA_SYNC_n);

   logic [10:0]	   hcount, hcount_adj;
   logic [9:0]     vcount;

   vga_counters counters(.clk50(clk), .*);

   logic [15:0] score, x_ball, y_ball, x_pad, brick1, brick2, brick3, brick4, brick5, brick6, sound_effect, game_status;
   logic [4:0] tile_x, tile_y;
   logic [10:0] vcount1;

   assign vcount1 = {1'b0,vcount};

   logic circle;
   logic peddle;
   logic waste;

   logic [10:0] x_ball_adj, x_pad_adj, y_ball_adj;

   assign x_ball_adj = {2'b0,x_ball[8:0]}*2 + 11'd224;
   assign x_pad_adj  = {2'b0,x_pad[8:0]}*2 + 11'd224;
   assign y_ball_adj = {2'b0, y_ball[8:0]};

   assign circle = (((hcount - x_ball_adj) * (hcount - x_ball_adj)) + 4 * ((vcount1 - y_ball_adj) * (vcount1 - y_ball_adj))) < 11'd100; // ball r = 10
   assign peddle = ((x_pad_adj - 11'd32) < hcount) && (hcount < (x_pad_adj + 11'd32)) && (10'd430 <= vcount) && (vcount < 10'd446); // pad 32 * 16
   assign waste = (hcount < 11'd192) || (hcount > 11'd1088); //needless area

   //Setup tile coordinates
   assign tile_y = vcount / 16;
   
   always_comb begin
	if (hcount >= 192 && hcount < 1088)
		tile_x = ((hcount / 2) - 96) / 16;
	else
		tile_x = 5'b11111;
   end


   always_ff @(posedge clk) begin
     if (reset) begin
    x_ball <= 16'd208;
  	y_ball <= 16'd425;
    x_pad <= 16'd208;
	score <= 16'h4840;
	brick1 <= 16'b0001000000000000;
	brick2 <= 16'b0010100000000000;
	brick3 <= 16'b0100010000000000;
	brick4 <= 16'b0110001000000000;
	brick5 <= 16'b1000000100000000;
	brick6 <= 16'b1010000010000000;
	sound_effect <= 16'b000;
    game_status <= 16'b0000000000000110;


     end else if (chipselect && write) begin
       case (address)
	 4'h0 : x_ball <= writedata;
	 4'h1 : y_ball <= writedata;
	 4'h2 : x_pad <= writedata;
	 4'h3 : score <= writedata;
	 4'h4 : brick1 <= writedata;
	 4'h5 : brick2 <= writedata;
	 4'h6 : brick3 <= writedata;
	 4'h7 : brick4 <= writedata;
	 4'h8 : brick5 <= writedata;
	 4'h9 : brick6 <= writedata;
	 4'ha : sound_effect <= writedata;
	 4'hb : game_status <= writedata;
       endcase
    end
    end
	//----------------Sound-----------------
	reg [13:0] counter;
	logic flag1;
	logic flag2;
	logic flag3;
	logic flag4;

	reg	[8:0]  address1;
	wire [15:0]  q1;
	hitbrick_ROM audio1(.address(address1), .clock(clk), .q(q1));//290

	reg	[9:0]  address2;
	wire [15:0]  q2;
	hitwall_ROM  audio2(.address(address2), .clock(clk), .q(q2));//784

	reg	[13:0]  address3;
	wire [15:0]  q3;
	gameover_ROM audio3(.address(address3), .clock(clk), .q(q3));//8489
	
	reg [12:0] address4;
	wire [15:0] q4;
	victory_ROM audio4(.address(address4), .clock(clk), .q(q4));//4504
	
	always_ff @(posedge clk) begin
		if(reset) begin
			counter <= 0;
			sample_valid_l <= 0; sample_valid_r <= 0;
		end
		else if(left_chan_ready == 1 && right_chan_ready == 1 && counter < 6250) begin
			counter <= counter + 1;
			sample_valid_l <= 0; sample_valid_r <= 0;
		end
		else if(left_chan_ready == 1 && right_chan_ready == 1 && counter == 6250) begin
			counter <= 0;
			sample_valid_l <= 1; sample_valid_r <= 1;
			//------Game over------
			if (sound_effect[2:0]==3'd3 || flag3 ==1'b0) begin
				if (address3 < 14'd8489) begin
					address3 <= address3+1;
					flag3 <= 1'b0;
				end
				else begin
					address3 <=0;
					flag3 <= 1'b1;
				end
				sample_data_l <= q3;
				sample_data_r <= q3;
			end
			//------Hit Brick------
			else if (sound_effect[2:0]==3'd1 || flag1 ==1'b0) begin
				if (address1 < 9'd290) begin
					address1 <= address1+1;
					flag1 <= 1'b0;
				end
				else begin
					address1 <=0;
					flag1 <= 1'b1;
				end
				sample_data_l <= q1;
				sample_data_r <= q1;
			end
			//------Hit Wall------
			else if (sound_effect[2:0]==3'd2 || flag2 ==1'b0) begin
				if (address2 < 10'd784) begin
					address2 <= address2+1;
					flag2 <= 1'b0;
				end
				else begin
					address2 <=0;
					flag2 <= 1'b1;
				end
				sample_data_l <= q2;
				sample_data_r <= q2;
			end
			//------Win------
			else if (sound_effect[2:0]==3'd4 || flag4 == 1'b0) begin
				if (address4 < 13'd4504) begin
					address4 <= address4 + 1;
					flag4 <= 1'b0;
				end else begin
					address4 <= 0;
					flag4 <= 1'b1;
				end
				sample_data_l <= q4;
				sample_data_r <= q4;
			end

			else begin
				sample_data_l <= 0;
				sample_data_r <= 0;
			end
		end
		else begin
		sample_valid_l <= 0; sample_valid_r <= 0;
		end
	end


	// ---------------ROM tilemap_ROM.v---------------
	// Assign objects to specific locations of the screen
	logic [8:0]  tile_addr;
	logic [31:0] tile_output;
	tilemap_ROM tile1(.address(tile_addr),.clock(clk),.q(tile_output));

	always_ff @(posedge clk) begin
      //-----Static elements------
      //Side boarder
      if (tile_y >= 3 && (tile_x == 0 || tile_x == 27)) begin
         tile_addr <= 9'h13;
      //Top boarder
      end else if (tile_x <= 26 && tile_x >= 1 && tile_y == 2) begin
        tile_addr <= vcount - 32;
      //L Corner
      end else if (tile_x == 0 && tile_y == 2) begin
        tile_addr <= vcount;
      //R Corner
      end else if (tile_x == 27 && tile_y == 2) begin
        tile_addr <= vcount + 16;
      //SCORE
      end else if (tile_x <= 16 && tile_x >= 12 && tile_y == 0) begin
		case (tile_x)
			5'd12 : tile_addr <= vcount + 96;
			5'd13 : tile_addr <= vcount + 112;
			5'd14 : tile_addr <= vcount + 128;
			5'd15 : tile_addr <= vcount + 144;
            5'd16 : tile_addr <= vcount + 160;
		endcase
	  //STAGE
	  end else if (tile_x <= 27 && tile_x >= 23 && tile_y == 0) begin
		case (tile_x)
			5'd23 : tile_addr <= vcount + 96;
			5'd24 : tile_addr <= vcount + 352;
			5'd25 : tile_addr <= vcount + 368;
			5'd26 : tile_addr <= vcount + 384;
			5'd27 : tile_addr <= vcount + 160;
		endcase
      //------Dynamic elements------
      //Score Numbers
      end else if (tile_x <= 16 && tile_x >= 13 && tile_y == 1) begin
        case (tile_x)
			5'd13 : tile_addr <= vcount + 160 + (score[15:12]*16);
			5'd14 : tile_addr <= vcount + 160 + (score[11:8]*16);
			5'd15 : tile_addr <= vcount + 160 + (score[7:4]*16);
			5'd16 : tile_addr <= vcount + 160 + (score[3:0]*16);
		endcase
	  //Stage Numbers
      end else if (tile_x == 27 && tile_y == 1) begin
		tile_addr <= game_status[0] ? (vcount + 192) : (vcount + 176);
	  //HP Indicator
	  end else if ((tile_x == 1 || tile_x == 2) && tile_y == 29) begin
		case (tile_x)
			5'd1 : 	begin 
					if (game_status[2])
						tile_addr <= vcount - 128;
					else
						tile_addr <= 0;
					end
			5'd2 :  begin
					if (game_status[1])
						tile_addr <= vcount - 128;
					else 
						tile_addr <= 0;
					end
		endcase
	  //Win & Lose
	  end else if (tile_y == 15 && tile_x <= 18 && tile_x >= 10) begin
		if (game_status[3]) begin
			if (game_status[4]) begin
				case (tile_x)
					5'd10 : tile_addr <= vcount - 128;
					5'd11 : tile_addr <= vcount - 112;
					5'd12 : tile_addr <= vcount + 160;
					5'd13 : tile_addr <= vcount + 144;
					5'd14 : tile_addr <= vcount - 96;
					5'd15 : tile_addr <= vcount + 128;
					5'd16 : tile_addr <= vcount + 112;
					5'd17 : tile_addr <= vcount - 144;
					5'd18 : tile_addr <= vcount + 178;
				endcase
			end else begin
				case (tile_x)
					5'd13 : tile_addr <= vcount + 144;
					5'd14 : tile_addr <= vcount + 144;
					5'd15 : tile_addr <= vcount + 178;
					default : tile_addr <= 0;
				endcase
			end
		end else begin
			tile_addr <= 0;
		end
	  //Bricks
      end else if (tile_x >= 1 && tile_x <= 26 && tile_y >= 5 && tile_y <= 10) begin
		case (tile_y)
		//Brick line 1
		5'd5 : begin
			if (brick1[12]) begin
				case (tile_x)
					5'd1 : tile_addr <= vcount - 16;
					5'd2 : tile_addr <= vcount;
				endcase
			end else begin
				case (tile_x)
					5'd1 : tile_addr <= 0;
					5'd2 : tile_addr <= 0;
				endcase
			end
			if (brick1[11]) begin
				case (tile_x)
					5'd3 : tile_addr <= vcount - 16;
					5'd4 : tile_addr <= vcount;
				endcase
			end else begin
				case (tile_x)
					5'd3 : tile_addr <= 0;
					5'd4 : tile_addr <= 0;
				endcase
			end
			if (brick1[10]) begin
				case (tile_x)
					5'd5 : tile_addr <= vcount - 16;
					5'd6 : tile_addr <= vcount;
				endcase
			end else begin
				case (tile_x)
					5'd5 : tile_addr <= 0;
					5'd6 : tile_addr <= 0;
				endcase
			end
			if (brick1[9]) begin
				case (tile_x)
					5'd7 : tile_addr <= vcount - 16;
					5'd8 : tile_addr <= vcount;
				endcase
			end else begin
				case (tile_x)
					5'd7 : tile_addr <= 0;
					5'd8 : tile_addr <= 0;
				endcase
			end
			if (brick1[8]) begin
				case (tile_x)
					5'd9 : tile_addr <= vcount - 16;
					5'd10: tile_addr <= vcount;
				endcase
			end else begin
				case (tile_x)
					5'd9 : tile_addr <= 0;
					5'd10: tile_addr <= 0;
				endcase
			end
			if (brick1[7]) begin
				case (tile_x)
					5'd11: tile_addr <= vcount - 16;
					5'd12 : tile_addr <= vcount;
				endcase
			end else begin
				case (tile_x)
					5'd11: tile_addr <= 0;
					5'd12 : tile_addr <= 0;
				endcase
			end
			if (brick1[6]) begin
				case (tile_x)
					5'd13: tile_addr <= vcount - 16;
					5'd14: tile_addr <= vcount;
				endcase
			end else begin
				case (tile_x)
					5'd13: tile_addr <= 0;
					5'd14: tile_addr <= 0;
				endcase
			end
			if (brick1[5]) begin
				case (tile_x)
					5'd15: tile_addr <= vcount - 16;
					5'd16: tile_addr <= vcount;
				endcase
			end else begin
				case (tile_x)
					5'd15: tile_addr <= 0;
					5'd16: tile_addr <= 0;
				endcase
			end
			if (brick1[4]) begin
				case (tile_x)
					5'd17: tile_addr <= vcount - 16;
					5'd18: tile_addr <= vcount;
				endcase
			end else begin
				case (tile_x)
					5'd17: tile_addr <= 0;
					5'd18: tile_addr <= 0;
				endcase
			end
			if (brick1[3]) begin
				case (tile_x)
					5'd19: tile_addr <= vcount - 16;
					5'd20: tile_addr <= vcount;
				endcase
			end else begin
				case (tile_x)
					5'd19: tile_addr <= 0;
					5'd20: tile_addr <= 0;
				endcase
			end
			if (brick1[2]) begin
				case (tile_x)
					5'd21: tile_addr <= vcount - 16;
					5'd22: tile_addr <= vcount;
				endcase
			end else begin
				case (tile_x)
					5'd21: tile_addr <= 0;
					5'd22: tile_addr <= 0;
				endcase
			end
			if (brick1[1]) begin
				case (tile_x)
					5'd23: tile_addr <= vcount - 16;
					5'd24: tile_addr <= vcount;
				endcase
			end else begin
				case (tile_x)
					5'd23: tile_addr <= 0;
					5'd24: tile_addr <= 0;
				endcase
			end
			if (brick1[0]) begin
				case (tile_x)
					5'd25: tile_addr <= vcount - 16;
					5'd26: tile_addr <= vcount;
				endcase
			end else begin
				case (tile_x)
					5'd25: tile_addr <= 0;
					5'd26: tile_addr <= 0;
				endcase
			end
		end
		//Brick line 2
		5'd6 : begin
			if (brick2[12]) begin
				case (tile_x)
					5'd1 : tile_addr <= vcount - 32;
					5'd2 : tile_addr <= vcount - 16;
				endcase
			end else begin
				case (tile_x)
					5'd1 : tile_addr <= 0;
					5'd2 : tile_addr <= 0;
				endcase
			end
			if (brick2[11]) begin
				case (tile_x)
					5'd3 : tile_addr <= vcount - 32;
					5'd4 : tile_addr <= vcount - 16;
				endcase
			end else begin
				case (tile_x)
					5'd3 : tile_addr <= 0;
					5'd4 : tile_addr <= 0;
				endcase
			end
			if (brick2[10]) begin
				case (tile_x)
					5'd5 : tile_addr <= vcount - 32;
					5'd6 : tile_addr <= vcount - 16;
				endcase
			end else begin
				case (tile_x)
					5'd5 : tile_addr <= 0;
					5'd6 : tile_addr <= 0;
				endcase
			end
			if (brick2[9]) begin
				case (tile_x)
					5'd7 : tile_addr <= vcount - 32;
					5'd8 : tile_addr <= vcount - 16;
				endcase
			end else begin
				case (tile_x)
					5'd7 : tile_addr <= 0;
					5'd8 : tile_addr <= 0;
				endcase
			end
			if (brick2[8]) begin
				case (tile_x)
					5'd9 : tile_addr <= vcount - 32;
					5'd10: tile_addr <= vcount - 16;
				endcase
			end else begin
				case (tile_x)
					5'd9 : tile_addr <= 0;
					5'd10: tile_addr <= 0;
				endcase
			end
			if (brick2[7]) begin
				case (tile_x)
					5'd11: tile_addr <= vcount - 32;
					5'd12: tile_addr <= vcount - 16;
				endcase
			end else begin
				case (tile_x)
					5'd11: tile_addr <= 0;
					5'd12: tile_addr <= 0;
				endcase
			end
			if (brick2[6]) begin
				case (tile_x)
					5'd13: tile_addr <= vcount - 32;
					5'd14: tile_addr <= vcount - 16;
				endcase
			end else begin
				case (tile_x)
					5'd13: tile_addr <= 0;
					5'd14: tile_addr <= 0;
				endcase
			end
			if (brick2[5]) begin
				case (tile_x)
					5'd15: tile_addr <= vcount - 32;
					5'd16: tile_addr <= vcount - 16;
				endcase
			end else begin
				case (tile_x)
					5'd15: tile_addr <= 0;
					5'd16: tile_addr <= 0;
				endcase
			end
			if (brick2[4]) begin
				case (tile_x)
					5'd17: tile_addr <= vcount - 32;
					5'd18: tile_addr <= vcount - 16;
				endcase
			end else begin
				case (tile_x)
					5'd17: tile_addr <= 0;
					5'd18: tile_addr <= 0;
				endcase
			end
			if (brick2[3]) begin
				case (tile_x)
					5'd19: tile_addr <= vcount - 32;
					5'd20: tile_addr <= vcount - 16;
				endcase
			end else begin
				case (tile_x)
					5'd19: tile_addr <= 0;
					5'd20: tile_addr <= 0;
				endcase
			end
			if (brick2[2]) begin
				case (tile_x)
					5'd21: tile_addr <= vcount - 32;
					5'd22: tile_addr <= vcount - 16;
				endcase
			end else begin
				case (tile_x)
					5'd21: tile_addr <= 0;
					5'd22: tile_addr <= 0;
				endcase
			end
			if (brick2[1]) begin
				case (tile_x)
					5'd23: tile_addr <= vcount - 32;
					5'd24: tile_addr <= vcount - 16;
				endcase
			end else begin
				case (tile_x)
					5'd23: tile_addr <= 0;
					5'd24: tile_addr <= 0;
				endcase
			end
			if (brick2[0]) begin
				case (tile_x)
					5'd25: tile_addr <= vcount - 32;
					5'd26: tile_addr <= vcount - 16;
				endcase
			end else begin
				case (tile_x)
					5'd25: tile_addr <= 0;
					5'd26: tile_addr <= 0;
				endcase
			end
		end
		//Brick line 3
		5'd7 : begin
			if (brick3[12]) begin
				case (tile_x)
					5'd1 : tile_addr <= vcount - 48;
					5'd2 : tile_addr <= vcount - 32;
				endcase
			end else begin
				case (tile_x)
					5'd1 : tile_addr <= 0;
					5'd2 : tile_addr <= 0;
				endcase
			end
			if (brick3[11]) begin
				case (tile_x)
					5'd3 : tile_addr <= vcount - 48;
					5'd4 : tile_addr <= vcount - 32;
				endcase
			end else begin
				case (tile_x)
					5'd3 : tile_addr <= 0;
					5'd4 : tile_addr <= 0;
				endcase
			end
			if (brick3[10]) begin
				case (tile_x)
					5'd5 : tile_addr <= vcount - 48;
					5'd6 : tile_addr <= vcount - 32;
				endcase
			end else begin
				case (tile_x)
					5'd5 : tile_addr <= 0;
					5'd6 : tile_addr <= 0;
				endcase
			end
			if (brick3[9]) begin
				case (tile_x)
					5'd7 : tile_addr <= vcount - 48;
					5'd8 : tile_addr <= vcount - 32;
				endcase
			end else begin
				case (tile_x)
					5'd7 : tile_addr <= 0;
					5'd8 : tile_addr <= 0;
				endcase
			end
			if (brick3[8]) begin
				case (tile_x)
					5'd9 : tile_addr <= vcount - 48;
					5'd10: tile_addr <= vcount - 32;
				endcase
			end else begin
				case (tile_x)
					5'd9 : tile_addr <= 0;
					5'd10: tile_addr <= 0;
				endcase
			end
			if (brick3[7]) begin
				case (tile_x)
					5'd11: tile_addr <= vcount - 48;
					5'd12: tile_addr <= vcount - 32;
				endcase
			end else begin
				case (tile_x)
					5'd11: tile_addr <= 0;
					5'd12: tile_addr <= 0;
				endcase
			end
			if (brick3[6]) begin
				case (tile_x)
					5'd13: tile_addr <= vcount - 48;
					5'd14: tile_addr <= vcount - 32;
				endcase
			end else begin
				case (tile_x)
					5'd13: tile_addr <= 0;
					5'd14: tile_addr <= 0;
				endcase
			end
			if (brick3[5]) begin
				case (tile_x)
					5'd15: tile_addr <= vcount - 48;
					5'd16: tile_addr <= vcount - 32;
				endcase
			end else begin
				case (tile_x)
					5'd15: tile_addr <= 0;
					5'd16: tile_addr <= 0;
				endcase
			end
			if (brick3[4]) begin
				case (tile_x)
					5'd17: tile_addr <= vcount - 48;
					5'd18: tile_addr <= vcount - 32;
				endcase
			end else begin
				case (tile_x)
					5'd17: tile_addr <= 0;
					5'd18: tile_addr <= 0;
				endcase
			end
			if (brick3[3]) begin
				case (tile_x)
					5'd19: tile_addr <= vcount - 48;
					5'd20: tile_addr <= vcount - 32;
				endcase
			end else begin
				case (tile_x)
					5'd19: tile_addr <= 0;
					5'd20: tile_addr <= 0;
				endcase
			end
			if (brick3[2]) begin
				case (tile_x)
					5'd21: tile_addr <= vcount - 48;
					5'd22: tile_addr <= vcount - 32;
				endcase
			end else begin
				case (tile_x)
					5'd21: tile_addr <= 0;
					5'd22: tile_addr <= 0;
				endcase
			end
			if (brick3[1]) begin
				case (tile_x)
					5'd23: tile_addr <= vcount - 48;
					5'd24: tile_addr <= vcount - 32;
				endcase
			end else begin
				case (tile_x)
					5'd23: tile_addr <= 0;
					5'd24: tile_addr <= 0;
				endcase
			end
			if (brick3[0]) begin
				case (tile_x)
					5'd25: tile_addr <= vcount - 48;
					5'd26: tile_addr <= vcount - 32;
				endcase
			end else begin
				case (tile_x)
					5'd25: tile_addr <= 0;
					5'd26: tile_addr <= 0;
				endcase
			end
		end
		//Brick line 4
		5'd8 : begin
			if (brick4[12]) begin
				case (tile_x)
					5'd1 : tile_addr <= vcount - 64;
					5'd2 : tile_addr <= vcount - 48;
				endcase
			end else begin
				case (tile_x)
					5'd1 : tile_addr <= 0;
					5'd2 : tile_addr <= 0;
				endcase
			end
			if (brick4[11]) begin
				case (tile_x)
					5'd3 : tile_addr <= vcount - 64;
					5'd4 : tile_addr <= vcount - 48;
				endcase
			end else begin
				case (tile_x)
					5'd3 : tile_addr <= 0;
					5'd4 : tile_addr <= 0;
				endcase
			end
			if (brick4[10]) begin
				case (tile_x)
					5'd5 : tile_addr <= vcount - 64;
					5'd6 : tile_addr <= vcount - 48;
				endcase
			end else begin
				case (tile_x)
					5'd5 : tile_addr <= 0;
					5'd6 : tile_addr <= 0;
				endcase
			end
			if (brick4[9]) begin
				case (tile_x)
					5'd7 : tile_addr <= vcount - 64;
					5'd8 : tile_addr <= vcount - 48;
				endcase
			end else begin
				case (tile_x)
					5'd7 : tile_addr <= 0;
					5'd8 : tile_addr <= 0;
				endcase
			end
			if (brick4[8]) begin
				case (tile_x)
					5'd9 : tile_addr <= vcount - 64;
					5'd10: tile_addr <= vcount - 48;
				endcase
			end else begin
				case (tile_x)
					5'd9 : tile_addr <= 0;
					5'd10: tile_addr <= 0;
				endcase
			end
			if (brick4[7]) begin
				case (tile_x)
					5'd11: tile_addr <= vcount - 64;
					5'd12: tile_addr <= vcount - 48;
				endcase
			end else begin
				case (tile_x)
					5'd11: tile_addr <= 0;
					5'd12: tile_addr <= 0;
				endcase
			end
			if (brick4[6]) begin
				case (tile_x)
					5'd13: tile_addr <= vcount - 64;
					5'd14: tile_addr <= vcount - 48;
				endcase
			end else begin
				case (tile_x)
					5'd13: tile_addr <= 0;
					5'd14: tile_addr <= 0;
				endcase
			end
			if (brick4[5]) begin
				case (tile_x)
					5'd15: tile_addr <= vcount - 64;
					5'd16: tile_addr <= vcount - 48;
				endcase
			end else begin
				case (tile_x)
					5'd15: tile_addr <= 0;
					5'd16: tile_addr <= 0;
				endcase
			end
			if (brick4[4]) begin
				case (tile_x)
					5'd17: tile_addr <= vcount - 64;
					5'd18: tile_addr <= vcount - 48;
				endcase
			end else begin
				case (tile_x)
					5'd17: tile_addr <= 0;
					5'd18: tile_addr <= 0;
				endcase
			end
			if (brick4[3]) begin
				case (tile_x)
					5'd19: tile_addr <= vcount - 64;
					5'd20: tile_addr <= vcount - 48;
				endcase
			end else begin
				case (tile_x)
					5'd19: tile_addr <= 0;
					5'd20: tile_addr <= 0;
				endcase
			end
			if (brick4[2]) begin
				case (tile_x)
					5'd21: tile_addr <= vcount - 64;
					5'd22: tile_addr <= vcount - 48;
				endcase
			end else begin
				case (tile_x)
					5'd21: tile_addr <= 0;
					5'd22: tile_addr <= 0;
				endcase
			end
			if (brick4[1]) begin
				case (tile_x)
					5'd23: tile_addr <= vcount - 64;
					5'd24: tile_addr <= vcount - 48;
				endcase
			end else begin
				case (tile_x)
					5'd23: tile_addr <= 0;
					5'd24: tile_addr <= 0;
				endcase
			end
			if (brick4[0]) begin
				case (tile_x)
					5'd25: tile_addr <= vcount - 64;
					5'd26: tile_addr <= vcount - 48;
				endcase
			end else begin
				case (tile_x)
					5'd25: tile_addr <= 0;
					5'd26: tile_addr <= 0;
				endcase
			end
		end
		//Brick line 5
		5'd9 : begin
			if (brick5[12]) begin
				case (tile_x)
					5'd1 : tile_addr <= vcount - 80;
					5'd2 : tile_addr <= vcount - 64;
				endcase
			end else begin
				case (tile_x)
					5'd1 : tile_addr <= 0;
					5'd2 : tile_addr <= 0;
				endcase
			end
			if (brick5[11]) begin
				case (tile_x)
					5'd3 : tile_addr <= vcount - 80;
					5'd4 : tile_addr <= vcount - 64;
				endcase
			end else begin
				case (tile_x)
					5'd3 : tile_addr <= 0;
					5'd4 : tile_addr <= 0;
				endcase
			end
			if (brick5[10]) begin
				case (tile_x)
					5'd5 : tile_addr <= vcount - 80;
					5'd6 : tile_addr <= vcount - 64;
				endcase
			end else begin
				case (tile_x)
					5'd5 : tile_addr <= 0;
					5'd6 : tile_addr <= 0;
				endcase
			end
			if (brick5[9]) begin
				case (tile_x)
					5'd7 : tile_addr <= vcount - 80;
					5'd8 : tile_addr <= vcount - 64;
				endcase
			end else begin
				case (tile_x)
					5'd7 : tile_addr <= 0;
					5'd8 : tile_addr <= 0;
				endcase
			end
			if (brick5[8]) begin
				case (tile_x)
					5'd9 : tile_addr <= vcount - 80;
					5'd10: tile_addr <= vcount - 64;
				endcase
			end else begin
				case (tile_x)
					5'd9 : tile_addr <= 0;
					5'd10: tile_addr <= 0;
				endcase
			end
			if (brick5[7]) begin
				case (tile_x)
					5'd11: tile_addr <= vcount - 80;
					5'd12: tile_addr <= vcount - 64;
				endcase
			end else begin
				case (tile_x)
					5'd11: tile_addr <= 0;
					5'd12: tile_addr <= 0;
				endcase
			end
			if (brick5[6]) begin
				case (tile_x)
					5'd13: tile_addr <= vcount - 80;
					5'd14: tile_addr <= vcount - 64;
				endcase
			end else begin
				case (tile_x)
					5'd13: tile_addr <= 0;
					5'd14: tile_addr <= 0;
				endcase
			end
			if (brick5[5]) begin
				case (tile_x)
					5'd15: tile_addr <= vcount - 80;
					5'd16: tile_addr <= vcount - 64;
				endcase
			end else begin
				case (tile_x)
					5'd15: tile_addr <= 0;
					5'd16: tile_addr <= 0;
				endcase
			end
			if (brick5[4]) begin
				case (tile_x)
					5'd17: tile_addr <= vcount - 80;
					5'd18: tile_addr <= vcount - 64;
				endcase
			end else begin
				case (tile_x)
					5'd17: tile_addr <= 0;
					5'd18: tile_addr <= 0;
				endcase
			end
			if (brick5[3]) begin
				case (tile_x)
					5'd19: tile_addr <= vcount - 80;
					5'd20: tile_addr <= vcount - 64;
				endcase
			end else begin
				case (tile_x)
					5'd19: tile_addr <= 0;
					5'd20: tile_addr <= 0;
				endcase
			end
			if (brick5[2]) begin
				case (tile_x)
					5'd21: tile_addr <= vcount - 80;
					5'd22: tile_addr <= vcount - 64;
				endcase
			end else begin
				case (tile_x)
					5'd21: tile_addr <= 0;
					5'd22: tile_addr <= 0;
				endcase
			end
			if (brick5[1]) begin
				case (tile_x)
					5'd23: tile_addr <= vcount - 80;
					5'd24: tile_addr <= vcount - 64;
				endcase
			end else begin
				case (tile_x)
					5'd23: tile_addr <= 0;
					5'd24: tile_addr <= 0;
				endcase
			end
			if (brick5[0]) begin
				case (tile_x)
					5'd25: tile_addr <= vcount - 80;
					5'd26: tile_addr <= vcount - 64;
				endcase
			end else begin
				case (tile_x)
					5'd25: tile_addr <= 0;
					5'd26: tile_addr <= 0;
				endcase
			end
		end
		//Brick line 6
		5'd10: begin
			if (brick6[12]) begin
				case (tile_x)
					5'd1 : tile_addr <= vcount - 96;
					5'd2 : tile_addr <= vcount - 80;
				endcase
			end else begin
				case (tile_x)
					5'd1 : tile_addr <= 0;
					5'd2 : tile_addr <= 0;
				endcase
			end
			if (brick6[11]) begin
				case (tile_x)
					5'd3 : tile_addr <= vcount - 96;
					5'd4 : tile_addr <= vcount - 80;
				endcase
			end else begin
				case (tile_x)
					5'd3 : tile_addr <= 0;
					5'd4 : tile_addr <= 0;
				endcase
			end
			if (brick6[10]) begin
				case (tile_x)
					5'd5 : tile_addr <= vcount - 96;
					5'd6 : tile_addr <= vcount - 80;
				endcase
			end else begin
				case (tile_x)
					5'd5 : tile_addr <= 0;
					5'd6 : tile_addr <= 0;
				endcase
			end
			if (brick6[9]) begin
				case (tile_x)
					5'd7 : tile_addr <= vcount - 96;
					5'd8 : tile_addr <= vcount - 80;
				endcase
			end else begin
				case (tile_x)
					5'd7 : tile_addr <= 0;
					5'd8 : tile_addr <= 0;
				endcase
			end
			if (brick6[8]) begin
				case (tile_x)
					5'd9 : tile_addr <= vcount - 96;
					5'd10: tile_addr <= vcount - 80;
				endcase
			end else begin
				case (tile_x)
					5'd9 : tile_addr <= 0;
					5'd10: tile_addr <= 0;
				endcase
			end
			if (brick6[7]) begin
				case (tile_x)
					5'd11: tile_addr <= vcount - 96;
					5'd12: tile_addr <= vcount - 80;
				endcase
			end else begin
				case (tile_x)
					5'd11: tile_addr <= 0;
					5'd12: tile_addr <= 0;
				endcase
			end
			if (brick6[6]) begin
				case (tile_x)
					5'd13: tile_addr <= vcount - 96;
					5'd14: tile_addr <= vcount - 80;
				endcase
			end else begin
				case (tile_x)
					5'd13: tile_addr <= 0;
					5'd14: tile_addr <= 0;
				endcase
			end
			if (brick6[5]) begin
				case (tile_x)
					5'd15: tile_addr <= vcount - 96;
					5'd16: tile_addr <= vcount - 80;
				endcase
			end else begin
				case (tile_x)
					5'd15: tile_addr <= 0;
					5'd16: tile_addr <= 0;
				endcase
			end
			if (brick6[4]) begin
				case (tile_x)
					5'd17: tile_addr <= vcount - 96;
					5'd18: tile_addr <= vcount - 80;
				endcase
			end else begin
				case (tile_x)
					5'd17: tile_addr <= 0;
					5'd18: tile_addr <= 0;
				endcase
			end
			if (brick6[3]) begin
				case (tile_x)
					5'd19: tile_addr <= vcount - 96;
					5'd20: tile_addr <= vcount - 80;
				endcase
			end else begin
				case (tile_x)
					5'd19: tile_addr <= 0;
					5'd20: tile_addr <= 0;
				endcase
			end
			if (brick6[2]) begin
				case (tile_x)
					5'd21: tile_addr <= vcount - 96;
					5'd22: tile_addr <= vcount - 80;
				endcase
			end else begin
				case (tile_x)
					5'd21: tile_addr <= 0;
					5'd22: tile_addr <= 0;
				endcase
			end
			if (brick6[1]) begin
				case (tile_x)
					5'd23: tile_addr <= vcount - 96;
					5'd24: tile_addr <= vcount - 80;
				endcase
			end else begin
				case (tile_x)
					5'd23: tile_addr <= 0;
					5'd24: tile_addr <= 0;
				endcase
			end
			if (brick6[0]) begin
				case (tile_x)
					5'd25: tile_addr <= vcount - 96;
					5'd26: tile_addr <= vcount - 80;
				endcase
			end else begin
				case (tile_x)
					5'd25: tile_addr <= 0;
					5'd26: tile_addr <= 0;
				endcase
			end
		end
	  	endcase
		end
	end

	// ----------------ROM colormap_ROM.v-------------
	// Assign specific color to each pixel
	assign addr_adj = tile_output >> (30 - ({hcount[10:1],1'b0} - (192 + (tile_x * 32))));
	logic [4:0] color_addr;
	logic [23:0] color_output;
	logic [31:0] addr_adj;
	colormap_ROM color1(.address(color_addr),.clock(clk),.q(color_output));

	always_ff @(posedge clk) begin
    //------Static elements------
		if ((tile_x <= 27 && tile_y == 2)|| //Corners + Top
            ((tile_x == 0 || tile_x == 27) && tile_y >= 3) //Side
		   )
			color_addr <= {3'b0,addr_adj[1:0]};
	//SCORE & Stage & Win/Lose
		else if ((tile_x <= 16 && tile_x >= 12 && tile_y == 0)||
		(tile_x <= 16 && tile_x >= 13 && tile_y == 1)||
		(tile_x <= 27 && tile_x >= 23 && tile_y == 0)||
		(tile_x == 27 && tile_y == 1)||
		((tile_x == 1 || tile_x == 2) && tile_y == 29)|| //HP
		(tile_y == 15 && tile_x <= 18 && tile_x >= 10) // Win or Lose
		)
			color_addr <= addr_adj[1:0] + 4;
    //------Dynamic elements------
	//Brick lines
		else if (tile_x >= 1 && tile_x <= 26) begin
			case (tile_y)
				5'd5 : color_addr <= addr_adj[1:0] + 8 + (4*brick1[15:13]);
				5'd6 : color_addr <= addr_adj[1:0] + 8 + (4*brick2[15:13]);
				5'd7 : color_addr <= addr_adj[1:0] + 8 + (4*brick3[15:13]);
				5'd8 : color_addr <= addr_adj[1:0] + 8 + (4*brick4[15:13]);
				5'd9 : color_addr <= addr_adj[1:0] + 8 + (4*brick5[15:13]);
				5'd10: color_addr <= addr_adj[1:0] + 8 + (4*brick6[15:13]);
			endcase
		end
	end

	//-------Display-------
	// Put everything on screen
    always_comb begin
      {VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0};
      if (VGA_BLANK_n)
	    if (circle) //Ball
	      	{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff};
       	else if (peddle) //Pad
        	{VGA_R, VGA_G, VGA_B} = {8'h0, 8'hff, 8'hff};
	    else if (waste) //Gray needless area
       		{VGA_R, VGA_G, VGA_B} = {8'h69, 8'h69, 8'h69};
        else if ((tile_x <= 27 && tile_y == 2)|| //Corners + Top
                ((tile_x == 0 || tile_x == 27)&& tile_y >= 3)|| //Side
                (tile_x <= 16 && tile_x >= 12 && tile_y == 0)|| //SCORE
                (tile_x <= 16 && tile_x >= 13 && tile_y == 1)|| //Score Number
			    (tile_x <= 27 && tile_x >= 23 && tile_y == 0)|| //STAGE
		        (tile_x == 27 && tile_y == 1)|| //Stage Number
				((tile_x == 1 || tile_x == 2) && tile_y == 29)|| //HP Indicator
				(tile_y == 15 && tile_x <= 18 && tile_x >= 10)|| //Win or Lose
			    (tile_x >= 1 && tile_x <= 26 && tile_y >= 5 && tile_y <= 10) //Bricks
				)
			{VGA_R, VGA_G, VGA_B} = color_output;
        else //Background
	        {VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0};

   end

endmodule

module vga_counters(
 input logic 	     clk50, reset,
 output logic [10:0] hcount,  // hcount[10:1] is pixel column
 output logic [9:0]  vcount,  // vcount[9:0] is pixel row
 output logic 	     VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 *
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 *
 *
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC +
                            HBACK_PORCH; // 1600

   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC +
                            VBACK_PORCH; // 525

   logic endOfLine;

   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;

   logic endOfField;

   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) &
		      !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1'b0; // For putting sync on the green signal; unused

   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: rising edge sensitive

endmodule
