module Table
  ( litLenTable,
    llHuffmanCode,
    distTable,
  )
where

import qualified Data.Word as W

{-
Block type 1 has static trees that are defined in RFC 1951.
This file contains those trees.
-}

{-
Value 0-256 can be converted to bits as is
Value 257-285 must be looked up according to the table
Given a length, return (code, num of extra bits, base)
-}
litLenTable :: W.Word16 -> (W.Word16, W.Word8, W.Word16)
litLenTable ll
  | ll < 3 = error "too small"
  | ll == 3 = (257, 0, 3)
  | ll == 4 = (258, 0, 4)
  | ll == 5 = (259, 0, 5)
  | ll == 6 = (260, 0, 6)
  | ll == 7 = (261, 0, 7)
  | ll == 8 = (262, 0, 8)
  | ll == 9 = (263, 0, 9)
  | ll == 10 = (264, 0, 10)
  | ll <= 12 = (265, 1, 11)
  | ll <= 14 = (266, 1, 13)
  | ll <= 16 = (267, 1, 15)
  | ll <= 18 = (268, 1, 17)
  | ll <= 22 = (269, 2, 19)
  | ll <= 26 = (270, 2, 23)
  | ll <= 30 = (271, 2, 27)
  | ll <= 34 = (272, 2, 31)
  | ll <= 42 = (273, 3, 35)
  | ll <= 50 = (274, 3, 43)
  | ll <= 58 = (275, 3, 51)
  | ll <= 66 = (276, 3, 59)
  | ll <= 82 = (277, 4, 67)
  | ll <= 98 = (278, 4, 83)
  | ll <= 114 = (279, 4, 99)
  | ll <= 130 = (280, 4, 115)
  | ll <= 162 = (281, 5, 131)
  | ll <= 194 = (282, 5, 163)
  | ll <= 226 = (283, 5, 195)
  | ll <= 257 = (284, 5, 227)
  | ll == 258 = (285, 0, 258)
  | otherwise = error "invalid length"

{-
Given a lit val, return the corresponding huffman tree val
and also bit length
-}
llHuffmanCode :: W.Word16 -> (W.Word16, W.Word8)
llHuffmanCode val
  | val <= 143 = (48 + val, 8)
  | val <= 255 = (400 + (val - 144), 9)
  | val <= 279 = (val - 256, 7)
  | val <= 287 = (192 + (val - 280), 8)
  | otherwise = error "invalid val"

{-
(code, num extra bits, base)
-}
distTable :: W.Word16 -> (W.Word16, W.Word8, W.Word16)
distTable d
  | d == 0 = error "invalid zero dist"
  | d == 1 = (0, 0, 1)
  | d == 2 = (1, 0, 2)
  | d == 3 = (2, 0, 3)
  | d == 4 = (3, 0, 4)
  | d <= 6 = (4, 1, 5)
  | d <= 8 = (5, 1, 7)
  | d <= 12 = (6, 2, 9)
  | d <= 16 = (7, 2, 13)
  | d <= 24 = (8, 3, 17)
  | d <= 32 = (9, 3, 25)
  | d <= 48 = (10, 4, 33)
  | d <= 64 = (11, 4, 49)
  | d <= 96 = (12, 5, 65)
  | d <= 128 = (13, 5, 97)
  | d <= 192 = (14, 6, 129)
  | d <= 256 = (15, 6, 193)
  | d <= 384 = (16, 7, 257)
  | d <= 512 = (17, 7, 385)
  | d <= 768 = (18, 8, 513)
  | d <= 1024 = (19, 8, 769)
  | d <= 1536 = (20, 9, 1025)
  | d <= 2048 = (21, 9, 1537)
  | d <= 3072 = (22, 10, 2049)
  | d <= 4096 = (23, 10, 3073)
  | d <= 6144 = (24, 11, 4097)
  | d <= 8192 = (25, 11, 6145)
  | d <= 12288 = (26, 12, 8193)
  | d <= 16384 = (27, 12, 12289)
  | d <= 24576 = (28, 13, 16385)
  | d <= 32768 = (29, 13, 24577)
  | otherwise = error "invalid dist"