module Lib
  ( zlibCompress,
    parCompress,
    seqCompress,
    writeOut,
    getHeader,
    getFooter,
  )
where

import qualified Block as B
import qualified Codec.Compression.GZip as GZip
import qualified Data.ByteString.Lazy as LBS
import qualified Data.Functor((<&>))

-- write the output to a given filename
writeOut :: String -> LBS.ByteString -> IO ()
writeOut = LBS.writeFile

-- use zlib for compression
zlibCompress :: String -> IO LBS.ByteString
zlibCompress fname = LBS.readFile fname Data.Functor.<&> GZip.compress

-- sequential implementation
seqCompress :: String -> IO LBS.ByteString
seqCompress fname =
  LBS.readFile fname >>= (\f -> return (LBS.concat [getHeader, B.doSeqCompress f]))

parCompress :: String -> IO LBS.ByteString
parCompress fname = 
  LBS.readFile fname >>= (\f -> return (LBS.concat [getHeader, B.doParCompress f]))

emptyCompress :: IO LBS.ByteString
emptyCompress = return $ LBS.concat [getHeader, x, getFooter]
  where
    x = LBS.pack [0x01, 0x00, 0x00, 0xff, 0xff]

-- add the 10-byte header for .gz files
getHeader :: LBS.ByteString
getHeader = LBS.pack [0x1f, 0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03]

-- add the 8-byte footer for .gz files
getFooter :: LBS.ByteString
getFooter = LBS.pack [0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00]
