module ParseAST where

import qualified Data.Map.Strict as Map
import Language.C
import Language.C.System.GCC


data EntryType = IdDecl | IdRef | IdCall | IdLabel | IdLocal deriving (Eq, Show)
-- Not meaningful, just in case of sorting for searching
instance Ord EntryType where
    IdLocal `compare` _         = EQ
    _       `compare` IdLocal   = EQ
    IdDecl  `compare` _         = LT
    IdRef   `compare` IdDecl    = GT
    IdRef   `compare` _         = LT
    IdCall  `compare` IdLabel   = LT
    IdCall  `compare` _         = GT
    IdLabel `compare` _         = GT

-- | IdEntryVal stores the information about a symbol:
--  (file, row, column, type)
type IdEntryVal = (String, Int, Int, EntryType)
-- | (ident, key)
-- type IdEntry = (String, IdEntryVal)
-- type IdDB = [IdEntry]

type IdEntry = IdEntryVal
type IdDB = Map.Map String [IdEntry]

-- dummy entry for local symbols to avoid unnecessary GC
dummyEntry :: IdEntry
dummyEntry = ("", 0, 0, IdLocal)

identToEntry :: Ident -> EntryType -> IdEntry
identToEntry ident entry_type =
    let id_file = case fileOfNode ident of
                    Nothing -> ""
                    Just p -> p in
    let id_pos = posOfNode $ nodeInfo ident in
    let id_row = posRow $ id_pos in
    let id_col = posColumn $ id_pos in
    (id_file, id_row, id_col, entry_type)

-- Just use linear search as the size of the local list should be handy
inLocalList :: IdDB -> String -> Bool
-- "true" and "false" are excluded since they are widely used as keywords
inLocalList _ "true" = True
inLocalList _ "false" = True
inLocalList db id_name =  case Map.lookup id_name db of
    Just _ -> True
    _ -> False

addEntry :: Ident -> EntryType -> IdDB -> IdDB
addEntry ident IdLocal gl =
    let id_name = (identToString ident) in
    Map.insert id_name [dummyEntry] gl
addEntry ident t gl =
    let id_name = (identToString ident) in
    let id_entry = identToEntry ident t in
    Map.insertWith mergeEntry id_name [id_entry] gl
    where
    mergeEntry :: [IdEntry] -> [IdEntry] -> [IdEntry]
    mergeEntry [n] o = n : o
    mergeEntry _ o = o -- we know new_value must be a singleton list

parseDeclList :: IdDB -> IdDB -> [(Maybe (CDeclarator a0), b0, c0)] ->
    (IdDB, IdDB)
parseDeclList gl ll [] = (gl, ll)
parseDeclList gl ll ((cDeclr, _, _):xs) = case cDeclr of
    Nothing -> (gl, ll)
    Just (CDeclr (Just ident) _ _ _ _) ->
        case null ll of
            True -> let gl' = addEntry ident IdDecl gl in parseDeclList gl' ll xs
            _ -> let ll' = addEntry ident IdLocal ll in parseDeclList gl ll' xs
    _ -> (gl, ll)

parseCSU :: IdDB -> IdDB -> CStructureUnion a -> (IdDB, IdDB)
parseCSU gl ll (CStruct _ mident mdecl _ _) = case mident of
    Just ident ->
        -- struct variable declarations are always indexed
        let gl' = addEntry ident IdDecl gl in
        case mdecl of
            Just declL -> (parseStructDeclList gl' declL, ll)
            _ -> (gl', ll)
    _ -> (gl, ll)
    where
        -- struct fields are always indexed
        parseStructDeclList gl' [] = gl'
        parseStructDeclList gl' (x:xs) =
            let (dl, _) = (parseDecl gl' Map.empty x) in parseStructDeclList dl xs

parseCType :: IdDB -> IdDB -> [CDeclarationSpecifier a] ->
    [(Maybe (CDeclarator a0), b0, c0)] -> (IdDB, IdDB)
parseCType gl ll [] _ = (gl, ll)
parseCType gl ll (cType:_) declList = case cType of
    -- struct or union
    CTypeSpec (CSUType (csu) _) ->
        let (gl', ll') = parseCSU gl ll csu in
        case declList of
            [] -> (gl', ll')
            _ -> parseDeclList gl' ll' declList
    -- other types
    _ -> parseDeclList gl ll declList

--parseDecl :: CDeclaration a -> IdEntry
parseDecl :: IdDB -> IdDB -> (CDeclaration a) ->
    (IdDB, IdDB)
parseDecl gl ll (CDecl cTypeList declrList _) =
    parseCType gl ll cTypeList declrList
parseDecl gl ll _ = (gl, ll)

-- expr and stmt won't introduce new symbols so local DB is always discarded
parseExprList :: IdDB -> IdDB -> [CExpression a] -> EntryType -> IdDB
parseExprList gl _ [] _ = gl
parseExprList gl ll (expr:xs) id_type =
    let gl' = parseExpr gl ll expr id_type in
    parseExprList gl' ll xs id_type

parseExpr :: IdDB -> IdDB -> (CExpression a) -> EntryType -> IdDB
parseExpr gl ll cexpr id_type = case cexpr of
    CComma exprList _ -> parseExprList gl ll exprList IdRef
    CAssign _ expr1 expr2 _ ->
        parseExpr2 gl ll (expr1, IdRef) (expr2, IdRef)
    CCond expr1 Nothing expr2 _ ->
        parseExpr2 gl ll (expr1, IdRef) (expr2, IdRef)
    CCond expr1 (Just expr2) expr3 _ ->
        parseExpr3 gl ll (expr1, IdRef) (expr2, IdRef) (expr3, IdRef)
    CBinary _ expr1 expr2 _ ->
        parseExpr2 gl ll (expr1, IdRef) (expr2, IdRef)
    CCast _ expr _ -> parseExpr gl ll expr IdRef
    CUnary _ expr _ -> parseExpr gl ll expr IdRef
    CSizeofExpr expr _ -> parseExpr gl ll expr IdRef
    CIndex expr1 expr2 _ ->
        parseExpr2 gl ll (expr1, IdRef) (expr2, IdRef)
    CCall expr exprList _ ->
        -- callee must be defined so ll can't be changed
        let gl' = parseExpr gl ll expr IdCall in
        parseExprList gl' ll exprList IdRef
    CMember struct field _ _ -> -- field :: Ident is always indexed
        let gl' = parseExpr gl ll struct IdRef in
        addEntry field IdRef gl'
    CVar ident _ ->
        -- if the ident is a local variable, just discard it
        if inLocalList ll (identToString ident)
            then gl
            else addEntry ident id_type gl
    _ -> gl

parseExpr2 :: IdDB -> IdDB -> (CExpression a, EntryType) ->
    (CExpression a, EntryType) -> IdDB
parseExpr2 gl ll (expr1, t1) (expr2, t2) =
    let gl' = parseExpr gl ll expr1 t1 in
    parseExpr gl' ll expr2 t2

parseExpr3 :: IdDB -> IdDB -> (CExpression a, EntryType) ->
    (CExpression a, EntryType) -> (CExpression a, EntryType) -> IdDB
parseExpr3 gl ll exprt1 exprt2 (expr3, t3)  =
    let gl' = parseExpr2 gl ll exprt1 exprt2 in
    parseExpr gl' ll expr3 t3

parseStmt :: IdDB -> IdDB -> (CStatement a) -> IdDB
parseStmt gl ll cstmt = case cstmt of
    CLabel label stmt _ _ ->
        let gl' = addEntry label IdLabel gl in
        parseStmt gl' ll stmt
    CCase expr stmt _ ->
        let gl' = parseExpr gl ll expr IdRef in
        parseStmt gl' ll stmt
    CCases expr1 expr2 stmt _ ->
        let gl' = parseExpr2 gl ll (expr1, IdRef) (expr2, IdRef) in
        parseStmt gl' ll stmt
    CDefault stmt _ ->
        parseStmt gl ll stmt
    CExpr (Just expr) _ ->
        parseExpr gl ll expr IdRef
    CCompound label compoundItems _ ->
        parseCompound gl ll label compoundItems
    CIf expr stmt Nothing _ ->
        let gl' = parseExpr gl ll expr IdRef in
        parseStmt gl' ll stmt
    CIf expr stmt1 (Just stmt2) _ ->
        let gl' = parseExpr gl ll expr IdRef in
        let gl'' = parseStmt gl' ll stmt1 in
        parseStmt gl'' ll stmt2
    CSwitch expr stmt _ ->
        let gl' = parseExpr gl ll expr IdRef in
        parseStmt gl' ll stmt
    CWhile expr stmt _ _ ->
        let gl' = parseExpr gl ll expr IdRef in
        parseStmt gl' ll stmt
    CFor _ _ _ _ _ -> parseCFor cstmt
    CGoto label _ ->
        addEntry label IdLabel gl
    CReturn (Just expr) _ ->
        parseExpr gl ll expr IdRef
    _ -> gl
    where
        mParseExpr gl' ll' mexpr = case mexpr of
            Nothing -> Just gl'
            Just expr -> Just (parseExpr gl' ll' expr IdRef)
        parseCFor (CFor (Left mexpr1) (mexpr2) (mexpr3) stmt _) =
             case mParseExpr gl ll mexpr1 >>= \gl1 ->
                  (mParseExpr gl1 ll) mexpr2 >>= \gl2 ->
                  (mParseExpr gl2 ll) mexpr3 of
                Nothing -> gl
                Just gl3 -> parseStmt gl3 ll stmt
        parseCFor (CFor _ _ _ _ _) = gl
        parseCFor _ = gl

-- C code compound, gl is global symbol DB, ll is local symbol DB
-- Updates to a local symbol in a compound is discarded when the compound
-- is parsed
parseCompound :: IdDB -> IdDB -> [Ident] -> [CCompoundBlockItem a]
    -> IdDB
parseCompound gl _ _ [] = gl -- end of parsing, ll is discarded
parseCompound gl ll labels (blockItem:xs) = case blockItem of
    CBlockStmt stmt -> -- Stmt won't introduce new symbols
        let gl' = parseStmt gl ll stmt in
        parseCompound gl' ll labels xs
    CBlockDecl decl ->
        let (gl', ll') = parseDecl gl ll decl in
        parseCompound gl' ll' labels xs
    CNestedFunDef (_) -> gl -- GNU C nested function is not supported

parseFunDeclr :: IdDB -> (CDerivedDeclarator a) -> IdDB
parseFunDeclr ll (CFunDeclr (Left _) _ _ ) = ll -- old-style function declaration is not supported
parseFunDeclr ll (CFunDeclr (Right (cDecls, _)) _ _) =
    forEachCDecl ll cDecls
    where
    forEachCDecl :: IdDB -> [CDeclaration a] -> IdDB
    forEachCDecl rl [] = rl
    forEachCDecl rl (cDecl:xs) =
        let (new_rl, _) = (parseDecl rl Map.empty cDecl) in forEachCDecl new_rl xs
parseFunDeclr ll _ = ll

-- Function definitions
parseDef :: IdDB -> (CFunctionDef a) -> IdDB
parseDef gl (CFunDef _ cDeclr _ cCompound _) = case cDeclr of
    (CDeclr (Just ident) [cFunDeclr] _ _ _) ->
        let gl' = addEntry ident IdDecl gl in -- add function name to global list
        let ll = parseFunDeclr Map.empty cFunDeclr in -- add function arguments to local list
        case cCompound of
            (CCompound labels items _) ->
                parseCompound gl' ll labels items
            _ -> gl'
    _ -> gl

parseTranslUnit :: IdDB -> [CExternalDeclaration a] -> IdDB
parseTranslUnit gl [] = gl
parseTranslUnit gl (x:xs) = case x of
    CDeclExt decl -> let (dl, _) = (parseDecl gl Map.empty decl) in parseTranslUnit dl xs
    CFDefExt def -> let dl = parseDef gl def in parseTranslUnit dl xs
    _ -> gl

parseAST :: CTranslationUnit a -> IdDB
parseAST (CTranslUnit l _) = parseTranslUnit Map.empty l

readWithPrep :: String -> IO ()
readWithPrep input_file = do
    ast <- errorOnLeftM "Parse Error" $
        parseCFile (newGCC "gcc") Nothing [""] input_file
    mapM_ print $ parseAST ast

errorOnLeft :: (Show a) => String -> (Either a b) -> IO b
errorOnLeft msg = either (error . ((msg ++ ": ")++).show) return
errorOnLeftM :: (Show a) => String -> IO (Either a b) -> IO b
errorOnLeftM msg action = action >>= errorOnLeft msg

