module Main where
import ParseAST

import Language.C
import Language.C.System.GCC

import System.Environment
import System.Directory
import System.Exit
import qualified Data.Map.Strict as Map

import Control.Monad
import System.FilePath
import System.Posix.Files

import Control.Parallel
import Control.Parallel.Strategies

usage :: IO ()
usage = do
    prog <- getProgName
    die $ "Usage: " ++ prog ++ " <filename>|<directory> -p|-s"

-- Borrowed from https://stackoverflow.com/a/23822913
traverseDir :: FilePath -> (FilePath -> Bool) -> IO [FilePath]
traverseDir top exclude = do
  ds <- getDirectoryContents top
  paths <- forM (filter (not.exclude) ds) $ \d -> do
    let path = top </> d
    s <- getFileStatus path
    if isDirectory s
      then traverseDir path exclude
      else return [path]
  return (concat paths)

filesToStreamList :: [FilePath] -> IO [(InputStream, FilePath)]
filesToStreamList fs = sequence $ map (\f -> do
                                        s <- readInputStream f
                                        return (s, f))
                                        fs

-- Credit: https://stackoverflow.com/questions/19117922/parallel-folding-in-haskell/19119503
pfold :: (a -> a -> a) -> [a] -> a
pfold _ [x] = x
pfold mappend' xs  = (ys `par` zs) `pseq` (ys `mappend'` zs) where
    len = length xs
    (ys', zs') = splitAt (len `div` 2) xs
    ys = pfold mappend' ys'
    zs = pfold mappend' zs'

doHandleStream :: (InputStream, FilePath) -> IdDB
doHandleStream (s, f) = case parseC s $ initPos f of
    Right tu -> case tu of
        CTranslUnit l _ -> parseTranslUnit Map.empty l
    Left _ -> Map.singleton "" [dummyEntry] -- XXX: debugging
handleStreams :: [(InputStream, FilePath)] -> IdDB
handleStreams ss = foldl (Map.unionWith unionResult) Map.empty $
                        map doHandleStream ss
parHandleStreams :: [(InputStream, FilePath)] -> IdDB
parHandleStreams ss =
    pfold (Map.unionWith unionResult) $
        withStrategy (parList rpar) . map doHandleStream $ ss
unionResult :: [IdEntry] -> [IdEntry] -> [IdEntry]
unionResult new old = new ++ old

-- Simple query interface for the database
loopQuery :: IdDB -> IO ()
loopQuery db = do
    putStrLn "Search symbol:"
    sym <- getLine
    print $ Map.lookup sym db
    loopQuery db

main :: IO ()
main = do
    args <- getArgs
    case args of
        [f, c] -> handleFileDir f c
        _   -> usage
    where
    handleFileDir f c = do
        isF <- doesFileExist f
        if isF then readWithPrep f
        else  handleDir f c
    handleDir f c = do
        isD <- doesDirectoryExist f
        if isD then do
            files <- traverseDir f excludeDot
            contents <- pseq () (filesToStreamList files)
            case c of
                "-s" ->
                    loopQuery $! handleStreams contents
                "-p" ->
                    loopQuery $! parHandleStreams contents
                _ -> usage
        else die $ ("File does not exists: " ++) $ show f
    excludeDot "." = True
    excludeDot ".." = True
    excludeDot _ = False

