module Main where

import qualified Data.Map                      as M
import qualified Data.Set                      as S
import qualified Particle                      as P
import PSO

import System.Console.GetOpt
import Control.Monad
import Control.Monad.State
import Control.DeepSeq
import System.IO
import System.Exit
import System.Environment
import System.Random
import Data.List(unfoldr)
import qualified Shekel
import qualified Rastrigin


data Options = Options  { optVerbose       :: Bool
                        , optHelp          :: Bool
                        , optSwarmSize     :: Int
                        , optMaxIteration  :: Int
                        , optChunkNumber   :: Int
                        , optStrategy      :: Int
                        }

defaultOptions :: Options
defaultOptions = Options  { optVerbose       = False
                          , optHelp          = False
                          , optSwarmSize     = 500
                          , optMaxIteration  = 1000
                          , optChunkNumber   = 10
                          , optStrategy      = 2
                          }

options :: [ OptDescr (Options -> Options) ]
options =
  [ Option "h" ["help"]      (NoArg $ \opt -> opt { optHelp = True })
                             "show help"
  , Option "v" ["verbose"]   (NoArg $ \opt -> opt { optVerbose = True })
                             "verbose"
  , Option "s" ["size"]      (ReqArg (\str opt -> opt { optSwarmSize = read str }) "SWARM_SIZE")
                             "size of swarm"
  , Option "i" ["iteration"] (ReqArg (\str opt -> opt { optMaxIteration = read str }) "MAX_ITERATION")
                             "number of max iteration to run"
  , Option "c" ["chunk"]     (ReqArg (\str opt -> opt {optChunkNumber = read str}) "CHUNK_NUMBER")
                              "number of chunks to run each iteration in"
  , Option "p" ["parallel"]  (ReqArg (\str opt -> opt {optStrategy = read str}) "PARALLEL_STRATEGY")
                              "parallel strategies: {0:sequential, 1:vanilla, 2:parList, 3:parListChunk}"
  ]

main :: IO ()
main = do
  args <- getArgs
  -- Parse options, getting a list of option actions
  let (actions, nonOptions, errors) = getOpt RequireOrder options args
      -- Here we thread startOptions through all supplied option actions
      opts = foldl (flip id) defaultOptions actions
      Options { optVerbose       = v
              , optHelp          = h
              , optSwarmSize     = s
              , optMaxIteration  = i
              , optChunkNumber   = c
              , optStrategy      = p } = opts

  if h || length errors /= 0 || p < 0 || p > 3 then usage
    else putStrLn $ "Running with " ++ (show s) ++ " particles, "
                    ++ (show i) ++ " iterations, " ++ (show c) ++ " chunks " ++
                    ", strategy " ++ case p of
                                          0 -> "sequential..."
                                          1 -> "vanilla..."
                                          2 -> "parList rdeepseq..."
                                          _ -> "parListChunk rdeepseq..."

  ps <- replicateM s $ mapM randomRIO Shekel.domain
  vs <- replicateM s $ mapM randomRIO Shekel.domain
  let sw = P.initSwarm Shekel.function ps vs
      chunksz = (s + c - 1) `div` c
      (_, b) = runState (runPSO p chunksz i) PSOState { swarm = sw
                                                      , swarmSize = s
                                                      , costFn = Shekel.function
                                                      , searchDomain = Shekel.domain
                                                      , trueOptima = Shekel.minima
                                                      , currOptima = map (const 0.0) Shekel.minima
                                                      , timestamp = 0
                                                      , weight = 0.5
                                                      , psoLog = [] }
  putStrLn $ unlines (take 1 (psoLog b))

usage :: IO ()
usage = do pn <- getProgName
           die $ "Usage: " ++ pn ++
             " [-hv] [-n <number particles>] [-i <number iterations>]" ++
             " [-c <number of chunks>]" ++
             " [-p <parallel strategy: {0:sequential, 1:vanilla, 2:parList, 3:parListChunk}>]"
