{-# LANGUAGE LambdaCase #-}
{-# LANGUAGE BlockArguments #-}
import System.Environment

import qualified Data.List as L
import qualified Data.Set as S

type Clause = [Int]
type Cnf = [Clause]
type Assignments = S.Set Int

data ClauseEvalResult = Unknown | Implies Int | Inconsistent | Satisfied

notAssigned ::  Assignments -> Int -> Bool 
notAssigned assignment var =
  not $ S.member var assignment || S.member (negate var) assignment

evalClause :: Clause -> Assignments -> ClauseEvalResult
evalClause clause assignments =
  if L.foldr (\var acc -> acc || S.member var assignments) False clause 
    then Satisfied
    else (
     case L.filter (notAssigned assignments) clause of 
       [] -> Inconsistent
       [v] -> Implies v
       _ -> Unknown
    )

data EvalResult = Unsolvable | Solved | Propagate (Cnf, Assignments)

propagateAndEvalCnf :: Cnf -> Assignments -> EvalResult
propagateAndEvalCnf cnf assignments =
  let clausesWithEvalResult = L.map (\clause -> (clause, evalClause clause assignments)) cnf in
  let clauseEvals = L.map snd clausesWithEvalResult in
  let returnedClauses =
       L.map fst $ L.filter (\case (_, Unknown) -> True
                                   _ -> False)
                             clausesWithEvalResult
  in
  let accum clauseEval partialEval = 
       case (partialEval, clauseEval) of
        (Unsolvable, _) -> Unsolvable
        (_, Inconsistent) -> Unsolvable
        (Solved, Satisfied) -> Solved
        (Solved, Unknown) -> Propagate (returnedClauses, assignments)
        (Solved, Implies v) -> Propagate (returnedClauses, S.insert v assignments)
        (Propagate (cs, vs), Implies v) -> (if S.member (negate v) vs  then Unsolvable else Propagate (cs, S.insert v vs))
        (Propagate vs, _) -> Propagate vs
  in
  L.foldr accum Solved clauseEvals

selectVar :: Int -> Assignments -> Maybe Int
selectVar num_vars assignment =
  let vars = take num_vars [1..] in
  L.find (notAssigned assignment) vars


solve :: Cnf -> Int -> Assignments -> Bool
solve cnf num_vars assignments =
  case propagateAndEvalCnf cnf assignments of
    Unsolvable -> False 
    Solved -> True 
    Propagate (unresolvedClauses, propagatedAssignments) -> (
     case selectVar num_vars propagatedAssignments of
       Nothing -> error "This can never happen"
       Just v -> solve unresolvedClauses num_vars (S.insert v propagatedAssignments) || solve unresolvedClauses num_vars (S.insert (negate v) propagatedAssignments )

     )

parse :: [String] -> Cnf
parse = map (L.nub . init . map read . words)

readCnf :: String -> IO Cnf
readCnf f = parse . tail . lines <$> readFile f

numVars :: Cnf -> Int 
numVars cnf =
  L.maximum $ L.concat cnf

main :: IO ()
main = do
    args <- getArgs
    case args of
        [f] -> do
            cnf <- readCnf f
            if solve cnf (numVars cnf) S.empty then putStrLn "SAT!" else putStrLn "Unsat :("
        _ -> putStrLn "Usage: ./sequential file.cnf"