module Graphs where

import qualified Data.IntMap.Strict as IntMap
import Control.Parallel.Strategies (runEval, rpar, Eval)
import Control.Monad (foldM)
import Data.Maybe (isJust)
import System.Random  (randomRs, mkStdGen)
import System.IO(openFile, hGetContents, IOMode( ReadMode ))
import Data.List (sortBy)

type Vertex = Int
type Color = Int
type Graph = IntMap.IntMap [Vertex]

type Individual = IntMap.IntMap Color
type Population = [Individual]

seed :: Int
seed = 1337

populationSize :: Int
populationSize = 1000

iterations :: Int
iterations = 10

numOfChunks :: Int
numOfChunks = 5

numToSelect :: Int
numToSelect = 100

numToMutate :: Int
numToMutate = 200

numToCrossover :: Int
numToCrossover = 800

succ1 :: Int -> Color -> Color
succ1 = getRandomColor

getRandomColor :: Int -> Int -> Int
getRandomColor k col = head  $ take 1  $ randomRs (0, k-1) (mkStdGen (k+col))

-- map a function over a list with Eval
mapParMap :: (a -> b) -> [a] -> Eval [b]
mapParMap _ [] = return []
mapParMap f (a:as) = do
    b <- rpar (f a)
    bs <- mapParMap f as
    return (b:bs)

-- Fitness function
fitness :: Graph -> Individual -> Maybe Int
fitness graph individual = do
    let counts = runEval (mapParMap (checkVertex graph individual) (IntMap.toList individual))
    foldM (fmap . (+)) 0 counts

-- checks if vertex has neighbors of same color
checkVertex :: Graph -> Individual -> (Vertex, Color) -> Maybe Int
checkVertex graph individual (vertex, color) = do
    neighbors <- IntMap.lookup vertex graph
    colors <- mapM (`IntMap.lookup` individual) neighbors
    if color `notElem` colors then return 0 else return 1

-- mutation operator
mutate :: Graph -> Int -> Individual -> Individual
mutate graph colors individual = IntMap.fromList(f (IntMap.toList individual))
    where
        f :: [(Int, Color)] -> [(Int, Color)]
        f [] = []
        f ((k,v):xs) = if checkVertex graph individual (k,v) == Just 0
            then (k,v) : f xs
            else (k, succ1 colors v) : f xs

-- crossover operator
crossover :: Int -> Individual -> Individual -> Individual
crossover col ind1 ind2 = IntMap.fromList (f l1 l2 k)
    where
        l1 = IntMap.toList ind1
        l2 = IntMap.toList ind2
        k = getRandomCrossover col (length l1)
        f :: [(Int, Color)] -> [(Int, Color)] -> Int -> [(Int, Color)]
        f [] [] _ = []
        f [] lst2 _ = lst2
        f lst1 [] _ = lst1
        f (x:xs) (_:ys) depth = if depth /= 0 then x : f xs ys (depth-1) else ys


-- split population to chunks
splitToChunks :: Int -> [a] -> [[a]]
splitToChunks n l = f n l where
    (k,_) = quotRem (length l) n
    f :: Int -> [a] -> [[a]]
    f _ [] = []
    f nch lst = if nch == 0 then [lst] else chunk : f (nch-1) tl
        where
            (chunk, tl) = splitAt k lst

-- sort a chunk
sortChunk :: Graph -> [Individual] -> [Individual]
sortChunk graph = sortBy (\x y -> compare (fitness graph x) (fitness graph y))

-- select K from population
selectK :: Graph -> Int -> Int -> Population -> Population
selectK graph nchunks k population = sortChunk graph $ concatMap (take k) chunksSorted where
    chunks = splitToChunks nchunks population
    chunksSorted = runEval $ mapParMap (sortChunk graph) chunks 

-- used to read row for dimacs
readRow :: String -> [(Int, [Int])]
readRow input = case head row of
    ['e'] -> [(a,[b]), (b,[a])]
    _ -> []
    where
        row = words input
        a' = read ((head . tail . tail) row) :: Int
        b' = read((head . tail) row) :: Int
        -- since vertices are 1-indexed 
        a = a'-1
        b = b'-1


readGraphDIMACS :: String -> IO Graph
readGraphDIMACS filename = do
    h <- openFile filename ReadMode
    contents <- hGetContents h
    let edgeList = concatMap readRow (lines contents)
    let graph = IntMap.fromListWith (++) edgeList
    return graph

getRandomNums :: Int -> Int -> [Int]
getRandomNums k size = take size  $ randomRs (0, k) (mkStdGen k)

getRandomCrossover :: Int -> Int -> Int
getRandomCrossover k individualSize = head  $ take 1  $ randomRs (0, individualSize-1) (mkStdGen k)

-- creates population
initializePopulation :: Int -> Int -> Population
initializePopulation individualSize colors = population
    where
        numbers = getRandomNums colors (individualSize*populationSize)
        individualChunks = splitToChunks populationSize numbers
        population = map (IntMap.fromList . zip [0..(individualSize-1)]) individualChunks

-- overall search
searchK :: Int -> Graph -> Maybe Int
searchK k graph = do
    let individualSize = IntMap.size graph
    if k == 0 then
        return individualSize
    else do

        let population = initializePopulation individualSize k
        res <- searchRoutine graph population k iterations
        if res then do searchK (k-1) graph
        else return k

-- search for particular k
searchRoutine :: Graph -> Population -> Int -> Int -> Maybe Bool
searchRoutine graph population k iter = do
    if iter == 0 then return False else do
        let newPopulation = selectK graph numOfChunks numToSelect population
        let fits = filter isJust (map (fitness graph) newPopulation)
        let solution = Just 0 `elem` fits
        if solution then return True
        else do
            let mutated = runEval $ mapParMap (mutate graph k) (take numToMutate newPopulation)
            let crossedover =  runEval $ mapParMap (uncurry (crossover k)) (take numToCrossover [(x,y) | x <- cycle newPopulation, y <- tail (cycle newPopulation)])
            let nextPopulation = crossedover ++ mutated
            searchRoutine graph nextPopulation k (iter-1)
